"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.QueueExtension = exports.TopicSubscription = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const events = require("aws-cdk-lib/aws-events");
const events_targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const subscription = require("aws-cdk-lib/aws-sns-subscriptions");
const sqs = require("aws-cdk-lib/aws-sqs");
const extension_interfaces_1 = require("../extension-interfaces");
/**
 * The `TopicSubscription` class represents an SNS Topic resource that can be subscribed to by the service queues.
 */
class TopicSubscription {
    constructor(props) {
        var _c, _d;
        this.topic = props.topic;
        if (props.topicSubscriptionQueue && props.queue) {
            throw Error('Either provide the `subscriptionQueue` or the `queue` (deprecated) for the topic subscription, but not both.');
        }
        this.subscriptionQueue = props.topicSubscriptionQueue;
        this.queue = (_c = props.queue) !== null && _c !== void 0 ? _c : (_d = props.topicSubscriptionQueue) === null || _d === void 0 ? void 0 : _d.queue;
    }
    /**
     * This method sets up SNS Topic subscriptions for the SQS queue provided by the user. If a `queue` is not provided,
     * the default `eventsQueue` subscribes to the given topic.
     *
     * @param extension `QueueExtension` added to the service
     * @returns the queue subscribed to the given topic
     */
    subscribe(extension) {
        var _c, _d, _e;
        const queue = (_e = (_d = (_c = this.subscriptionQueue) === null || _c === void 0 ? void 0 : _c.queue) !== null && _d !== void 0 ? _d : this.queue) !== null && _e !== void 0 ? _e : extension.eventsQueue;
        this.topic.addSubscription(new subscription.SqsSubscription(queue));
        return queue;
    }
}
exports.TopicSubscription = TopicSubscription;
_a = JSII_RTTI_SYMBOL_1;
TopicSubscription[_a] = { fqn: "cdk-ecs-service-extensions.TopicSubscription", version: "0.0.2" };
/**
 * This hook modifies the application container's environment to
 * add the queue URL for the events queue of the service.
 */
class QueueExtensionMutatingHook extends extension_interfaces_1.ContainerMutatingHook {
    constructor(props) {
        super();
        this.environment = props.environment;
    }
    mutateContainerDefinition(props) {
        return {
            ...props,
            environment: { ...(props.environment || {}), ...this.environment },
        };
    }
}
/**
 * This extension creates a default `eventsQueue` for the service (if not provided) and accepts a list of objects of
 * type `ISubscribable` that the `eventsQueue` subscribes to. It creates the subscriptions and sets up permissions
 * for the service to consume messages from the SQS Queues.
 *
 * It also configures a target tracking scaling policy for the service to maintain an acceptable queue latency by tracking
 * the backlog per task. For more information, please refer: https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-using-sqs-queue.html .
 *
 * The default queue for this service can be accessed using the getter `<extension>.eventsQueue`.
 */
class QueueExtension extends extension_interfaces_1.ServiceExtension {
    constructor(props) {
        super('queue');
        this.subscriptionQueues = new Set();
        this.environment = {};
        this.props = props;
    }
    /**
     * This hook creates (if required) and sets the default queue `eventsQueue`. It also sets up the subscriptions for
     * the provided `ISubscribable` objects.
     *
     * @param service The parent service which this extension has been added to
     * @param scope The scope that this extension should create resources in
     */
    prehook(service, scope) {
        var _c, _d, _e, _f, _g;
        this.parentService = service;
        this.scope = scope;
        let eventsQueue = (_c = this.props) === null || _c === void 0 ? void 0 : _c.eventsQueue;
        if (!eventsQueue) {
            const deadLetterQueue = new sqs.Queue(this.scope, 'EventsDeadLetterQueue', {
                retentionPeriod: aws_cdk_lib_1.Duration.days(14),
            });
            eventsQueue = new sqs.Queue(this.scope, 'EventsQueue', {
                deadLetterQueue: {
                    queue: deadLetterQueue,
                    maxReceiveCount: 3,
                },
            });
        }
        this._eventsQueue = eventsQueue;
        this._autoscalingOptions = (_d = this.props) === null || _d === void 0 ? void 0 : _d.scaleOnLatency;
        this.environment[`${this.parentService.id.toUpperCase()}_QUEUE_URI`] = this._eventsQueue.queueUrl;
        if ((_e = this.props) === null || _e === void 0 ? void 0 : _e.subscriptions) {
            for (const subs of this.props.subscriptions) {
                const subsQueue = subs.subscribe(this);
                if (subsQueue !== this._eventsQueue) {
                    if (((_f = subs.subscriptionQueue) === null || _f === void 0 ? void 0 : _f.scaleOnLatency) && !this._autoscalingOptions) {
                        throw Error(`Autoscaling for a topic-specific queue cannot be configured as autoscaling based on SQS Queues hasn’t been set up for the service '${this.parentService.id}'. If you want to enable autoscaling for this service, please also specify 'scaleOnLatency' in the 'QueueExtension'.`);
                    }
                    const subscriptionQueue = (_g = subs.subscriptionQueue) !== null && _g !== void 0 ? _g : {
                        queue: subsQueue,
                    };
                    this.subscriptionQueues.add(subscriptionQueue);
                }
            }
        }
    }
    /**
     * Add hooks to the main application extension so that it is modified to
     * add the events queue URL to the container environment.
     */
    addHooks() {
        const container = this.parentService.serviceDescription.get('service-container');
        if (!container) {
            throw new Error('Queue Extension requires an application extension');
        }
        container.addContainerMutatingHook(new QueueExtensionMutatingHook({
            environment: this.environment,
        }));
    }
    /**
     * After the task definition has been created, this hook grants SQS permissions to the task role.
     *
     * @param taskDefinition The created task definition
     */
    useTaskDefinition(taskDefinition) {
        this._eventsQueue.grantConsumeMessages(taskDefinition.taskRole);
        for (const subsQueue of this.subscriptionQueues) {
            subsQueue.queue.grantConsumeMessages(taskDefinition.taskRole);
        }
    }
    /**
     * When this hook is implemented by extension, it allows the extension
     * to use the service which has been created. It is used to add target tracking
     * scaling policies for the SQS Queues of the service. It also creates an AWS Lambda
     * Function for calculating the backlog per task metric.
     *
     * @param service - The generated service.
     */
    useService(service) {
        var _c;
        if (!this._autoscalingOptions) {
            return;
        }
        if (!this.parentService.scalableTaskCount) {
            throw Error(`Auto scaling target for the service '${this.parentService.id}' hasn't been configured. Please use Service construct to configure 'minTaskCount' and 'maxTaskCount'.`);
        }
        this.addQueueScalingPolicy(this._eventsQueue, this._autoscalingOptions);
        for (const subsQueue of this.subscriptionQueues) {
            const autoscalingOpts = (_c = subsQueue.scaleOnLatency) !== null && _c !== void 0 ? _c : this._autoscalingOptions;
            this.addQueueScalingPolicy(subsQueue.queue, autoscalingOpts);
        }
        this.parentService.enableAutoScalingPolicy();
        this.createLambdaFunction(service);
    }
    /**
     * This method adds a target tracking policy based on the backlog per task custom metric
     * to the auto scaling target configured for this service.
     *
     * @param queue The queue for which backlog per task metric is being configured
     * @param queueDelay The auto scaling options for the queue
     */
    addQueueScalingPolicy(queue, queueDelay) {
        var _c;
        const messageProcessingTime = queueDelay.messageProcessingTime.toSeconds();
        const acceptableLatency = queueDelay.acceptableLatency.toSeconds();
        if (messageProcessingTime > acceptableLatency) {
            throw Error(`Message processing time (${messageProcessingTime}s) for the queue cannot be greater acceptable queue latency (${acceptableLatency}s).`);
        }
        const acceptableBacklog = acceptableLatency / messageProcessingTime;
        (_c = this.parentService.scalableTaskCount) === null || _c === void 0 ? void 0 : _c.scaleToTrackCustomMetric(`${queue.node.id}-autoscaling-policy`, {
            metric: new cloudwatch.Metric({
                namespace: `${this.parentService.environment.id}-${this.parentService.id}`,
                metricName: 'BacklogPerTask',
                dimensionsMap: { QueueName: queue.queueName },
                unit: cloudwatch.Unit.COUNT,
            }),
            targetValue: acceptableBacklog,
        });
    }
    /**
     * This method is used to create the AWS Lambda Function for calculating backlog
     * per task metric and a Cloudwatch event trigger for this function.
     *
     * @param service - The generated service.
     */
    createLambdaFunction(service) {
        const queueNames = [this._eventsQueue.queueName];
        this.subscriptionQueues.forEach(subs => queueNames.push(subs.queue.queueName));
        const backLogPerTaskCalculator = new lambda.Function(this.scope, 'BackLogPerTaskCalculatorFunction', {
            runtime: lambda.Runtime.PYTHON_3_9,
            code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', '..', 'lambda', 'queue')),
            handler: 'index.queue_handler',
            environment: {
                CLUSTER_NAME: this.parentService.cluster.clusterName,
                SERVICE_NAME: service.serviceName,
                NAMESPACE: `${this.parentService.environment.id}-${this.parentService.id}`,
                QUEUE_NAMES: queueNames.join(','),
            },
            initialPolicy: [new iam.PolicyStatement({
                    actions: ['ecs:DescribeServices'],
                    resources: [`${service.serviceArn}`],
                    conditions: {
                        ArnEquals: {
                            'ecs:cluster': this.parentService.cluster.clusterArn,
                        },
                    },
                })],
        });
        const queueArns = [this._eventsQueue.queueArn];
        this.subscriptionQueues.forEach(subs => queueArns.push(subs.queue.queueArn));
        backLogPerTaskCalculator.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: [
                'sqs:GetQueueAttributes',
                'sqs:GetQueueUrl',
            ],
            resources: queueArns,
        }));
        new events.Rule(this.scope, 'BacklogPerTaskScheduledRule', {
            schedule: events.Schedule.rate(aws_cdk_lib_1.Duration.seconds(60)),
            targets: [new events_targets.LambdaFunction(backLogPerTaskCalculator)],
        });
        this.logGroup = new logs.LogGroup(this.scope, `${this.parentService.id}-BackLogPerTaskCalculatorLogs`, {
            logGroupName: `/aws/lambda/${backLogPerTaskCalculator.functionName}`,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            retention: logs.RetentionDays.THREE_DAYS,
        });
    }
    get eventsQueue() {
        return this._eventsQueue;
    }
    get autoscalingOptions() {
        return this._autoscalingOptions;
    }
}
exports.QueueExtension = QueueExtension;
_b = JSII_RTTI_SYMBOL_1;
QueueExtension[_b] = { fqn: "cdk-ecs-service-extensions.QueueExtension", version: "0.0.2" };
//# sourceMappingURL=data:application/json;base64,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