import { Construct } from 'constructs';
/**
 * Mode for accessing a data source.
 *
 * @see https://grafana.com/docs/grafana/latest/administration/provisioning/#example-data-source-config-file
 * @stability stable
 */
export declare enum AccessType {
    /**
     * Access via proxy.
     *
     * @stability stable
     */
    PROXY = "proxy",
    /**
     * Access directly (via server or browser in UI).
     *
     * @stability stable
     */
    DIRECT = "direct"
}
/**
 * @stability stable
 */
export interface DataSourceProps {
    /**
     * Name of the data source.
     *
     * @stability stable
     */
    readonly name: string;
    /**
     * Type of the data source.
     *
     * @stability stable
     */
    readonly type: string;
    /**
     * Access type of the data source.
     *
     * @stability stable
     */
    readonly access: AccessType;
    /**
     * Description of the data source.
     *
     * @default - no description
     * @stability stable
     */
    readonly description?: string;
    /**
     * URL of the data source.
     *
     * Most resources besides the 'testdata' data source
     * type require this field in order to retrieve data.
     *
     * @default - default url for data source type
     * @stability stable
     */
    readonly url?: string;
    /**
     * Labels to apply to the kubernetes resource.
     *
     * When adding a data source to a Grafana instance using `grafana.addDataSource`,
     * labels provided to Grafana will be automatically applied. Otherwise,
     * labels must be added manually.
     *
     * @default - no labels
     * @stability stable
     */
    readonly labels?: {
        [name: string]: string;
    };
    /**
     * Namespace to apply to the kubernetes resource.
     *
     * When adding a data source to a Grafana instance using `grafana.addDataSource`,
     * the namespace will be automatically inherited.
     *
     * @default - undefined (will be assigned to the 'default' namespace)
     * @stability stable
     */
    readonly namespace?: string;
}
/**
 * A Grafana data source.
 *
 * @see https://grafana.com/docs/grafana/latest/administration/provisioning/#example-data-source-config-file
 * @stability stable
 */
export declare class DataSource extends Construct {
    /**
     * Name of the data source.
     *
     * @stability stable
     */
    readonly name: string;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: DataSourceProps);
}
