"""
Gemini Chat streaming function using the official google-generativeai library.
A single function for streaming chat with Gemini API.
"""

import asyncio
import base64
import mimetypes
import os
from dataclasses import dataclass, field
from enum import Enum
from pathlib import Path
from typing import List, Optional, Callable, Awaitable, Union

import google.generativeai as genai
from google.generativeai.types import GenerationConfig, HarmCategory, HarmBlockThreshold
from google.api_core import exceptions as google_exceptions

from .template import COT


class MessageRole(Enum):
    """Message roles for chat completion."""

    USER = "user"
    ASSISTANT = "assistant"


class MediaType(Enum):
    """Supported media types for multimodal input."""

    IMAGE = "image"
    AUDIO = "audio"
    VIDEO = "video"
    DOCUMENT = "document"


def _detect_mime_type_from_data(data: bytes) -> Optional[str]:
    """Detect MIME type from file data using magic bytes."""
    if not data:
        return None

    # Check common image formats
    if data.startswith(b"\xff\xd8\xff"):
        return "image/jpeg"
    elif data.startswith(b"\x89PNG\r\n\x1a\n"):
        return "image/png"
    elif data.startswith(b"GIF8"):
        return "image/gif"
    elif data.startswith(b"RIFF") and b"WEBP" in data[:12]:
        return "image/webp"
    elif data.startswith(b"\x00\x00\x00\x18ftypheic") or data.startswith(
        b"\x00\x00\x00\x1cftypmif1"
    ):
        return "image/heic"

    # Check common video formats
    elif data.startswith(b"\x00\x00\x00\x18ftyp") or data.startswith(
        b"\x00\x00\x00\x20ftyp"
    ):
        return "video/mp4"
    elif data.startswith(b"RIFF") and b"AVI " in data[:12]:
        return "video/avi"

    # Check common audio formats
    elif data.startswith(b"ID3") or data.startswith(b"\xff\xfb"):
        return "audio/mp3"
    elif data.startswith(b"RIFF") and b"WAVE" in data[:12]:
        return "audio/wav"
    elif data.startswith(b"fLaC"):
        return "audio/flac"

    # Check PDF
    elif data.startswith(b"%PDF"):
        return "application/pdf"

    # Default to binary if unknown
    return "application/octet-stream"


def _get_media_type_from_mime(mime_type: str) -> MediaType:
    """Get MediaType from MIME type."""
    if mime_type.startswith("image/"):
        return MediaType.IMAGE
    elif mime_type.startswith("audio/"):
        return MediaType.AUDIO
    elif mime_type.startswith("video/"):
        return MediaType.VIDEO
    else:
        return MediaType.DOCUMENT


def _is_supported_media_type(mime_type: str) -> bool:
    """Check if the MIME type is supported by Gemini API."""
    supported_types = {
        # Images
        "image/png",
        "image/jpeg",
        "image/webp",
        "image/heic",
        "image/heif",
        # Audio
        "audio/wav",
        "audio/mp3",
        "audio/aiff",
        "audio/aac",
        "audio/ogg",
        "audio/flac",
        # Video
        "video/mp4",
        "video/mpeg",
        "video/mov",
        "video/avi",
        "video/x-flv",
        "video/mpg",
        "video/webm",
        "video/wmv",
        "video/3gpp",
        # Documents
        "application/pdf",
        "text/plain",
    }
    return mime_type in supported_types


def _validate_media_file(
    media_file: "MediaFile", max_file_size: int = 20 * 1024 * 1024
) -> None:
    """Validate media file size and format."""
    if not media_file.data:
        raise ValueError("Media file data is empty")

    if len(media_file.data) > max_file_size:
        raise ValueError(
            f"File size {len(media_file.data)} bytes exceeds maximum {max_file_size} bytes"
        )

    if not media_file.mime_type:
        raise ValueError("Could not determine MIME type for media file")

    if not _is_supported_media_type(media_file.mime_type):
        raise ValueError(f"Unsupported media type: {media_file.mime_type}")


def _prepare_media_for_api(media_file: "MediaFile") -> dict:
    """Prepare media file for Gemini API."""
    _validate_media_file(media_file)

    return {
        "mime_type": media_file.mime_type,
        "data": base64.b64encode(media_file.data).decode("utf-8"),
    }


@dataclass
class MediaFile:
    """Represents a media file for multimodal input."""

    file_path: Optional[str] = None
    data: Optional[bytes] = None
    mime_type: Optional[str] = None
    media_type: Optional[MediaType] = None

    def __post_init__(self):
        """Validate and infer media type and MIME type."""
        if not self.file_path and not self.data:
            raise ValueError("Either file_path or data must be provided")

        if self.file_path and not self.data:
            # Read file data
            path = Path(self.file_path)
            if not path.exists():
                raise FileNotFoundError(f"File not found: {self.file_path}")

            self.data = path.read_bytes()

        # Infer MIME type if not provided
        if not self.mime_type:
            if self.file_path:
                self.mime_type, _ = mimetypes.guess_type(self.file_path)

            if not self.mime_type:
                # Try to detect from data
                self.mime_type = _detect_mime_type_from_data(self.data)

        # Infer media type from MIME type
        if not self.media_type and self.mime_type:
            self.media_type = _get_media_type_from_mime(self.mime_type)


@dataclass
class ChatMessage:
    """Chat message that can contain text and media files."""

    role: MessageRole
    content: str
    media_files: List[MediaFile] = field(default_factory=list)

    def add_media_file(
        self,
        file_path: Optional[str] = None,
        data: Optional[bytes] = None,
        mime_type: Optional[str] = None,
    ) -> None:
        """Add a media file to this message."""
        media_file = MediaFile(file_path=file_path, data=data, mime_type=mime_type)
        self.media_files.append(media_file)


class StreamCancellation:
    """Simple cancellation token for stream control."""

    def __init__(self):
        self.cancelled = False

    def cancel(self):
        """Cancel the stream."""
        self.cancelled = True

    def is_cancelled(self) -> bool:
        """Check if stream is cancelled."""
        return self.cancelled


def _process_reasoning_content(text: str, model: str) -> str:
    """Process reasoning model content, filtering out <thought> tag content."""
    if "thinking" not in model:
        return text

    result = ""
    in_thought = False
    i = 0
    while i < len(text):
        if text[i : i + 9] == "<thought>":
            in_thought = True
            i += 9
        elif text[i : i + 10] == "</thought>":
            in_thought = False
            i += 10
        else:
            if not in_thought:
                result += text[i]
            i += 1
    return result


async def stream_chat(
    api_key: str,
    callback: Callable[[str], Awaitable[None]],
    history: Optional[List[ChatMessage]] = None,
    user_message: Optional[str] = None,
    user_media_files: Optional[List[Union[str, MediaFile]]] = None,
    model: str = "gemini-2.5-flash",  # Updated to a common, modern model
    system_prompt: Optional[str] = None,
    temperature: float = 1.0,
    max_tokens: int = 8192,
    top_p: float = 0.95,
    top_k: int = 40,
    cancellation_token: Optional[StreamCancellation] = None,
    timeout: float = 300.0,
) -> str:
    """
    Send a message and get a streaming response from the Gemini API using the official library.

    Args:
        api_key: Gemini API key
        callback: Function to call with each chunk of response
        history: Optional list of previous chat messages
        user_message: Optional user's message to send (if None, uses only history)
        user_media_files: Optional list of media files (file paths or MediaFile objects) to include with user_message
        model: Gemini model to use (use vision models for image processing)
        system_prompt: Optional system prompt
        temperature: Sampling temperature (0.0-1.0)
        max_tokens: Maximum tokens to generate
        top_p: Top-p sampling parameter
        top_k: Top-k sampling parameter
        cancellation_token: Optional token to cancel the stream
        timeout: Request timeout in seconds

    Returns:
        Complete response text

    Raises:
        ValueError: If API request fails or no response generated
        asyncio.CancelledError: If cancelled via cancellation_token or asyncio
    """
    if not user_message and not history and not user_media_files:
        raise ValueError(
            "Either history, user_message, or user_media_files must be provided."
        )

    try:
        genai.configure(api_key=api_key)

        # Combine user system prompt with the CoT template
        full_system_prompt = f"# I have double checked that my CoT settings are as follows, I will never disobey them:\n{COT}"

        # Prepare generation config and safety settings
        generation_config = GenerationConfig(
            temperature=temperature,
            top_p=top_p,
            top_k=top_k,
            max_output_tokens=max_tokens,
        )

        safety_settings = {
            HarmCategory.HARM_CATEGORY_HATE_SPEECH: HarmBlockThreshold.BLOCK_NONE,
            HarmCategory.HARM_CATEGORY_HARASSMENT: HarmBlockThreshold.BLOCK_NONE,
            HarmCategory.HARM_CATEGORY_SEXUALLY_EXPLICIT: HarmBlockThreshold.BLOCK_NONE,
            HarmCategory.HARM_CATEGORY_DANGEROUS_CONTENT: HarmBlockThreshold.BLOCK_NONE,
        }

        # Instantiate the model with system prompt and configs
        generative_model = genai.GenerativeModel(
            model_name=model,
            system_instruction=full_system_prompt,
            generation_config=generation_config,
            safety_settings=safety_settings,
        )

        # Build conversation history for the API
        # The library uses 'model' for the assistant's role.
        api_history = []
        if system_prompt:
            api_history.append(
                {
                    "role": "model",
                    "parts": [
                        f"# I have double checked that my basic system settings are as follows, I will never disobey them:\n{system_prompt}"
                    ],
                }
            )

        if history:
            for message in history:
                role = "user" if message.role == MessageRole.USER else "model"
                parts = [message.content] if message.content else []

                # Add media files if present
                if hasattr(message, "media_files") and message.media_files:
                    for media_file in message.media_files:
                        parts.append(_prepare_media_for_api(media_file))

                if parts:  # Only add if there are parts
                    api_history.append({"role": role, "parts": parts})

        # Add the user's new message to the end of the history to be sent
        messages_to_send = list(api_history)
        if user_message or user_media_files:
            parts = []

            # Add text content if provided
            if user_message:
                parts.append(user_message)

            # Add media files if provided
            if user_media_files:
                for media_item in user_media_files:
                    if isinstance(media_item, str):
                        # File path provided
                        media_file = MediaFile(file_path=media_item)
                    elif isinstance(media_item, MediaFile):
                        # MediaFile object provided
                        media_file = media_item
                    else:
                        raise ValueError(f"Invalid media item type: {type(media_item)}")

                    parts.append(_prepare_media_for_api(media_file))

            if parts:
                messages_to_send.append({"role": "user", "parts": parts})

        if not messages_to_send:
            raise ValueError("No content to send to the model.")

        # Start the stream generation
        response = await generative_model.generate_content_async(
            contents=messages_to_send,
            stream=True,
            request_options={"timeout": timeout},
        )

        full_response_text = ""
        has_received_data = False
        last_chunk = None  # **修正 1**: 初始化变量以跟踪最后一个响应块

        async for chunk in response:
            last_chunk = chunk  # **修正 1**: 在循环中更新最后一个响应块

            if cancellation_token and cancellation_token.is_cancelled():
                break

            has_received_data = True

            # **修正 2 (核心崩溃修复)**: 在访问 .text 之前，先安全地检查 chunk.parts 是否存在
            if chunk.parts:
                # 由于已检查 parts，现在可以安全访问 .text
                text_content = chunk.text
                processed_text = _process_reasoning_content(text_content, model)
                if processed_text:
                    await callback(processed_text)
                    full_response_text += processed_text

        # **修正 3 (改进的空响应/错误处理)**
        # 如果循环结束但没有生成任何文本，我们将进行诊断
        if (
            not full_response_text
            and has_received_data
            and not (cancellation_token and cancellation_token.is_cancelled())
        ):
            # 如果我们收到了数据但没有生成文本，检查最后一个响应块以找出原因
            if last_chunk:
                try:
                    # 从最后一个响应块获取精确的停止原因
                    finish_reason = last_chunk.candidates[0].finish_reason.name
                    # 安全评级信息也在最后一个响应块上
                    safety_ratings = last_chunk.candidates[0].safety_ratings
                    # 抛出一个信息更丰富的异常
                    raise ValueError(
                        f"No text generated. The model stopped for the following reason: '{finish_reason}'. Safety Ratings: {safety_ratings}"
                    )
                except (AttributeError, IndexError):
                    # 如果最后一个响应块的结构异常，提供一个后备错误信息
                    raise ValueError(
                        "Stream finished but generated no text. This could be due to safety filters or an internal model decision."
                    )
            else:
                # 这种情况很少见，意味着我们根本没有收到任何响应块
                raise ValueError("No data received from the stream.")

        # 如果循环被取消或正常结束且没有输出，则返回累积的文本
        return full_response_text

    except asyncio.CancelledError:
        raise
    except (
        google_exceptions.GoogleAPICallError,
        google_exceptions.RetryError,
        google_exceptions.InvalidArgument,
    ) as e:
        raise ValueError(f"Gemini API request failed: {str(e)}") from e
    except Exception as e:
        # 重新包装异常以提供更清晰的上下文
        # 避免在已经处理过的ValueError上再次包装
        if isinstance(e, ValueError):
            raise
        raise ValueError(f"Stream chat failed unexpectedly: {str(e)}") from e


async def fetch_available_models(api_key: str) -> List[str]:
    """Get list of available Gemini models using the official library."""
    try:
        genai.configure(api_key=api_key)

        models = []
        for m in genai.list_models():
            # We only want models that support content generation.
            if "generateContent" in m.supported_generation_methods:
                model_id = m.name

                # The library returns names like 'models/gemini-pro'. Strip the prefix.
                if model_id.startswith("models/"):
                    cleaned_id = model_id[7:]
                else:
                    cleaned_id = model_id

                if _is_valid_gemini_model(cleaned_id):
                    models.append(cleaned_id)
        return models
    except (
        google_exceptions.GoogleAPICallError,
        google_exceptions.RetryError,
        google_exceptions.Unauthenticated,
    ) as e:
        raise ValueError(f"Failed to fetch models: {str(e)}") from e


def _is_valid_gemini_model(model_id: str) -> bool:
    """Check if model meets our filtering criteria."""
    import re

    # Check if it matches gemini-[version]-* pattern
    pattern = re.compile(r"^gemini-([1-9]|10)\.([0-9]|10)-")
    if not pattern.match(model_id):
        # Also allow models without version numbers like 'gemini-pro'
        if not model_id.startswith("gemini-"):
            return False

    # Exclude models containing specific keywords (updated for multimodal support)
    excluded_keywords = [
        "embedding",
        "tts",
        "exp",
        "native",
        "dialog",
        "live",
        # Removed "vision", "audio", "image" to support multimodal models
        # "thinking" is kept to allow reasoning models but filter their output
    ]

    model_lower = model_id.lower()
    for keyword in excluded_keywords:
        if keyword in model_lower:
            return False

    return True


def create_multimodal_message(
    content: str, media_files: Optional[List[Union[str, MediaFile]]] = None
) -> ChatMessage:
    """
    Create a ChatMessage with text and media files.

    Args:
        content: Text content of the message
        media_files: List of file paths or MediaFile objects

    Returns:
        ChatMessage with text and media content
    """
    message = ChatMessage(role=MessageRole.USER, content=content)

    if media_files:
        for media_item in media_files:
            if isinstance(media_item, str):
                message.add_media_file(file_path=media_item)
            elif isinstance(media_item, MediaFile):
                message.media_files.append(media_item)
            else:
                raise ValueError(f"Invalid media item type: {type(media_item)}")

    return message


def suggest_model_for_content(
    has_images: bool = False, has_audio: bool = False, has_video: bool = False
) -> str:
    """
    Suggest an appropriate Gemini model based on the content type.

    Args:
        has_images: Whether the content includes images
        has_audio: Whether the content includes audio
        has_video: Whether the content includes video

    Returns:
        Suggested model name
    """
    if has_images or has_video:
        return "gemini-1.5-pro-vision-latest"  # Best for visual content
    elif has_audio:
        return "gemini-1.5-pro"  # Good for audio processing
    else:
        return "gemini-2.5-flash"  # Default for text-only
