"""Comprehensive tests for DataBento Live API integration.

Tests:
1. Live API connectivity
2. Symbol resolution
3. Trade aggregation
4. Minute bar building
5. Latency verification (<1 minute)
6. API routing (Live vs Historical)
7. Long lookback handling
"""

import os
import time
import pytest
from datetime import datetime, timedelta, timezone
from dotenv import load_dotenv

# Load environment variables
load_dotenv()


@pytest.mark.skipif(
    not os.environ.get('DATABENTO_API_KEY'),
    reason="DATABENTO_API_KEY environment variable not set"
)
def test_symbol_resolution():
    """Test that symbols are properly resolved to contract codes"""
    from lumibot.entities import Asset
    from lumibot.data_sources.databento_data_polars_live import DataBentoDataPolarsLive
    
    print("\n" + "="*60)
    print("TEST 1: Symbol Resolution")
    print("="*60)
    
    data_source = DataBentoDataPolarsLive(
        api_key=os.getenv('DATABENTO_API_KEY'),
        has_paid_subscription=True,
        enable_live_stream=False  # Don't need streaming for this test
    )
    
    # Test ES resolution
    es_asset = Asset('ES', asset_type='CONT_FUTURE')
    es_symbol = data_source._resolve_futures_symbol(es_asset)
    print(f"✓ ES resolved to: {es_symbol}")
    assert es_symbol.startswith('ES'), f"Expected ES contract, got {es_symbol}"
    assert len(es_symbol) == 4, f"Expected 4 char symbol (e.g., ESZ5), got {es_symbol}"
    
    # Test NQ resolution
    nq_asset = Asset('NQ', asset_type='CONT_FUTURE')
    nq_symbol = data_source._resolve_futures_symbol(nq_asset)
    print(f"✓ NQ resolved to: {nq_symbol}")
    assert nq_symbol.startswith('NQ'), f"Expected NQ contract, got {nq_symbol}"
    
    print("✅ Symbol resolution test PASSED")
    return True


@pytest.mark.skipif(
    not os.environ.get('DATABENTO_API_KEY'),
    reason="DATABENTO_API_KEY environment variable not set"
)
def test_live_api_connection():
    """Test Live API connectivity and subscription"""
    import databento as db
    
    print("\n" + "="*60)
    print("TEST 2: Live API Connection")
    print("="*60)
    
    try:
        client = db.Live(key=os.getenv('DATABENTO_API_KEY'))
        print("✓ Live API client created")
        
        # Test subscription (but don't iterate to avoid consuming data)
        client.subscribe(
            dataset="GLBX.MDP3",
            schema="trades",
            stype_in="raw_symbol",
            symbols=["ESZ5"],
            start=(datetime.now(timezone.utc) - timedelta(minutes=1)).isoformat()
        )
        print("✓ Successfully subscribed to ESZ5")
        
        print("✅ Live API connection test PASSED")
        return True
        
    except Exception as e:
        print(f"❌ Live API connection failed: {e}")
        return False


@pytest.mark.skipif(
    not os.environ.get('DATABENTO_API_KEY'),
    reason="DATABENTO_API_KEY environment variable not set"
)
def test_minute_bar_aggregation():
    """Test minute bar aggregation with <1 minute lag"""
    from lumibot.entities import Asset
    from lumibot.data_sources.databento_data_polars_live import DataBentoDataPolarsLive
    
    print("\n" + "="*60)
    print("TEST 3: Minute Bar Aggregation & Latency")
    print("="*60)
    
    # Initialize with Live API
    data_source = DataBentoDataPolarsLive(
        api_key=os.getenv('DATABENTO_API_KEY'),
        has_paid_subscription=True,
        enable_live_stream=True
    )
    
    # Test with ES futures
    asset = Asset('ES', asset_type='CONT_FUTURE')
    
    print("Requesting 5 minute bars...")
    start_time = datetime.now(timezone.utc)
    
    # Get historical prices (should use Live API for recent data)
    bars = data_source.get_historical_prices(
        asset=asset,
        length=5,
        timestep='minute'
    )
    
    end_time = datetime.now(timezone.utc)
    request_time = (end_time - start_time).total_seconds()
    
    if bars is not None and len(bars) > 0:
        print(f"✓ Got {len(bars)} bars in {request_time:.1f} seconds")
        
        # Check data freshness
        df = bars.df
        # Bars object may have different time column names
        time_col = None
        for col in ['datetime', 'time', 'timestamp']:
            if col in df.columns:
                time_col = col
                break
        
        if time_col:
            latest_time = df[time_col].max()
        else:
            # Time is probably in the index (common for time series data)
            if hasattr(df, 'index'):
                latest_time = df.index.max()
            else:
                print(f"❌ No time data found. Columns: {list(df.columns)}")
                return False
        
        lag_seconds = (end_time - latest_time).total_seconds()
        lag_minutes = lag_seconds / 60
        
        print(f"Latest bar time: {latest_time}")
        print(f"Current time: {end_time}")
        print(f"Data lag: {lag_minutes:.2f} minutes ({lag_seconds:.0f} seconds)")
        
        if lag_minutes <= 1:
            print("✅ Latency test PASSED - Under 1 minute lag achieved!")
            return True
        elif lag_minutes <= 5:
            print(f"⚠️  Latency test WARNING - {lag_minutes:.2f} minute lag (acceptable)")
            return True
        else:
            print(f"❌ Latency test FAILED - {lag_minutes:.2f} minute lag is too high")
            return False
    else:
        print("❌ No bars received")
        return False


@pytest.mark.skipif(
    not os.environ.get('DATABENTO_API_KEY'),
    reason="DATABENTO_API_KEY environment variable not set"
)
def test_api_routing():
    """Test that correct API is used based on time range"""
    from lumibot.data_sources.databento_data_polars_live import DataBentoDataPolarsLive
    
    print("\n" + "="*60)
    print("TEST 4: API Routing (Live vs Historical)")
    print("="*60)
    
    data_source = DataBentoDataPolarsLive(
        api_key=os.getenv('DATABENTO_API_KEY'),
        has_paid_subscription=True,
        enable_live_stream=True
    )
    
    current_time = datetime.now(timezone.utc)
    
    # Test 1: Recent data (should use Live API)
    start_recent = current_time - timedelta(hours=2)
    should_use_live = data_source._should_use_live_api(start_recent, current_time)
    print(f"2 hours ago: {'✓' if should_use_live else '❌'} Use Live API = {should_use_live}")
    assert should_use_live, "Should use Live API for 2 hour old data"
    
    # Test 2: Old data (should use Historical API)
    start_old = current_time - timedelta(days=2)
    end_old = current_time - timedelta(days=1.5)
    should_use_live = data_source._should_use_live_api(start_old, end_old)
    print(f"2 days ago: {'✓' if not should_use_live else '❌'} Use Live API = {should_use_live}")
    assert not should_use_live, "Should use Historical API for 2 day old data"
    
    # Test 3: Mixed range (should use Live API if any part is recent)
    start_mixed = current_time - timedelta(days=2)
    should_use_live = data_source._should_use_live_api(start_mixed, current_time)
    print(f"Mixed range: {'✓' if should_use_live else '❌'} Use Live API = {should_use_live}")
    assert should_use_live, "Should use Live API for mixed range"
    
    print("✅ API routing test PASSED")
    return True


@pytest.mark.skipif(
    not os.environ.get('DATABENTO_API_KEY'),
    reason="DATABENTO_API_KEY environment variable not set"
)
def test_long_time_periods():
    """Test different time periods including long periods (500+ bars)"""
    from lumibot.entities import Asset
    from lumibot.data_sources.databento_data_polars_live import DataBentoDataPolarsLive
    
    print("\n" + "="*60)
    print("TEST 5: Long Time Period Handling (500+ bars)")
    print("="*60)
    
    data_source = DataBentoDataPolarsLive(
        api_key=os.getenv('DATABENTO_API_KEY'),
        has_paid_subscription=True,
        enable_live_stream=True
    )
    
    # Test different time periods
    test_cases = [
        (10, "Short period (10 bars)"),
        (100, "Medium period (100 bars)"),
        (500, "Long period (500 bars) - THE CRITICAL TEST"),
    ]
    
    asset = Asset('MNQ', asset_type='CONT_FUTURE')
    results = []
    
    for length, description in test_cases:
        print(f"\n{description}:")
        current_time = datetime.now(timezone.utc)
        
        # Get bars
        bars = data_source.get_historical_prices(
            asset=asset,
            length=length,
            timestep='minute'
        )
        
        if bars and len(bars) > 0:
            df = bars.df
            print(f"  ✓ Got {len(df)} bars")
            
            # Check data freshness
            if 'datetime' in df.columns:
                latest_time = df['datetime'].max()
            elif 'time' in df.columns:
                latest_time = df['time'].max()
            else:
                latest_time = df.index.max() if hasattr(df, 'index') else None
            
            if latest_time:
                # Ensure both times are timezone-aware for comparison
                if hasattr(latest_time, 'tzinfo') and latest_time.tzinfo is None:
                    latest_time = latest_time.replace(tzinfo=timezone.utc)
                
                lag_seconds = (current_time - latest_time).total_seconds()
                lag_minutes = lag_seconds / 60
                print(f"  Latest bar: {latest_time}")
                print(f"  Data lag: {lag_minutes:.2f} minutes ({lag_seconds:.0f} seconds)")
                
                # STRICT Success criteria - must be under 90 seconds
                if lag_minutes <= 1.5:  # 90 seconds
                    print("  ✅ SUCCESS - Data is fresh (<90 seconds)!")
                    results.append(True)
                else:
                    print(f"  ❌ FAILED - {lag_minutes:.2f} minute lag is too high (must be <1.5 min)")
                    results.append(False)
                    
                    # Debug info when failing
                    print(f"    Current time: {current_time}")
                    print(f"    Latest bar time: {latest_time}")
                    print(f"    This is the ACTUAL problem the bot sees!")
            else:
                print("  ❌ No time data found")
                results.append(False)
        else:
            print("  ❌ No bars received")
            results.append(False)
        
        # Wait between tests
        if length < 500:
            time.sleep(2)
    
    # Summary
    print(f"\n{'='*50}")
    print("LONG TIME PERIOD TEST RESULTS:")
    print("="*50)
    
    for i, (length, description) in enumerate(test_cases):
        status = "✅ PASSED" if results[i] else "❌ FAILED"
        print(f"{description}: {status}")
    
    # The critical test is 500 bars
    critical_test_passed = results[2] if len(results) > 2 else False
    
    if critical_test_passed:
        print("\n🎉 CRITICAL TEST PASSED! 500 bars work with live streaming! 🎉")
        return True
    else:
        print("\n❌ CRITICAL TEST FAILED! 500 bars still have high latency.")
        return False


@pytest.mark.skipif(
    not os.environ.get('DATABENTO_API_KEY'),
    reason="DATABENTO_API_KEY environment variable not set"
)
def test_continuous_latency_monitoring():
    """Run continuous tests to verify consistent <1 minute lag"""
    from lumibot.entities import Asset
    from lumibot.data_sources.databento_data_polars_live import DataBentoDataPolarsLive
    
    print("\n" + "="*60)
    print("TEST 6: Continuous Latency Monitoring")
    print("="*60)
    print("Running 5 consecutive tests to verify consistent low latency...")
    
    data_source = DataBentoDataPolarsLive(
        api_key=os.getenv('DATABENTO_API_KEY'),
        has_paid_subscription=True,
        enable_live_stream=True
    )
    
    asset = Asset('ES', asset_type='CONT_FUTURE')
    
    success_count = 0
    lag_times = []
    
    for i in range(5):
        print(f"\nTest #{i+1}:")
        current_time = datetime.now(timezone.utc)
        
        bars = data_source.get_historical_prices(
            asset=asset,
            length=5,
            timestep='minute'
        )
        
        if bars and len(bars) > 0:
            df = bars.df
            
            # Get latest time from columns or index
            if 'datetime' in df.columns:
                latest_time = df['datetime'].max()
            elif 'time' in df.columns:
                latest_time = df['time'].max()
            else:
                # Try index
                latest_time = df.index.max() if hasattr(df, 'index') else None
                if latest_time is None:
                    continue
            
            lag_seconds = (current_time - latest_time).total_seconds()
            lag_minutes = lag_seconds / 60
            lag_times.append(lag_minutes)
            
            print(f"  Lag: {lag_minutes:.2f} minutes")
            
            if lag_minutes <= 1:
                print("  ✓ Under 1 minute")
                success_count += 1
            elif lag_minutes <= 5:
                print(f"  ⚠️  {lag_minutes:.2f} minutes (acceptable)")
                success_count += 1
            else:
                print(f"  ❌ {lag_minutes:.2f} minutes (too high)")
        
        # Small delay between tests
        if i < 4:
            time.sleep(2)
    
    # Summary
    print(f"\n{'='*40}")
    print(f"Results: {success_count}/5 tests passed")
    if lag_times:
        avg_lag = sum(lag_times) / len(lag_times)
        min_lag = min(lag_times)
        max_lag = max(lag_times)
        print(f"Average lag: {avg_lag:.2f} minutes")
        print(f"Min lag: {min_lag:.2f} minutes")
        print(f"Max lag: {max_lag:.2f} minutes")
    
    if success_count >= 4:  # 80% success rate
        print("✅ Continuous latency test PASSED")
        return True
    else:
        print("❌ Continuous latency test FAILED")
        return False


def run_all_tests():
    """Run all tests and report results"""
    print("\n" + "🚀"*30)
    print("DATABENTO LIVE API TEST SUITE")
    print("🚀"*30)
    
    test_results = {}
    
    # Run each test
    tests = [
        ("Symbol Resolution", test_symbol_resolution),
        ("Live API Connection", test_live_api_connection),
        ("Minute Bar Aggregation", test_minute_bar_aggregation),
        ("API Routing", test_api_routing),
        ("Long Time Periods", test_long_time_periods),
        ("Continuous Latency", test_continuous_latency_monitoring)
    ]
    
    for test_name, test_func in tests:
        try:
            result = test_func()
            test_results[test_name] = result
        except Exception as e:
            print(f"\n❌ {test_name} failed with error: {e}")
            test_results[test_name] = False
    
    # Final summary
    print("\n" + "="*60)
    print("FINAL TEST RESULTS")
    print("="*60)
    
    for test_name, result in test_results.items():
        status = "✅ PASSED" if result else "❌ FAILED"
        print(f"{test_name}: {status}")
    
    total_passed = sum(1 for r in test_results.values() if r)
    total_tests = len(test_results)
    
    print(f"\nOverall: {total_passed}/{total_tests} tests passed")
    
    if total_passed == total_tests:
        print("\n🎉🎉🎉 ALL TESTS PASSED! System is working with <1 minute lag! 🎉🎉🎉")
        return True
    else:
        print(f"\n❌ {total_tests - total_passed} test(s) failed. Please review the errors above.")
        return False


if __name__ == "__main__":
    # Run all tests
    success = run_all_tests()
    exit(0 if success else 1)