# ModelSensor 🌍

**Let LLM sense the world** - A Python library for detecting system information, location, time, and environment to enhance AI model awareness.

[![PyPI version](https://badge.fury.io/py/modelsensor.svg)](https://badge.fury.io/py/modelsensor)
[![Python 3.7+](https://img.shields.io/badge/python-3.7+-blue.svg)](https://www.python.org/downloads/)
[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)

## 🚀 Features

- **🕒 Time Detection**: Current time, timezone, UTC time, and formatted timestamps
- **🌍 Location Information**: IP-based geolocation with country, city, coordinates
- **💻 System Information**: OS, hardware, processor, Python environment details
- **📊 Resource Monitoring**: CPU usage, memory, disk space, network statistics
- **🔧 Environment Detection**: Working directory, virtual environments, runtime context
- **🌐 Network Information**: Network interfaces, default gateway, connection details
- **📄 Multiple Output Formats**: JSON, Markdown, and summary formats
- **🔌 Easy Integration**: Perfect for Ollama and other LLM applications

## 📦 Installation

### From PyPI (Recommended)

```bash
pip install modelsensor
```

### From Source

```bash
git clone https://github.com/EasyCam/modelsensor.git
cd modelsensor
pip install -e .
```

## 🛠️ Usage

### Command Line Interface

```bash
# Basic usage - JSON output
modelsensor

# Include location information
modelsensor --location

# Markdown format
modelsensor --format markdown

# Save to file
modelsensor --output system_info.json

# Compact JSON
modelsensor --compact

# Quick summary
modelsensor --format summary
```

### Python API

```python
from modelsensor import ModelSensor

# Create sensor instance
sensor = ModelSensor()

# Collect all system information
data = sensor.collect_all_data(include_location=True)

# Get JSON output
json_output = sensor.to_json(indent=2)
print(json_output)

# Get specific information
time_info = sensor.get_time_info()
system_info = sensor.get_system_info()
resource_info = sensor.get_resource_info()
```

### With Formatters

```python
from modelsensor import ModelSensor, MarkdownFormatter, JSONFormatter

sensor = ModelSensor()
data = sensor.collect_all_data()

# Markdown output
markdown_report = MarkdownFormatter.format(data)
print(markdown_report)

# Compact JSON
compact_json = JSONFormatter.format_compact(data)
print(compact_json)

# Summary
summary = MarkdownFormatter.format_summary(data)
print(summary)
```

## 🤖 Integration with Ollama

Perfect for providing system context to your local LLM:

```python
import ollama
from modelsensor import ModelSensor

# Get system information
sensor = ModelSensor()
system_context = sensor.to_json(include_location=True, mode="full")

# Create enhanced prompt
prompt = f"""
System Context:
{system_context}

User Question: What can you tell me about my current system?
"""

# Send to Ollama
response = ollama.chat(model='qwen3:0.6b', messages=[
    {'role': 'user', 'content': prompt}
])

print(response['message']['content'])
```

## 📊 Example Output

### JSON Format
```json
{
  "sensor_info": {
    "library": "modelsensor",
    "version": "1.1.1",
    "collection_time": "2024-01-15T10:30:45.123456"
  },
  "time": {
    "current_time": "2024-01-15T10:30:45.123456",
    "utc_time": "2024-01-15T15:30:45.123456",
    "timezone": "EST",
    "weekday": "Monday",
    "formatted_time": "2024-01-15 10:30:45"
  },
  "system": {
    "system": "Darwin",
    "platform": "macOS-12.6-x86_64-i386-64bit",
    "machine": "x86_64",
    "processor": "i386",
    "python_version": "3.9.16"
  },
  "resources": {
    "cpu": {
      "usage_percent": 15.2,
      "count": 8,
      "physical_cores": 4
    },
    "memory": {
      "total_gb": 16.0,
      "used_gb": 8.5,
      "available_gb": 7.5,
      "percentage": 53.1
    }
  }
}
```

### Markdown Format
```markdown
# System Information Report
*Generated by ModelSensor at 2024-01-15T10:30:45.123456*

## 🕒 Time Information
- **Current Time**: 2024-01-15 10:30:45
- **UTC Time**: 2024-01-15T15:30:45.123456
- **Timezone**: EST
- **Day of Week**: Monday

## 💻 System Information
- **Operating System**: Darwin 21.6.0
- **Platform**: macOS-12.6-x86_64-i386-64bit
- **Machine**: x86_64
- **Python Version**: 3.9.16
```

## 🎯 Use Cases

- **🤖 AI/LLM Context**: Provide real-world awareness to language models
- **📊 System Monitoring**: Track system resources and performance
- **🔍 Environment Detection**: Identify runtime environments and configurations
- **📋 System Reporting**: Generate comprehensive system reports
- **🛠️ DevOps Tools**: System information for deployment and monitoring scripts

## 🔧 API Reference

### ModelSensor Class
- `collect_all_data(include_location=False)` - Gather all available information
- `to_json(indent=2, include_location=False, mode="brief")` - JSON string output（mode 可选 "brief" 或 "full"，默认 "brief"）
- `to_dict(include_location=False, mode="brief")` - Dictionary output（mode 可选 "brief" 或 "full"，默认 "brief"）

### Formatters

- `JSONFormatter.format(data, indent=2)` - Pretty JSON formatting
- `JSONFormatter.format_compact(data)` - Compact JSON formatting
- `MarkdownFormatter.format(data)` - Full Markdown report
- `MarkdownFormatter.format_summary(data)` - Brief summary


## Screenshots

![Use Directly](./images/use_directly.png)

![Use in Python](./images/use_in_python.png)

![Use with Ollama](./images/use_with_ollama.png)

## 🚨 Privacy Notice

- **Location data** is optional and only collected when explicitly requested
- **No data is transmitted** except for optional IP geolocation lookup
- **Environment variables** are included but can be filtered in your application
- All data collection is **local and transparent**

## 🤝 Contributing

Contributions are welcome! Please feel free to submit a Pull Request.

1. Fork the repository
2. Create your feature branch (`git checkout -b feature/AmazingFeature`)
3. Commit your changes (`git commit -m 'Add some AmazingFeature'`)
4. Push to the branch (`git push origin feature/AmazingFeature`)
5. Open a Pull Request

## 📄 License

This project is licensed under the MIT License - see the [LICENSE](LICENSE) file for details.

## 🔗 Links

- **GitHub**: https://github.com/EasyCam/modelsensor
- **Issues**: https://github.com/EasyCam/modelsensor/issues
- **Documentation**: https://github.com/EasyCam/modelsensor#readme

## 📝 Changelog

### Version 1.1.1
- Initial release
- Core system information detection
- JSON and Markdown formatters
- Command line interface
- Location detection support
- Ollama integration examples
