"""
Formatters for ModelSensor output
"""

import json
from typing import Dict, Any
from datetime import datetime


class JSONFormatter:
    """Format sensor data as JSON"""
    
    @staticmethod
    def format(data: Dict[str, Any], indent: int = 2) -> str:
        """Format data as JSON string"""
        return json.dumps(data, indent=indent, ensure_ascii=False, default=str)
    
    @staticmethod
    def format_compact(data: Dict[str, Any]) -> str:
        """Format data as compact JSON string"""
        return json.dumps(data, separators=(',', ':'), ensure_ascii=False, default=str)


class MarkdownFormatter:
    """Format sensor data as Markdown"""
    
    @staticmethod
    def format(data: Dict[str, Any]) -> str:
        """Format data as Markdown string"""
        md_content = []
        
        # Header
        md_content.append("# System Information Report")
        md_content.append(f"*Generated by ModelSensor at {data.get('sensor_info', {}).get('collection_time', 'unknown time')}*")
        md_content.append("")
        
        # Time Information
        if 'time' in data:
            md_content.append("## 🕒 Time Information")
            time_info = data['time']
            md_content.append(f"- **Current Time**: {time_info.get('formatted_time', 'N/A')}")
            md_content.append(f"- **UTC Time**: {time_info.get('utc_time', 'N/A')}")
            md_content.append(f"- **Timezone**: {time_info.get('timezone', 'N/A')}")
            md_content.append(f"- **Day of Week**: {time_info.get('weekday', 'N/A')}")
            md_content.append("")
        
        # Location Information
        if 'location' in data:
            md_content.append("## 🌍 Location Information")
            location = data['location']
            if 'error' not in location:
                md_content.append(f"- **Country**: {location.get('country', 'N/A')}")
                md_content.append(f"- **Region**: {location.get('region', 'N/A')}")
                md_content.append(f"- **City**: {location.get('city', 'N/A')}")
                md_content.append(f"- **Coordinates**: {location.get('latitude', 'N/A')}, {location.get('longitude', 'N/A')}")
                md_content.append(f"- **ISP**: {location.get('isp', 'N/A')}")
            else:
                md_content.append(f"- **Error**: {location.get('error', 'Unknown error')}")
            md_content.append("")
        
        # System Information
        if 'system' in data:
            md_content.append("## 💻 System Information")
            system = data['system']
            md_content.append(f"- **Operating System**: {system.get('system', 'N/A')} {system.get('release', '')}")
            md_content.append(f"- **Platform**: {system.get('platform', 'N/A')}")
            md_content.append(f"- **Machine**: {system.get('machine', 'N/A')}")
            md_content.append(f"- **Processor**: {system.get('processor', 'N/A')}")
            md_content.append(f"- **Hostname**: {system.get('hostname', 'N/A')}")
            md_content.append(f"- **User**: {system.get('user', 'N/A')}")
            md_content.append(f"- **Python Version**: {system.get('python_version', 'N/A').split()[0]}")
            md_content.append("")
        
        # Resource Information
        if 'resources' in data:
            md_content.append("## 📊 System Resources")
            resources = data['resources']
            
            # CPU
            if 'cpu' in resources:
                cpu = resources['cpu']
                md_content.append("### CPU")
                md_content.append(f"- **Usage**: {cpu.get('usage_percent', 'N/A')}%")
                md_content.append(f"- **Cores**: {cpu.get('count', 'N/A')} ({cpu.get('physical_cores', 'N/A')} physical)")
                if cpu.get('frequency'):
                    freq = cpu['frequency']
                    md_content.append(f"- **Frequency**: {freq.get('current', 'N/A')} MHz")
                md_content.append("")
            
            # Memory
            if 'memory' in resources:
                memory = resources['memory']
                md_content.append("### Memory")
                md_content.append(f"- **Total**: {memory.get('total_gb', 'N/A')} GB")
                md_content.append(f"- **Used**: {memory.get('used_gb', 'N/A')} GB ({memory.get('percentage', 'N/A')}%)")
                md_content.append(f"- **Available**: {memory.get('available_gb', 'N/A')} GB")
                md_content.append("")
            
            # Disk
            if 'disk' in resources:
                disk = resources['disk']
                md_content.append("### Disk Space")
                md_content.append(f"- **Total**: {disk.get('total_gb', 'N/A')} GB")
                md_content.append(f"- **Used**: {disk.get('used_gb', 'N/A')} GB ({disk.get('percentage', 0):.1f}%)")
                md_content.append(f"- **Free**: {disk.get('free_gb', 'N/A')} GB")
                md_content.append("")
        
        # Environment Information
        if 'environment' in data:
            md_content.append("## 🔧 Environment")
            env = data['environment']
            md_content.append(f"- **Working Directory**: `{env.get('working_directory', 'N/A')}`")
            md_content.append(f"- **Home Directory**: `{env.get('home_directory', 'N/A')}`")
            md_content.append(f"- **Shell**: {env.get('shell', 'N/A')}")
            
            if 'runtime_context' in env:
                context = env['runtime_context']
                md_content.append("### Runtime Context")
                for key, value in context.items():
                    status = "✅" if value else "❌"
                    md_content.append(f"- **{key.replace('is_', '').replace('_', ' ').title()}**: {status}")
            md_content.append("")
        
        # Network Information
        if 'network' in data:
            md_content.append("## 🌐 Network")
            network = data['network']
            if network.get('default_gateway'):
                md_content.append(f"- **Default Gateway**: {network['default_gateway']}")
            
            if 'interfaces' in network:
                md_content.append("### Network Interfaces")
                for interface in network['interfaces'][:3]:  # Show first 3 interfaces
                    md_content.append(f"- **{interface['name']}**")
                    for addr in interface['addresses']:
                        if addr['address'] and not addr['address'].startswith('fe80'):  # Skip link-local
                            md_content.append(f"  - {addr['address']}")
            md_content.append("")
        
        return "\n".join(md_content)
    
    @staticmethod
    def format_summary(data: Dict[str, Any]) -> str:
        """Format data as a brief summary in Markdown"""
        summary = []
        
        # Basic info
        if 'time' in data:
            summary.append(f"**Time**: {data['time'].get('formatted_time', 'N/A')}")
        
        if 'system' in data:
            system = data['system']
            summary.append(f"**System**: {system.get('system', 'N/A')} on {system.get('machine', 'N/A')}")
        
        if 'resources' in data:
            resources = data['resources']
            if 'cpu' in resources:
                summary.append(f"**CPU**: {resources['cpu'].get('usage_percent', 'N/A')}% usage")
            if 'memory' in resources:
                memory = resources['memory']
                summary.append(f"**Memory**: {memory.get('used_gb', 'N/A')}/{memory.get('total_gb', 'N/A')} GB")
        
        if 'location' in data and 'error' not in data['location']:
            location = data['location']
            summary.append(f"**Location**: {location.get('city', 'N/A')}, {location.get('country', 'N/A')}")
        
        return " | ".join(summary) 