from typing import List, Optional
from datetime import datetime
from uuid import UUID
import logging
from pydantic import BaseModel, Field

from psqlpy.extra_types import PgVector

from context_bridge.database.postgres_manager import PostgreSQLManager


logger = logging.getLogger(__name__)


class Chunk(BaseModel):
    """
    Chunk model representing a vectorized content chunk.

    Attributes:
        id: Unique chunk identifier
        document_id: Parent document ID
        group_id: Optional UUID for future grouping feature
        chunk_index: Index within the document
        content: Chunk text content
        embedding: Vector embedding as list of floats
        created_at: Timestamp when chunk was created
    """

    id: int
    document_id: int
    group_id: Optional[UUID] = None
    chunk_index: int
    content: str
    embedding: List[float]
    created_at: datetime


class SearchResult(BaseModel):
    """
    Search result with relevance score and ranking.

    Attributes:
        chunk: The matching chunk
        score: Relevance score (0-1 for normalized, raw for BM25)
        rank: Position in results (1-based)
    """

    chunk: Chunk
    score: float
    rank: int


class ChunkRepository:
    """
    Repository for chunk CRUD operations and hybrid search using PSQLPy.

    This repository handles all database operations for chunks,
    including creation, retrieval, vector search, BM25 search, and hybrid search.
    Follows PSQLPy best practices with proper error handling and
    connection management.
    """

    def __init__(self, db_manager: PostgreSQLManager):
        """
        Initialize chunk repository.

        Args:
            db_manager: PostgreSQL connection manager
        """
        self.db_manager = db_manager
        logger.debug("ChunkRepository initialized")

    async def create(
        self,
        document_id: int,
        chunk_index: int,
        content: str,
        embedding: List[float],
        group_id: Optional[UUID] = None,
    ) -> int:
        """
        Create a chunk with embedding.
        BM25 vector is auto-generated by trigger.

        Args:
            document_id: Parent document ID
            chunk_index: Index within document
            content: Chunk text content
            embedding: Embedding vector
            group_id: Optional UUID for future grouping feature

        Returns:
            ID of created chunk

        Raises:
            RuntimeError: If chunk creation fails
            Exception: Database errors
        """
        try:
            query = """
                INSERT INTO chunks (document_id, chunk_index, content, embedding, group_id)
                VALUES ($1, $2, $3, $4, $5)
                RETURNING id
            """
            pg_vector = PgVector(embedding)
            async with self.db_manager.connection() as conn:
                result = await conn.execute(
                    query, [document_id, chunk_index, content, pg_vector, group_id]
                )
                rows = result.result()
                if rows:
                    chunk_id = rows[0]["id"]  # PSQLPy returns dicts
                    logger.info(f"Created chunk {chunk_id} for document {document_id}")
                    return chunk_id
                else:
                    raise RuntimeError(f"Failed to create chunk for document {document_id}")
        except Exception as e:
            logger.error(f"Failed to create chunk for document {document_id}: {e}")
            raise

    async def create_batch(self, chunks: List[dict]) -> List[int]:
        """
        Create multiple chunks efficiently.

        Args:
            chunks: List of dicts with keys: document_id, chunk_index, content, embedding, group_id (optional)

        Returns:
            List of created chunk IDs

        Raises:
            Exception: Database errors
        """
        try:
            if not chunks:
                return []

            # Build batch insert query
            values_list = []
            params = []
            param_count = 1

            for chunk in chunks:
                values_list.append(
                    f"(${param_count}, ${param_count + 1}, ${param_count + 2}, ${param_count + 3}, ${param_count + 4})"
                )
                params.extend(
                    [
                        chunk["document_id"],
                        chunk["chunk_index"],
                        chunk["content"],
                        PgVector(chunk["embedding"]),
                        chunk.get("group_id"),
                    ]
                )
                param_count += 5

            values_clause = ", ".join(values_list)
            query = f"""
                INSERT INTO chunks (document_id, chunk_index, content, embedding, group_id)
                VALUES {values_clause}
                RETURNING id
            """

            async with self.db_manager.connection() as conn:
                result = await conn.execute(query, params)
                rows = result.result()
                chunk_ids = [row["id"] for row in rows]
                logger.info(f"Created {len(chunk_ids)} chunks in batch")
                return chunk_ids
        except Exception as e:
            logger.error(f"Failed to create chunk batch: {e}")
            raise

    async def get_by_id(self, chunk_id: int) -> Optional[Chunk]:
        """
        Get chunk by ID.

        Args:
            chunk_id: Chunk ID

        Returns:
            Chunk if found, None otherwise

        Raises:
            Exception: Database errors
        """
        try:
            query = """
                SELECT id, document_id, group_id, chunk_index, content, embedding, created_at
                FROM chunks
                WHERE id = $1
            """
            async with self.db_manager.connection() as conn:
                result = await conn.execute(query, [chunk_id])
                rows = result.result()
                if rows:
                    chunk = self._row_to_chunk(rows[0])
                    logger.debug(f"Retrieved chunk ID {chunk_id}")
                    return chunk
                logger.debug(f"Chunk ID {chunk_id} not found")
                return None
        except Exception as e:
            logger.error(f"Failed to get chunk by ID {chunk_id}: {e}")
            raise

    async def list_by_document(
        self, document_id: int, offset: int = 0, limit: int = 100
    ) -> List[Chunk]:
        """
        List chunks for a document.

        Args:
            document_id: Document ID
            offset: Number of records to skip
            limit: Maximum number of records to return

        Returns:
            List of chunks

        Raises:
            Exception: Database errors
        """
        try:
            query = """
                SELECT id, document_id, group_id, chunk_index, content, embedding, created_at
                FROM chunks
                WHERE document_id = $1
                ORDER BY chunk_index ASC
                LIMIT $2 OFFSET $3
            """
            async with self.db_manager.connection() as conn:
                result = await conn.execute(query, [document_id, limit, offset])
                rows = result.result()
                chunks = [self._row_to_chunk(row) for row in rows]
                logger.debug(f"Listed {len(chunks)} chunks for document {document_id}")
                return chunks
        except Exception as e:
            logger.error(f"Failed to list chunks for document {document_id}: {e}")
            raise

    async def count_by_document(self, document_id: int) -> int:
        """
        Count chunks for a document.

        Args:
            document_id: Document ID

        Returns:
            Number of chunks

        Raises:
            Exception: Database errors
        """
        try:
            query = "SELECT COUNT(*) as count FROM chunks WHERE document_id = $1"
            async with self.db_manager.connection() as conn:
                result = await conn.execute(query, [document_id])
                rows = result.result()
                count = rows[0]["count"] if rows else 0
                logger.debug(f"Counted {count} chunks for document {document_id}")
                return count
        except Exception as e:
            logger.error(f"Failed to count chunks for document {document_id}: {e}")
            raise

    async def get_max_chunk_index(self, document_id: int) -> int:
        """
        Get the maximum chunk_index for a document.

        Args:
            document_id: Document ID

        Returns:
            Maximum chunk_index, or -1 if no chunks exist

        Raises:
            Exception: Database errors
        """
        try:
            query = "SELECT MAX(chunk_index) as max_index FROM chunks WHERE document_id = $1"
            async with self.db_manager.connection() as conn:
                result = await conn.execute(query, [document_id])
                rows = result.result()
                max_index = (
                    rows[0]["max_index"] if rows and rows[0]["max_index"] is not None else -1
                )
                logger.debug(f"Max chunk_index for document {document_id}: {max_index}")
                return max_index
        except Exception as e:
            logger.error(f"Failed to get max chunk_index for document {document_id}: {e}")
            raise

    async def vector_search(
        self,
        document_id: int,
        query_embedding: List[float],
        limit: int = 10,
        similarity_threshold: float = 0.7,
    ) -> List[SearchResult]:
        """
        Vector similarity search using pgvector.
        Returns chunks ordered by cosine similarity.

        Args:
            document_id: Document ID to search within
            query_embedding: Query embedding vector
            limit: Maximum number of results
            similarity_threshold: Minimum similarity score (0-1)

        Returns:
            List of SearchResult ordered by similarity

        Raises:
            Exception: Database errors
        """
        try:
            pg_vector = PgVector(query_embedding)
            query = """
                SELECT
                    id, document_id, group_id, chunk_index, content, embedding, created_at,
                    1 - (embedding <=> $1) AS similarity_score
                FROM chunks
                WHERE document_id = $2
                  AND embedding IS NOT NULL
                  AND 1 - (embedding <=> $1) >= $3
                ORDER BY embedding <=> $1
                LIMIT $4
            """
            async with self.db_manager.connection() as conn:
                result = await conn.execute(
                    query, [pg_vector, document_id, similarity_threshold, limit]
                )
                rows = result.result()
                results = []
                for rank, row in enumerate(rows, 1):
                    chunk = self._row_to_chunk(row)
                    score = float(row["similarity_score"])
                    results.append(SearchResult(chunk=chunk, score=score, rank=rank))
                logger.debug(
                    f"Vector search found {len(results)} chunks for document {document_id}"
                )
                return results
        except Exception as e:
            logger.error(f"Failed vector search for document {document_id}: {e}")
            raise

    async def bm25_search(
        self, document_id: int, query: str, limit: int = 10, min_score: float = 0.0
    ) -> List[SearchResult]:
        """
        BM25 full-text search using vchord_bm25.
        Returns chunks ordered by BM25 relevance.

        Args:
            document_id: Document ID to search within
            query: Search query text
            limit: Maximum number of results
            min_score: Minimum BM25 score threshold

        Returns:
            List of SearchResult ordered by BM25 score

        Raises:
            Exception: Database errors
        """
        try:
            # Use proper BM25 with to_bm25query() and tokenize()
            # The index name 'idx_chunks_bm25' must match the BM25 index in schema
            query_sql = """
                SELECT
                    id, document_id, group_id, chunk_index, content, embedding, created_at,
                    bm25_vector <&> to_bm25query('idx_chunks_bm25', tokenize($1, 'bert')) AS bm25_score
                FROM chunks
                WHERE document_id = $2
                  AND bm25_vector IS NOT NULL
                  AND bm25_vector <&> to_bm25query('idx_chunks_bm25', tokenize($1, 'bert')) >= $3
                ORDER BY bm25_score DESC
                LIMIT $4
            """
            async with self.db_manager.connection() as conn:
                result = await conn.execute(query_sql, [query, document_id, min_score, limit])
                rows = result.result()
                results = []
                for rank, row in enumerate(rows, 1):
                    chunk = self._row_to_chunk(row)
                    score = float(row["bm25_score"])
                    results.append(SearchResult(chunk=chunk, score=score, rank=rank))
                logger.debug(f"BM25 search found {len(results)} chunks for document {document_id}")
                return results
        except Exception as e:
            logger.error(f"Failed BM25 search for document {document_id}: {e}")
            raise

    async def hybrid_search(
        self,
        document_id: int,
        query: str,
        query_embedding: List[float],
        vector_weight: float = 0.7,
        bm25_weight: float = 0.3,
        limit: int = 10,
        min_vector_score: float = 0.0,
        min_bm25_score: float = 0.0,
    ) -> List[SearchResult]:
        """
        Hybrid search combining vector and BM25 with weighted scores.

        Algorithm:
        1. Perform vector search (top 50)
        2. Perform BM25 search using proper vchord_bm25 (top 50)
        3. Normalize scores to 0-1 range using min-max normalization
        4. Combine: final_score = (normalized_vector_score * vector_weight) + (normalized_bm25_score * bm25_weight)
        5. Return top N by final score

        Args:
            document_id: Document ID to search within
            query: Search query text for BM25
            query_embedding: Query embedding for vector search
            vector_weight: Weight for vector similarity (0-1)
            bm25_weight: Weight for BM25 score (0-1)
            limit: Maximum number of results
            min_vector_score: Minimum vector similarity threshold (0-1)
            min_bm25_score: Minimum BM25 score threshold

        Returns:
            List of SearchResult ordered by combined score

        Raises:
            Exception: Database errors
        """
        try:
            pg_vector = PgVector(query_embedding)

            # Get top 50 from each search for better combination
            top_k = min(50, limit * 5)

            query_sql = """
                WITH vector_results AS (
                    SELECT
                        id,
                        1 - (embedding <=> $1) AS vector_score
                    FROM chunks
                    WHERE document_id = $2 
                      AND embedding IS NOT NULL
                      AND 1 - (embedding <=> $1) >= $8
                    ORDER BY embedding <=> $1
                    LIMIT $3
                ),
                bm25_results AS (
                    SELECT
                        id,
                        bm25_vector <&> to_bm25query('idx_chunks_bm25', tokenize($4, 'bert')) AS bm25_score
                    FROM chunks
                    WHERE document_id = $2 
                      AND bm25_vector IS NOT NULL
                      AND bm25_vector <&> to_bm25query('idx_chunks_bm25', tokenize($4, 'bert')) >= $9
                    ORDER BY bm25_score DESC
                    LIMIT $3
                ),
                score_ranges AS (
                    SELECT
                        MAX(vector_score) AS max_vector,
                        MIN(vector_score) AS min_vector,
                        MAX(bm25_score) AS max_bm25,
                        MIN(bm25_score) AS min_bm25
                    FROM (
                        SELECT vector_score, NULL AS bm25_score FROM vector_results
                        UNION ALL
                        SELECT NULL AS vector_score, bm25_score FROM bm25_results
                    ) all_scores
                ),
                combined_results AS (
                    SELECT
                        c.id, c.document_id, c.group_id, c.chunk_index, c.content, c.embedding, c.created_at,
                        COALESCE(v.vector_score, 0) AS raw_vector_score,
                        COALESCE(b.bm25_score, 0) AS raw_bm25_score,
                        -- Normalize scores to 0-1 range using min-max normalization
                        CASE 
                            WHEN sr.max_vector > sr.min_vector THEN 
                                (COALESCE(v.vector_score, 0) - sr.min_vector) / (sr.max_vector - sr.min_vector)
                            ELSE COALESCE(v.vector_score, 0)
                        END AS norm_vector_score,
                        CASE 
                            WHEN sr.max_bm25 > sr.min_bm25 THEN 
                                (COALESCE(b.bm25_score, 0) - sr.min_bm25) / (sr.max_bm25 - sr.min_bm25)
                            ELSE COALESCE(b.bm25_score, 0)
                        END AS norm_bm25_score
                    FROM chunks c
                    LEFT JOIN vector_results v ON c.id = v.id
                    LEFT JOIN bm25_results b ON c.id = b.id
                    CROSS JOIN score_ranges sr
                    WHERE c.document_id = $2
                      AND (v.vector_score IS NOT NULL OR b.bm25_score IS NOT NULL)
                ),
                final_results AS (
                    SELECT 
                        *,
                        (norm_vector_score * $5 + norm_bm25_score * $6) AS combined_score
                    FROM combined_results
                )
                SELECT * FROM final_results
                WHERE combined_score > 0
                ORDER BY combined_score DESC
                LIMIT $7
            """

            async with self.db_manager.connection() as conn:
                result = await conn.execute(
                    query_sql,
                    [
                        pg_vector,
                        document_id,
                        top_k,
                        query,
                        vector_weight,
                        bm25_weight,
                        limit,
                        min_vector_score,
                        min_bm25_score,
                    ],
                )
                rows = result.result()
                results = []
                for rank, row in enumerate(rows, 1):
                    chunk = self._row_to_chunk(row)
                    score = float(row["combined_score"])
                    results.append(SearchResult(chunk=chunk, score=score, rank=rank))
                logger.debug(
                    f"Hybrid search found {len(results)} chunks for document {document_id} "
                    f"(vector_weight={vector_weight}, bm25_weight={bm25_weight})"
                )
                return results
        except Exception as e:
            logger.error(f"Failed hybrid search for document {document_id}: {e}")
            raise

    async def delete_by_document(self, document_id: int) -> int:
        """
        Delete all chunks for a document. Returns count deleted.

        Args:
            document_id: Document ID

        Returns:
            Number of chunks deleted

        Raises:
            Exception: Database errors
        """
        try:
            query = "DELETE FROM chunks WHERE document_id = $1"
            async with self.db_manager.connection() as conn:
                result = await conn.execute(query, [document_id])
                # PSQLPy DELETE returns result that can be checked
                result_str = str(result.result())
                # Try to extract count from result like "DELETE 5"
                if "DELETE " in result_str:
                    try:
                        count = int(result_str.split("DELETE ")[1])
                        logger.info(f"Deleted {count} chunks for document {document_id}")
                        return count
                    except (ValueError, IndexError):
                        pass
                # Fallback: assume success and return 0 (count unknown)
                logger.info(f"Deleted chunks for document {document_id}")
                return 0
        except Exception as e:
            logger.error(f"Failed to delete chunks for document {document_id}: {e}")
            raise

    async def delete_by_group(self, group_id: int) -> int:
        """
        Delete all chunks for a group. Returns count deleted.

        Args:
            group_id: Group ID

        Returns:
            Number of chunks deleted

        Raises:
            Exception: Database errors
        """
        try:
            query = "DELETE FROM chunks WHERE group_id = $1"
            async with self.db_manager.connection() as conn:
                result = await conn.execute(query, [group_id])
                # PSQLPy DELETE returns result that can be checked
                result_str = str(result.result())
                # Try to extract count from result like "DELETE 5"
                if "DELETE " in result_str:
                    try:
                        count = int(result_str.split("DELETE ")[1])
                        logger.info(f"Deleted {count} chunks for group {group_id}")
                        return count
                    except (ValueError, IndexError):
                        pass
                # Fallback: assume success and return 0 (count unknown)
                logger.info(f"Deleted chunks for group {group_id}")
                return 0
        except Exception as e:
            logger.error(f"Failed to delete chunks for group {group_id}: {e}")
            raise

    def _row_to_chunk(self, row: dict) -> Chunk:
        """
        Convert database row dict to Chunk model.

        PSQLPy returns rows as dicts with column names as keys.

        Args:
            row: Database row dict with columns: id, document_id, group_id,
                 chunk_index, content, embedding, created_at

        Returns:
            Chunk model instance
        """
        return Chunk(
            id=row["id"],
            document_id=row["document_id"],
            group_id=row.get("group_id"),
            chunk_index=row["chunk_index"],
            content=row["content"],
            embedding=row["embedding"],  # PgVector should convert back to list
            created_at=row["created_at"],
        )
