"""Convenience class to create Vector Stores."""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../../../nbs/01_ingest.stores.factory.ipynb.

# %% auto 0
__all__ = ['VectorStoreFactory']

# %% ../../../nbs/01_ingest.stores.factory.ipynb 3
from .dense import DenseStore, ChromaStore
from .sparse import SparseStore, WhooshStore, ElasticsearchSparseStore
from .dual import DualStore, ElasticsearchStore

class VectorStoreFactory:
    """
    Convenience factory for creating vector stores with sensible defaults.
    
    Provides a simple interface to create the most commonly used vector stores:
    - ChromaStore (default) - Dense vector store using Chroma
    - WhooshStore - Sparse text search using Whoosh
    - ElasticsearchStore - Unified dense + sparse using Elasticsearch
    """
    
    @classmethod
    def create(cls, 
               kind: str = 'chroma',
               persist_location: str = None,
               **kwargs):
        """
        Create a vector store instance.
        
        Args:
            kind: Type of store to create. One of:
                  - 'chroma' (default): ChromaStore for dense vector search
                  - 'whoosh': WhooshStore for sparse text search  
                  - 'chroma+-whoosh': a DualStore using  ChromaStore and WhooshStore 
                  - 'elasticsearch': ElasticsearchStore for unified dense + sparse
                  - 'elasticsearch_sparse': For use with pre-existing Elasticsearch indices without dense vectors
            persist_location: Where to store the index/database
            **kwargs: Additional arguments passed to the store constructor
            
        Returns:
            VectorStore instance
            
        Examples:
            # Create default ChromaStore
            store = VectorStoreFactory.create()
            
            # Create WhooshStore for text search
            store = VectorStoreFactory.create('whoosh', persist_location='./search_index')
            
            # Create ElasticsearchStore for hybrid search
            store = VectorStoreFactory.create('elasticsearch', 
                                             persist_location='http://localhost:9200',
                                             index_name='my_docs')
        """
        if kind == 'chroma':
            return ChromaStore(persist_location=persist_location, **kwargs)
        elif kind == 'whoosh':
            return WhooshStore(persist_location=persist_location, **kwargs)
        elif kind== 'chroma+whoosh':
            dense_path = os.path.join(self.vectordb_path, 'dense')
            sparse_path = os.path.join(self.vectordb_path, 'sparse')
            return DualStore(
                    sparse_persist_location=sparse_path,
                    dense_persist_location=dense_path,
                    **kwargs
                )
        elif kind == 'elasticsearch':
            return ElasticsearchStore(persist_location=persist_location, **kwargs)
        elif kind == 'elasticsearch_sparse':
            return ElasticsearchSparseStore(
                        persist_location=persist_location, # 'https://localhost:9200'
                        **kwargs                           # e.g., index_name, basic_auth, etc.
                    )
        else:
            raise ValueError(f"Unknown store kind: {kind}. Supported vector store types: 'chroma', 'whoosh', 'chroma-whoosh', 'elasticsearch', 'elasticsearch-sparse")

