"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const client_codedeploy_1 = require("@aws-sdk/client-codedeploy");
const logger_1 = require("./logger");
/**
 * The lambda function called from CloudFormation for this custom resource.
 *
 * @param event
 * @returns attribues of the deployment that was created
 */
async function handler(event) {
    const logger = new logger_1.Logger();
    const codedeployClient = new client_codedeploy_1.CodeDeploy({});
    logger.appendKeys({
        stackEvent: event.RequestType,
    });
    switch (event.RequestType) {
        case 'Create':
        case 'Update': {
            // create deployment
            const props = event.ResourceProperties;
            let autoRollbackConfiguration;
            if (props.autoRollbackConfigurationEnabled === 'true') {
                autoRollbackConfiguration = {
                    enabled: true,
                    events: props.autoRollbackConfigurationEvents.split(','),
                };
            }
            else if (props.autoRollbackConfigurationEnabled === 'false') {
                autoRollbackConfiguration = {
                    enabled: false,
                };
            }
            const resp = await codedeployClient.createDeployment({
                applicationName: props.applicationName,
                deploymentConfigName: props.deploymentConfigName,
                deploymentGroupName: props.deploymentGroupName,
                autoRollbackConfiguration,
                description: props.description,
                revision: {
                    revisionType: 'AppSpecContent',
                    appSpecContent: {
                        content: props.revisionAppSpecContent,
                    },
                },
            });
            if (!resp.deploymentId) {
                throw new Error('No deploymentId received from call to CreateDeployment');
            }
            logger.appendKeys({
                deploymentId: resp.deploymentId,
            });
            logger.info('Created new deployment');
            return {
                PhysicalResourceId: resp.deploymentId,
                Data: {
                    deploymentId: resp.deploymentId,
                },
            };
        }
        case 'Delete':
            logger.appendKeys({
                deploymentId: event.PhysicalResourceId,
            });
            // cancel deployment and rollback
            try {
                const resp = await codedeployClient.stopDeployment({
                    deploymentId: event.PhysicalResourceId,
                    autoRollbackEnabled: true,
                });
                logger.info(`Stopped deployment: ${resp.status} ${resp.statusMessage}`);
            }
            catch (e) {
                logger.warn('Ignoring error', { error: e });
            }
            return {
                PhysicalResourceId: event.PhysicalResourceId,
                Data: {
                    deploymentId: event.PhysicalResourceId,
                },
            };
        default:
            logger.error('Unknown stack event');
            throw new Error(`Unknown request type: ${event.RequestType}`);
    }
}
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,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