"""
Event system for event-driven backtesting architecture.
"""

from abc import ABC, abstractmethod
from datetime import datetime
from typing import Any, Dict, Optional
from enum import Enum
from dataclasses import dataclass

class EventType(Enum):
    """Types of events in the backtesting system."""
    MARKET = "MARKET"
    SIGNAL = "SIGNAL"
    ORDER = "ORDER"
    FILL = "FILL"

class Event(ABC):
    """Abstract base class for all events."""
    
    def __init__(self, timestamp: datetime, event_type: EventType):
        self.timestamp = timestamp
        self.event_type = event_type
    
    @abstractmethod
    def __str__(self) -> str:
        """String representation of the event."""
        pass

class MarketEvent(Event):
    """Market data event containing price and volume information."""
    
    def __init__(self, timestamp: datetime, symbol: str, open_price: float,
                 high_price: float, low_price: float, close_price: float,
                 volume: int, bid_price: Optional[float] = None,
                 ask_price: Optional[float] = None, bid_size: Optional[int] = None,
                 ask_size: Optional[int] = None):
        super().__init__(timestamp, EventType.MARKET)
        self.symbol = symbol
        self.open_price = open_price
        self.high_price = high_price
        self.low_price = low_price
        self.close_price = close_price
        self.volume = volume
        
        # Set bid/ask if not provided (use close price with small spread)
        if bid_price is None:
            spread = max(0.01, close_price * 0.001)  # 0.1% spread minimum $0.01
            self.bid_price = close_price - spread / 2
            self.ask_price = close_price + spread / 2
        else:
            self.bid_price = bid_price
            self.ask_price = ask_price
        
        if bid_size is None:
            self.bid_size = volume // 10  # Assume 10% of volume at bid
            self.ask_size = volume // 10
        else:
            self.bid_size = bid_size
            self.ask_size = ask_size
    
    def __str__(self) -> str:
        return (f"MarketEvent({self.symbol} @ {self.timestamp}: "
                f"OHLC={self.open_price:.2f}/{self.high_price:.2f}/"
                f"{self.low_price:.2f}/{self.close_price:.2f}, Vol={self.volume})")

class SignalType(Enum):
    """Types of trading signals."""
    BUY = "BUY"
    SELL = "SELL"
    HOLD = "HOLD"
    EXIT = "EXIT"

class SignalEvent(Event):
    """Trading signal event generated by strategy."""
    
    def __init__(self, timestamp: datetime, symbol: str, signal_type: SignalType,
                 strength: float = 1.0, target_quantity: Optional[int] = None,
                 target_price: Optional[float] = None, stop_loss: Optional[float] = None,
                 take_profit: Optional[float] = None, metadata: Optional[Dict[str, Any]] = None):
        super().__init__(timestamp, EventType.SIGNAL)
        self.symbol = symbol
        self.signal_type = signal_type
        self.strength = strength
        self.target_quantity = target_quantity
        self.target_price = target_price
        self.stop_loss = stop_loss
        self.take_profit = take_profit
        self.metadata = metadata
    
    def __str__(self) -> str:
        return (f"SignalEvent({self.symbol} @ {self.timestamp}: "
                f"{self.signal_type.value}, strength={self.strength:.2f})")

class OrderType(Enum):
    """Types of orders."""
    MARKET = "MARKET"
    LIMIT = "LIMIT"
    STOP = "STOP"
    STOP_LIMIT = "STOP_LIMIT"
    BRACKET = "BRACKET"
    OCO = "OCO"  # One-Cancels-Other

class OrderSide(Enum):
    """Order side (buy/sell)."""
    BUY = "BUY"
    SELL = "SELL"

class OrderEvent(Event):
    """Order event to be sent to execution system."""
    
    def __init__(self, timestamp: datetime, symbol: str, order_type: OrderType,
                 side: OrderSide, quantity: int, price: Optional[float] = None,
                 stop_price: Optional[float] = None, time_in_force: str = "DAY",
                 order_id: Optional[str] = None, parent_order_id: Optional[str] = None,
                 metadata: Optional[Dict[str, Any]] = None):
        super().__init__(timestamp, EventType.ORDER)
        self.symbol = symbol
        self.order_type = order_type
        self.side = side
        self.quantity = quantity
        self.price = price
        self.stop_price = stop_price
        self.time_in_force = time_in_force
        self.parent_order_id = parent_order_id
        self.metadata = metadata
        
        if order_id is None:
            # Generate unique order ID
            self.order_id = f"{symbol}_{timestamp.strftime('%Y%m%d_%H%M%S_%f')}"
        else:
            self.order_id = order_id
    
    def __str__(self) -> str:
        price_str = f" @ {self.price:.2f}" if self.price else ""
        return (f"OrderEvent({self.order_id}: {self.side.value} {self.quantity} "
                f"{self.symbol} {self.order_type.value}{price_str})")

class FillStatus(Enum):
    """Fill status."""
    FILLED = "FILLED"
    PARTIAL_FILL = "PARTIAL_FILL"
    REJECTED = "REJECTED"

class FillEvent(Event):
    """Order fill event from execution system."""
    
    def __init__(self, timestamp: datetime, order_id: str, symbol: str,
                 side: OrderSide, quantity: int, fill_price: float,
                 fill_quantity: int, remaining_quantity: int, commission: float,
                 fill_status: FillStatus, exchange: str = "SIMULATED",
                 execution_id: Optional[str] = None, metadata: Optional[Dict[str, Any]] = None):
        super().__init__(timestamp, EventType.FILL)
        self.order_id = order_id
        self.symbol = symbol
        self.side = side
        self.quantity = quantity
        self.fill_price = fill_price
        self.fill_quantity = fill_quantity
        self.remaining_quantity = remaining_quantity
        self.commission = commission
        self.fill_status = fill_status
        self.exchange = exchange
        self.metadata = metadata
        
        if execution_id is None:
            self.execution_id = f"EXEC_{timestamp.strftime('%Y%m%d_%H%M%S_%f')}"
        else:
            self.execution_id = execution_id
    
    def __str__(self) -> str:
        return (f"FillEvent({self.execution_id}: {self.fill_status.value} "
                f"{self.fill_quantity}/{self.quantity} {self.symbol} "
                f"@ {self.fill_price:.2f}, comm={self.commission:.2f})")

# Event queue for managing events
import queue
from typing import List

class EventQueue:
    """Thread-safe event queue for backtesting."""
    
    def __init__(self):
        self._queue = queue.Queue()
        self._event_history: List[Event] = []
    
    def put(self, event: Event) -> None:
        """Add event to queue."""
        self._queue.put(event)
        self._event_history.append(event)
    
    def get(self, block: bool = True, timeout: Optional[float] = None) -> Event:
        """Get next event from queue."""
        return self._queue.get(block=block, timeout=timeout)
    
    def empty(self) -> bool:
        """Check if queue is empty."""
        return self._queue.empty()
    
    def size(self) -> int:
        """Get queue size."""
        return self._queue.qsize()
    
    def get_history(self) -> List[Event]:
        """Get event history."""
        return self._event_history.copy()
    
    def clear_history(self) -> None:
        """Clear event history."""
        self._event_history.clear()

# Event handlers
class EventHandler(ABC):
    """Abstract base class for event handlers."""
    
    @abstractmethod
    def handle_event(self, event: Event) -> None:
        """Handle an event."""
        pass

class EventDispatcher:
    """Event dispatcher for routing events to appropriate handlers."""
    
    def __init__(self):
        self._handlers: Dict[EventType, List[EventHandler]] = {
            EventType.MARKET: [],
            EventType.SIGNAL: [],
            EventType.ORDER: [],
            EventType.FILL: []
        }
    
    def register_handler(self, event_type: EventType, handler: EventHandler) -> None:
        """Register an event handler."""
        self._handlers[event_type].append(handler)
    
    def unregister_handler(self, event_type: EventType, handler: EventHandler) -> None:
        """Unregister an event handler."""
        if handler in self._handlers[event_type]:
            self._handlers[event_type].remove(handler)
    
    def dispatch_event(self, event: Event) -> None:
        """Dispatch event to registered handlers."""
        for handler in self._handlers[event.event_type]:
            try:
                handler.handle_event(event)
            except Exception as e:
                print(f"Error handling event {event}: {e}")
                # In production, this would be logged properly
    
    def get_handlers(self, event_type: EventType) -> List[EventHandler]:
        """Get handlers for event type."""
        return self._handlers[event_type].copy()

if __name__ == "__main__":
    # Example usage
    from datetime import datetime
    
    # Create sample events
    market_event = MarketEvent(
        timestamp=datetime.now(),
        symbol="AAPL",
        open_price=150.0,
        high_price=152.0,
        low_price=149.0,
        close_price=151.0,
        volume=1000000
    )
    
    signal_event = SignalEvent(
        timestamp=datetime.now(),
        symbol="AAPL",
        signal_type=SignalType.BUY,
        strength=0.8,
        target_quantity=100
    )
    
    order_event = OrderEvent(
        timestamp=datetime.now(),
        symbol="AAPL",
        order_type=OrderType.MARKET,
        side=OrderSide.BUY,
        quantity=100
    )
    
    fill_event = FillEvent(
        timestamp=datetime.now(),
        order_id=order_event.order_id,
        symbol="AAPL",
        side=OrderSide.BUY,
        quantity=100,
        fill_price=151.0,
        fill_quantity=100,
        remaining_quantity=0,
        commission=1.0,
        fill_status=FillStatus.FILLED
    )
    
    # Test event queue
    event_queue = EventQueue()
    event_queue.put(market_event)
    event_queue.put(signal_event)
    event_queue.put(order_event)
    event_queue.put(fill_event)
    
    print("Event Queue Example:")
    print(f"Queue size: {event_queue.size()}")
    
    while not event_queue.empty():
        event = event_queue.get()
        print(f"  {event}")
    
    print(f"\nEvent history: {len(event_queue.get_history())} events")