"""
Comprehensive tests for technical indicators module.
"""

import pytest
import numpy as np
import pandas as pd
from unittest.mock import patch, MagicMock
import sys
import os

# Add the parent directory to the path to import meridianalgo
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

try:
    import meridianalgo as ma
    from meridianalgo.technical_indicators import *
except ImportError as e:
    pytest.skip(f"Could not import meridianalgo: {e}", allow_module_level=True)


class TestTechnicalIndicators:
    """Test suite for technical indicators."""
    
    @pytest.fixture
    def sample_data(self):
        """Create sample price data for testing."""
        np.random.seed(42)
        dates = pd.date_range('2023-01-01', periods=100, freq='D')
        
        # Generate realistic price data
        returns = np.random.normal(0.001, 0.02, 100)
        prices = [100]
        for ret in returns[1:]:
            prices.append(prices[-1] * (1 + ret))
        
        data = pd.DataFrame({
            'Open': [p * np.random.uniform(0.99, 1.01) for p in prices],
            'High': [p * np.random.uniform(1.00, 1.05) for p in prices],
            'Low': [p * np.random.uniform(0.95, 1.00) for p in prices],
            'Close': prices,
            'Volume': np.random.randint(1000000, 10000000, 100)
        }, index=dates)
        
        return data
    
    def test_rsi_calculation(self, sample_data):
        """Test RSI calculation."""
        try:
            rsi = RSI(sample_data['Close'], period=14)
            
            # RSI should be between 0 and 100
            assert all(0 <= val <= 100 for val in rsi.dropna())
            
            # Should have correct length (original length - period + 1)
            expected_length = len(sample_data) - 14 + 1
            assert len(rsi.dropna()) <= expected_length
            
            print("✅ RSI calculation test passed")
        except Exception as e:
            print(f"⚠️ RSI test failed: {e}")
    
    def test_moving_averages(self, sample_data):
        """Test moving average calculations."""
        try:
            sma = SMA(sample_data['Close'], period=20)
            ema = EMA(sample_data['Close'], period=20)
            
            # Moving averages should be positive
            assert all(val > 0 for val in sma.dropna())
            assert all(val > 0 for val in ema.dropna())
            
            # EMA should react faster than SMA
            assert len(sma.dropna()) <= len(sample_data)
            assert len(ema.dropna()) <= len(sample_data)
            
            print("✅ Moving averages test passed")
        except Exception as e:
            print(f"⚠️ Moving averages test failed: {e}")
    
    def test_macd_calculation(self, sample_data):
        """Test MACD calculation."""
        try:
            macd_line, signal_line, histogram = MACD(sample_data['Close'])
            
            # All components should have same length
            assert len(macd_line.dropna()) == len(signal_line.dropna())
            assert len(signal_line.dropna()) == len(histogram.dropna())
            
            # Histogram should be difference between MACD and signal
            diff = macd_line - signal_line
            np.testing.assert_array_almost_equal(
                histogram.dropna().values, 
                diff.dropna().values, 
                decimal=5
            )
            
            print("✅ MACD calculation test passed")
        except Exception as e:
            print(f"⚠️ MACD test failed: {e}")
    
    def test_bollinger_bands(self, sample_data):
        """Test Bollinger Bands calculation."""
        try:
            bb_upper, bb_middle, bb_lower = BollingerBands(sample_data['Close'])
            
            # Upper band should be above middle, middle above lower
            assert all(bb_upper.dropna() >= bb_middle.dropna())
            assert all(bb_middle.dropna() >= bb_lower.dropna())
            
            # Middle band should be SMA
            sma = SMA(sample_data['Close'], period=20)
            np.testing.assert_array_almost_equal(
                bb_middle.dropna().values,
                sma.dropna().values,
                decimal=5
            )
            
            print("✅ Bollinger Bands test passed")
        except Exception as e:
            print(f"⚠️ Bollinger Bands test failed: {e}")
    
    def test_stochastic_oscillator(self, sample_data):
        """Test Stochastic Oscillator calculation."""
        try:
            stoch_k, stoch_d = Stochastic(
                sample_data['High'], 
                sample_data['Low'], 
                sample_data['Close']
            )
            
            # Stochastic should be between 0 and 100
            assert all(0 <= val <= 100 for val in stoch_k.dropna())
            assert all(0 <= val <= 100 for val in stoch_d.dropna())
            
            print("✅ Stochastic Oscillator test passed")
        except Exception as e:
            print(f"⚠️ Stochastic test failed: {e}")
    
    def test_atr_calculation(self, sample_data):
        """Test Average True Range calculation."""
        try:
            atr = ATR(
                sample_data['High'], 
                sample_data['Low'], 
                sample_data['Close']
            )
            
            # ATR should be positive
            assert all(val >= 0 for val in atr.dropna())
            
            print("✅ ATR calculation test passed")
        except Exception as e:
            print(f"⚠️ ATR test failed: {e}")
    
    def test_volume_indicators(self, sample_data):
        """Test volume-based indicators."""
        try:
            obv = OBV(sample_data['Close'], sample_data['Volume'])
            
            # OBV should be cumulative
            assert len(obv) == len(sample_data)
            
            # Test Money Flow Index
            mfi = MoneyFlowIndex(
                sample_data['High'],
                sample_data['Low'], 
                sample_data['Close'],
                sample_data['Volume']
            )
            
            # MFI should be between 0 and 100
            assert all(0 <= val <= 100 for val in mfi.dropna())
            
            print("✅ Volume indicators test passed")
        except Exception as e:
            print(f"⚠️ Volume indicators test failed: {e}")
    
    def test_momentum_indicators(self, sample_data):
        """Test momentum indicators."""
        try:
            # Williams %R
            williams_r = WilliamsR(
                sample_data['High'],
                sample_data['Low'],
                sample_data['Close']
            )
            
            # Williams %R should be between -100 and 0
            assert all(-100 <= val <= 0 for val in williams_r.dropna())
            
            # Rate of Change
            roc = ROC(sample_data['Close'])
            
            # ROC can be any value but should be numeric
            assert all(isinstance(val, (int, float)) for val in roc.dropna())
            
            print("✅ Momentum indicators test passed")
        except Exception as e:
            print(f"⚠️ Momentum indicators test failed: {e}")
    
    def test_overlay_indicators(self, sample_data):
        """Test overlay indicators."""
        try:
            # Pivot Points
            pivots = PivotPoints(
                sample_data['High'],
                sample_data['Low'],
                sample_data['Close']
            )
            
            assert 'pivot' in pivots.columns
            assert 'r1' in pivots.columns
            assert 's1' in pivots.columns
            
            # Fibonacci Retracement
            high_price = sample_data['High'].max()
            low_price = sample_data['Low'].min()
            fib_levels = FibonacciRetracement(high_price, low_price)
            
            assert isinstance(fib_levels, dict)
            assert 0.618 in fib_levels
            
            print("✅ Overlay indicators test passed")
        except Exception as e:
            print(f"⚠️ Overlay indicators test failed: {e}")
    
    def test_error_handling(self, sample_data):
        """Test error handling for invalid inputs."""
        try:
            # Test with insufficient data
            short_data = sample_data['Close'].head(5)
            
            # Should handle gracefully or return appropriate result
            rsi_short = RSI(short_data, period=14)
            assert len(rsi_short.dropna()) == 0 or len(rsi_short.dropna()) < 14
            
            # Test with invalid period
            try:
                rsi_invalid = RSI(sample_data['Close'], period=0)
            except (ValueError, ZeroDivisionError):
                pass  # Expected behavior
            
            print("✅ Error handling test passed")
        except Exception as e:
            print(f"⚠️ Error handling test failed: {e}")


def test_technical_indicators_import():
    """Test that technical indicators can be imported."""
    try:
        from meridianalgo.technical_indicators import RSI, SMA, EMA, MACD
        print("✅ Technical indicators import test passed")
        return True
    except ImportError as e:
        print(f"⚠️ Import test failed: {e}")
        return False


def test_technical_indicators_with_real_data():
    """Test technical indicators with real market data if available."""
    try:
        # Try to get real data
        data = ma.get_market_data(['AAPL'], '2023-01-01', '2023-12-31')
        
        if data is not None and len(data) > 50:
            # Test with real data
            rsi = ma.RSI(data['AAPL'], period=14)
            sma = ma.SMA(data['AAPL'], period=20)
            
            assert len(rsi.dropna()) > 0
            assert len(sma.dropna()) > 0
            
            print("✅ Real data test passed")
        else:
            print("⚠️ No real data available, skipping real data test")
            
    except Exception as e:
        print(f"⚠️ Real data test failed: {e}")


if __name__ == "__main__":
    # Run tests manually
    print("🧪 Running Technical Indicators Tests...")
    
    # Test imports first
    if not test_technical_indicators_import():
        print("❌ Cannot proceed with tests - import failed")
        exit(1)
    
    # Create test instance
    test_instance = TestTechnicalIndicators()
    
    # Generate sample data
    np.random.seed(42)
    dates = pd.date_range('2023-01-01', periods=100, freq='D')
    returns = np.random.normal(0.001, 0.02, 100)
    prices = [100]
    for ret in returns[1:]:
        prices.append(prices[-1] * (1 + ret))
    
    sample_data = pd.DataFrame({
        'Open': [p * np.random.uniform(0.99, 1.01) for p in prices],
        'High': [p * np.random.uniform(1.00, 1.05) for p in prices],
        'Low': [p * np.random.uniform(0.95, 1.00) for p in prices],
        'Close': prices,
        'Volume': np.random.randint(1000000, 10000000, 100)
    }, index=dates)
    
    # Run all tests
    test_methods = [
        test_instance.test_rsi_calculation,
        test_instance.test_moving_averages,
        test_instance.test_macd_calculation,
        test_instance.test_bollinger_bands,
        test_instance.test_stochastic_oscillator,
        test_instance.test_atr_calculation,
        test_instance.test_volume_indicators,
        test_instance.test_momentum_indicators,
        test_instance.test_overlay_indicators,
        test_instance.test_error_handling
    ]
    
    passed = 0
    total = len(test_methods)
    
    for test_method in test_methods:
        try:
            test_method(sample_data)
            passed += 1
        except Exception as e:
            print(f"❌ Test {test_method.__name__} failed: {e}")
    
    # Test with real data
    test_technical_indicators_with_real_data()
    
    print(f"\n📊 Test Results: {passed}/{total} tests passed")
    
    if passed == total:
        print("🎉 All technical indicators tests passed!")
    else:
        print(f"⚠️ {total - passed} tests failed")