import threading
import time
import json
import random
import paho.mqtt.client as mqtt
from django.conf import settings
from django.template.loader import render_to_string
from abc import ABC, abstractmethod
from simo.core.utils.helpers import classproperty
from simo.core.events import GatewayObjectCommand, get_event_obj
from simo.core.loggers import get_gw_logger
from simo.core.utils.mqtt import connect_with_retry, install_reconnect_handler




class BaseGatewayHandler(ABC):
    # Whether core should auto-create a Gateway row for this handler
    # at hub startup (via on_http_start). Opt-in to avoid noisy defaults
    # and side effects for handlers that require credentials/hardware.
    auto_create = False

    @property
    @abstractmethod
    def name(self) -> str:
        """
        :return: name of this gateway decriptor
        """

    @property
    @abstractmethod
    def config_form(self):
        """
        :return: Config form of this gateway class
        """

    @classproperty
    @classmethod
    def uid(cls):
        return ".".join([cls.__module__, cls.__name__])


    @classproperty
    @classmethod
    def info(cls):
        return

    def __init__(self, gateway_instance):
        self.gateway_instance = gateway_instance
        super().__init__()
        assert self.name, "Gateway needs a name"
        assert self.config_form, "Gateway needs config_form"



class BaseObjectCommandsGatewayHandler(BaseGatewayHandler):
    periodic_tasks = ()

    exit = None

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.mqtt_client = mqtt.Client()
        self.mqtt_client.username_pw_set('root', settings.SECRET_KEY)
        self.mqtt_client.on_connect = self._on_mqtt_connect
        self.mqtt_client.on_message = self._on_mqtt_message
        try:
            self.mqtt_client.reconnect_delay_set(min_delay=1, max_delay=30)
        except Exception:
            pass


    def run(self, exit):
        self.exit = exit
        self.logger = get_gw_logger(self.gateway_instance.id)

        for task, period in self.periodic_tasks:
            threading.Thread(
                target=self._run_periodic_task, args=(self.exit, task, period), daemon=True
            ).start()

        install_reconnect_handler(
            self.mqtt_client,
            logger=self.logger,
            stop_event=self.exit,
            description=f"{self.gateway_instance} MQTT",
        )
        if not connect_with_retry(
            self.mqtt_client,
            logger=self.logger,
            stop_event=self.exit,
            description=f"{self.gateway_instance} MQTT",
        ):
            return

        self.mqtt_client.loop_start()

        while not self.exit.is_set():
            time.sleep(1)

        self.mqtt_client.loop_stop()
        try:
            self.mqtt_client.disconnect()
        except Exception:
            pass

    def _run_periodic_task(self, exit, task, period):
        first_run = True
        while not exit.is_set():
            try:
                #print(f"Run periodic task {task}!")
                getattr(self, task)()
            except Exception as e:
                self.logger.error(e, exc_info=True)
            # spread tasks around so that they do not happen all
            # at once all the time
            if first_run:
                first_run = False
                randomized_sleep = random.randint(0, period) + random.random()
                time.sleep(randomized_sleep)
            else:
                time.sleep(period)


    def _on_mqtt_connect(self, mqtt_client, userdata, flags, rc):
        print("MQTT Connected!")
        self.mqtt_client = mqtt_client
        command = GatewayObjectCommand(self.gateway_instance)
        self.mqtt_client.subscribe(command.get_topic())

    def _on_mqtt_message(self, client, userdata, msg):
        from simo.core.models import Component
        payload = json.loads(msg.payload)
        if 'set_val' in payload:
            component = get_event_obj(payload, Component)
            if not component:
                return
            print(f"Perform Value ({str(payload['set_val'])}) Send to {component}")
            try:
                self.perform_value_send(component, payload['set_val'])
            except Exception as e:
                self.logger.error(e, exc_info=True)

        if 'bulk_send' in payload:
            self.perform_bulk_send(payload['bulk_send'])

    def perform_value_send(self, component, value):
        raise NotImplemented()

    def perform_bulk_send(self, data):
        from simo.core.models import Component
        for comp_id, val in data.items():
            component = Component.objects.filter(
                pk=comp_id, gateway=self.gateway_instance
            ).first()
            if not component:
                continue
            try:
                self.perform_value_send(component, val)
            except Exception as e:
                self.logger.error(e, exc_info=True)
