import dataclasses
import json
import os
import re
import sys
from pathlib import Path
from typing import List, Optional

from chalk._monitoring.Chart import Chart
from chalk._monitoring.gql_conversion import convert_chart
from chalk._version import __version__
from chalk.config.project_config import ProjectSettings, load_project_config
from chalk.features import FeatureSetBase, unwrap_feature
from chalk.features.resolver import Resolver, SinkResolver, StreamResolver
from chalk.importer import FailedImport
from chalk.parsed._graph_validation import validate_graph
from chalk.parsed.duplicate_input_gql import (
    ChalkPYInfo,
    EnvironmentSettingsGQL,
    FeatureSettings,
    MetadataSettings,
    ProjectSettingsGQL,
    UpsertGraphGQL,
    UpsertSQLSourceGQL,
    ValidationSettings,
)
from chalk.parsed.json_conversions import convert_type_to_gql, gather_cdc_sources
from chalk.sql._internal.sql_source import BaseSQLSource
from chalk.utils import paths
from chalk.utils.paths import get_directory_root


def _is_relative_to(x: Path, other: Path) -> bool:
    try:
        x.relative_to(other)
        return True
    except ValueError:
        return False


def project_settings_to_gql(config: ProjectSettings) -> ProjectSettingsGQL:
    def read_packages(filename: str) -> Optional[List[str]]:
        reqs = list()
        try:
            with open(filename) as f:
                for r in f.readlines():
                    cleaned = re.sub("#.*", "", r).rstrip("\n").strip()
                    if cleaned != "":
                        reqs.append(cleaned)
            return reqs
        except OSError:
            return None

    return ProjectSettingsGQL(
        project=config.project,
        environments=None
        if config.environments is None
        else [
            EnvironmentSettingsGQL(
                id=i,
                runtime=e.runtime,
                requirements=e.requirements,
                dockerfile=e.dockerfile,
                requiresPackages=None
                if e.requirements is None
                else read_packages(
                    os.path.join(
                        os.path.dirname(config.local_path),
                        e.requirements,
                    )
                ),
            )
            for i, e in config.environments.items()
        ],
        validation=ValidationSettings(
            feature=FeatureSettings(
                metadata=[MetadataSettings(name=m.name, missing=m.missing) for m in config.validation.feature.metadata]
                if config.validation.feature.metadata
                else None
            )
            if config.validation.feature
            else None
        )
        if config.validation
        else None,
    )


def get_registered_types_as_json(scope_to: Path, failed: List[FailedImport], indent: int = 2) -> str:
    features = []
    feature_classes = []
    for x in FeatureSetBase.registry.values():
        if x.__module__ in sys.modules and _is_relative_to(paths.get_classpath(x), scope_to):
            feature_classes.append(convert_type_to_gql(x))
            for feature in x.features:
                if not feature.is_autogenerated and not feature.no_display:
                    try:
                        converted = convert_type_to_gql(feature)
                        features.append(converted)
                    except Exception as e:
                        raise e

    project_path = get_directory_root()
    path_prefix_to_remove = project_path.as_posix() if project_path is not None else None

    stream_resolvers = []
    for stream in StreamResolver.registry:
        if _is_relative_to(Path(stream.filename), scope_to):
            stream_resolvers.append(convert_type_to_gql(stream, path_prefix=path_prefix_to_remove))

    resolvers = []
    for resolver in Resolver.registry:
        if _is_relative_to(Path(resolver.filename), scope_to):
            resolvers.append(convert_type_to_gql(resolver, path_prefix=path_prefix_to_remove))

    sink_resolvers = []
    for sink in SinkResolver.registry:
        if _is_relative_to(Path(sink.filename), scope_to):
            sink_resolvers.append(convert_type_to_gql(sink, path_prefix=path_prefix_to_remove))

    charts = []
    for chart in Chart._registry:
        charts.append(convert_chart(chart))

    config = load_project_config()
    if config is not None:
        config = project_settings_to_gql(config)

    graph = UpsertGraphGQL(
        streams=stream_resolvers,
        sinks=sink_resolvers,
        features=features,
        config=config,
        failed=failed,
        resolvers=resolvers,
        charts=charts,
        chalkpy=ChalkPYInfo(version=__version__),
        cdcSources=gather_cdc_sources(),
        sqlSources=[
            UpsertSQLSourceGQL(
                name=source.name,
                kind=source.kind,
            )
            for source in BaseSQLSource.registry
        ],
        featureClasses=feature_classes,
    )
    errors = validate_graph(graph)
    graph.errors = errors

    return json.dumps(
        dataclasses.asdict(graph),
        indent=indent,
    )


def get_upsert_graph_gql() -> UpsertGraphGQL:
    resolvers = [convert_type_to_gql(r) for r in Resolver.registry]
    features = []
    feature_classes = []
    for fs in FeatureSetBase.registry.values():
        feature_classes.append(convert_type_to_gql(fs))
        all_features = [f for f in fs.features if not f.is_autogenerated]
        deduped_features = list({unwrap_feature(f).fqn: f for f in all_features}.values())
        features.extend(convert_type_to_gql(f) for f in deduped_features)
    graph_gql = UpsertGraphGQL(
        resolvers=resolvers,
        features=features,
        featureClasses=feature_classes,
        config=None,  # load_project_config()
    )
    errs = validate_graph(graph_gql)
    graph_gql.errors = errs
    return graph_gql
