from __future__ import annotations
import logging
import logging.config
import os
from pathlib import Path
from typing import Any, Dict, Optional, Union
from datetime import datetime

# Process start time captured once per interpreter run
START_TIME = datetime.now().strftime("%Y%m%d-%H%M%S")

# Default logging configuration
_DEFAULT_CONFIG: Dict[str, Any] = {
    "version": 1,
    "disable_existing_loggers": False,
    "formatters": {
        "basic": {
            "format": "[%(levelname)s] %(name)s %(filename)s: %(message)s",
            "datefmt": "%Y-%m-%d %H:%M:%S",
        },
        "verbose": {
            "format": "%(asctime)s [%(levelname)s] %(name)s %(filename)s:%(lineno)d - %(message)s",
            "datefmt": "%Y-%m-%d %H:%M:%S",
        },
        # Optional JSON formatter if the user installs python-json-logger
        "json": {
            "class": "pythonjsonlogger.jsonlogger.JsonFormatter",
            "format": "%(asctime)s %(levelname)s %(name)s %(message)s",
        },
    },
    "handlers": {
        "console": {
            "class": "logging.StreamHandler",
            "level": "INFO",
            "formatter": "basic",
            "stream": "ext://sys.stdout",
        },
        "file": {
            "class": "logging.FileHandler",
            "level": "DEBUG",
            "formatter": "verbose",
            "filename": f"malware_classifier-{START_TIME}.log",
            "encoding": "utf-8",
        },
    },
    "root": {
        "level": "INFO",
        "handlers": ["console", "file"],
    },
}


def _apply_env_overrides(config: Dict[str, Any]) -> None:
    """Apply environment variable overrides to the logging configuration."""
    level = os.getenv("MALCLASS_LOG_LEVEL")
    if level:
        config.setdefault("root", {}).setdefault("level", level.upper())
        if "handlers" in config and "console" in config["handlers"]:
            config["handlers"]["console"]["level"] = level.upper()

    # Highest priority: explicit full path
    log_file = os.getenv("MALCLASS_LOG_FILE")
    if log_file and "handlers" in config and "file" in config["handlers"]:
        config["handlers"]["file"]["filename"] = log_file

    # Optional: log directory joins with the default or current filename
    log_dir = os.getenv("MALCLASS_LOG_DIR")
    if log_dir and "handlers" in config and "file" in config["handlers"]:
        current = Path(config["handlers"]["file"].get("filename", f"malware_classifier-{START_TIME}.log"))
        config["handlers"]["file"]["filename"] = str(Path(log_dir) / current.name)

    fmt = os.getenv("MALCLASS_LOG_FORMATTER")  # basic / verbose / json
    if fmt and fmt in config.get("formatters", {}):
        if "handlers" in config and "console" in config["handlers"]:
            config["handlers"]["console"]["formatter"] = fmt
        if "handlers" in config and "file" in config["handlers"]:
            config["handlers"]["file"]["formatter"] = fmt


def _merge(a: Dict[str, Any], b: Dict[str, Any]) -> Dict[str, Any]:
    """Recursively merge two dictionaries (b overrides a)."""
    result = dict(a)
    for k, v in b.items():
        if isinstance(v, dict) and isinstance(result.get(k), dict):
            result[k] = _merge(result[k], v)
        else:
            result[k] = v
    return result


def _inject_start_time(config: Dict[str, Any]) -> None:
    """Replace {start_time} placeholders and normalize file paths."""
    handlers = config.get("handlers", {})
    file_cfg = handlers.get("file")
    if isinstance(file_cfg, dict):
        filename = file_cfg.get("filename")
        if isinstance(filename, str):
            # Replace placeholder with captured process start time
            filename = filename.replace("{start_time}", START_TIME)
            file_cfg["filename"] = str(Path(filename))


def setup_logging(
    config: Optional[Dict[str, Any]] = None,
    config_path: Optional[Union[str, Path]] = None,
    reset_handlers: bool = True,
    log_dir: Optional[Union[str, Path]] = None,
) -> None:
    """
    Initialize the logging system.

    Priority order:
    1. Default configuration
    2. Config dict passed directly via the `config` argument
    3. Environment variable overrides: MALCLASS_LOG_LEVEL, MALCLASS_LOG_FILE, MALCLASS_LOG_DIR, MALCLASS_LOG_FORMATTER
    4. `log_dir` argument (highest priority except explicit MALCLASS_LOG_FILE)

    Notes:
    - `filename` supports a `{start_time}` placeholder replaced with the process start timestamp.
    - Each run writes to a single log file.
    """
    base = dict(_DEFAULT_CONFIG)

    if config_path:
        raise ValueError("Please load your configuration file externally and pass it via the `config` argument.")

    if config:
        base = _merge(base, config)

    # Replace {start_time} before applying environment overrides (env may fully override filename)
    _inject_start_time(base)
    _apply_env_overrides(base)
    _inject_start_time(base)  # Run again in case env provided a placeholder

    # Highest priority for log directory if provided explicitly
    if log_dir and "handlers" in base and "file" in base["handlers"]:
        current = Path(base["handlers"]["file"]["filename"])
        base["handlers"]["file"]["filename"] = str(Path(log_dir) / current.name)

    if reset_handlers:
        # Remove existing handlers to avoid duplicates
        root_logger = logging.getLogger()
        for h in list(root_logger.handlers):
            root_logger.removeHandler(h)

    logging.config.dictConfig(base)


def get_logger(name: Optional[str] = None) -> logging.Logger:
    """Retrieve a logger instance by name."""
    return logging.getLogger(name if name else "malware_classifier")
