import os
import json
from box import Box

class MalwareClassifier():
    '''
    Model for malware classification.
    This class handles data extraction, vectorization, training, and prediction.
    '''
    
    def __init__(self, config_path="./config.json"):
        '''
        Read the configuration file, create necessary directories, and setup logging.

        Parameters:
        config_path (str): Path to the configuration file (default: "./config.json").
        '''
        # Read config.json
        self.config_path = config_path
        with open(config_path) as f:
            self.config = Box(json.load(f))
        
        # Create necessary directories
        self.mkdir()

    def getFeature(self):
        '''
        Extracts features from the malware dataset.
        '''
        raise NotImplementedError("This method (getFeature) should be overridden by subclasses")

    def getVector(self):
        '''
        Vectorizes the extracted features.
        '''
        raise NotImplementedError("This method (getVector) should be overridden by subclasses")

    def getModel(self, action:str="train"):
        '''
        Trains the model.
        
        Parameters:
        action (str): The action to perform (default: "train").
            - "train": Train the model.
            - "predict": Perform inference with the model.
        '''
        raise NotImplementedError("This method (getModel) should be overridden by subclasses")

    def getPrediction(self):
        '''
        Predicts the given files.
        '''
        raise NotImplementedError("This method (getPrediction) should be overridden by subclasses")
        
    def mkdir(self):
        '''
        Creates necessary directories for the model.
        '''
        for folder_name, folder_path in self.config.folder.items():
            os.makedirs(folder_path, exist_ok=True)
        
if __name__ == "__main__":
    classifier = MalwareClassifier()
    classifier.feature()
    classifier.vector()
    classifier.model(action="train")
    classifier.predict()