import argparse
import re
import sys
import textwrap
from pathlib import Path
from typing import Any, Dict
import getpass
import requests
from easy_acumatica.client import AcumaticaClient

project_root = Path(__file__).resolve().parent
sys.path.insert(0, str(project_root / 'src'))

from easy_acumatica.helpers import _raise_with_detail


def _generate_service_docstring(service_name: str, operation_id: str, details: Dict[str, Any], is_get_files: bool = False) -> str:
    """Generates a detailed docstring from OpenAPI schema details."""
    if is_get_files:
        description = f"Retrieves files attached to a {service_name} entity."
        args_section = [
            "Args:",
            "    entity_id (str): The primary key of the entity.",
            "    api_version (str, optional): The API version to use for this request."
        ]
        returns_section = "Returns:\n    A list of file information dictionaries."
        full_docstring = f"{description}\n\n"
        full_docstring += "\n".join(args_section) + "\n\n"
        full_docstring += returns_section
        return textwrap.indent(full_docstring, '    ')


    summary = details.get("summary", "No summary available.")
    description = f"{summary} for the {service_name} entity."

    args_section = ["Args:"]
    if 'requestBody' in details:
        try:
            ref = details['requestBody']['content']['application/json']['schema']['$ref']
            model_name = ref.split('/')[-1]
            if "InvokeAction" in operation_id:
                args_section.append(f"    invocation (models.{model_name}): The action invocation data.")
            else:
                args_section.append(f"    data (Union[dict, models.{model_name}]): The entity data to create or update.")
        except KeyError:
            args_section.append("    data (dict): The entity data.")

    if 'parameters' in details:
        for param in details['parameters']:
            if param['$ref'].split("/")[-1] == "id":
                args_section.append("    entity_id (str): The primary key of the entity.")

    if "PutFile" in operation_id:
        args_section.append("    entity_id (str): The primary key of the entity.")
        args_section.append("    filename (str): The name of the file to upload.")
        args_section.append("    data (bytes): The file content.")
        args_section.append("    comment (str, optional): A comment about the file.")

    if any(s in operation_id for s in ["GetList", "GetById", "GetByKeys", "PutEntity"]):
        args_section.append("    options (QueryOptions, optional): OData query options.")

    args_section.append("    api_version (str, optional): The API version to use for this request.")

    returns_section = "Returns:\n"
    try:
        response_schema = details['responses']['200']['content']['application/json']['schema']
        if '$ref' in response_schema:
            model_name = response_schema['$ref'].split('/')[-1]
            returns_section += f"    A dictionary or a {model_name} data model instance."
        elif response_schema.get('type') == 'array':
            item_ref = response_schema['items']['$ref']
            model_name = item_ref.split('/')[-1]
            returns_section += f"    A list of dictionaries or {model_name} data model instances."
        else:
            returns_section += "    The JSON response from the API."
    except KeyError:
        returns_section += "    The JSON response from the API or None."


    full_docstring = f"{description}\n\n"
    if len(args_section) > 1:
        full_docstring += "\n".join(args_section) + "\n\n"
    full_docstring += returns_section

    return textwrap.indent(full_docstring, '    ')


def _generate_model_docstring(name: str, definition: Dict[str, Any]) -> str:
    """Generates a docstring for a model in a .pyi stub file."""

    def get_display_type(details: Dict[str, Any]) -> str:
        """Gets a clean, readable type name for the docstring."""
        schema_type = details.get("type")
        schema_format = details.get("format")

        if schema_type == "string":
            return "datetime" if schema_format == "date-time" else "str"
        if schema_type == "integer": return "int"
        if schema_type == "number": return "float"
        if schema_type == "boolean": return "bool"
        if schema_type == "object": return "Any"

        if "$ref" in details:
            ref_name = details["$ref"].split("/")[-1]
            if "Value" in ref_name:
                if "String" in ref_name or "Guid" in ref_name: return "str"
                if "Decimal" in ref_name or "Double" in ref_name: return "float"
                if "Int" in ref_name or "Short" in ref_name or "Long" in ref_name or "Byte" in ref_name: return "int"
                if "Boolean" in ref_name: return "bool"
                if "DateTime" in ref_name: return "datetime"
            return ref_name

        if schema_type == "array":
            item_display = get_display_type(details.get("items", {}))
            return f"List[{item_display}]"

        return "Any"

    description = definition.get("description", f"Represents the {name} entity.")
    docstring_lines = [f"{description}\n"]
    docstring_lines.append("Attributes:")

    required_fields = definition.get("required", [])
    properties = {}
    if 'allOf' in definition:
        for item in definition['allOf']:
            if 'properties' in item:
                properties.update(item['properties'])
    else:
        properties = definition.get("properties", {})

    if not properties:
        docstring_lines.append("    This model has no defined properties.")
    else:
        for prop_name, prop_details in sorted(properties.items()):
            if prop_name in ["note", "rowNumber", "error", "_links"]:
                continue

            type_display = get_display_type(prop_details)
            required_marker = " (required)" if prop_name in required_fields else ""
            docstring_lines.append(f"    {prop_name} ({type_display}){required_marker}")

    indented_docstring = textwrap.indent('"""\n' + "\n".join(docstring_lines) + '\n"""', "    ")
    return indented_docstring


def _map_schema_type_to_python_type(prop_details: Dict[str, Any], is_required: bool) -> str:
    """Maps a schema property to a Python type hint string."""

    def get_base_hint(details: Dict[str, Any]) -> str:
        schema_type = details.get("type")
        schema_format = details.get("format")

        if schema_type == "string":
            return "datetime" if schema_format == "date-time" else "str"
        if schema_type == "integer": return "int"
        if schema_type == "number": return "float"
        if schema_type == "boolean": return "bool"

        if "$ref" in details:
            ref_name = details["$ref"].split("/")[-1]
            if "Value" in ref_name:
                if "String" in ref_name or "Guid" in ref_name: return "str"
                if "Decimal" in ref_name or "Double" in ref_name: return "float"
                if "Int" in ref_name or "Short" in ref_name or "Long" in ref_name or "Byte" in ref_name: return "int"
                if "Boolean" in ref_name: return "bool"
                if "DateTime" in ref_name: return "datetime"
            return f"'{ref_name}'"

        if schema_type == "array":
            item_hint = _map_schema_type_to_python_type(details.get("items", {}), False)
            return f"List[{item_hint}]"

        return "Any"

    base_hint = get_base_hint(prop_details)
    return base_hint if is_required else f"Optional[{base_hint}]"


def generate_model_stubs(schema: Dict[str, Any]) -> str:
    pyi_content = [
        "from __future__ import annotations",
        "from typing import Any, List, Optional, Union",
        "from dataclasses import dataclass",
        "from datetime import datetime",
        "from .core import BaseDataClassModel\n"
    ]

    schemas = schema.get("components", {}).get("schemas", {})
    primitive_wrappers = {
        "StringValue", "DecimalValue", "BooleanValue", "DateTimeValue",
        "GuidValue", "IntValue", "ShortValue", "LongValue", "ByteValue", "DoubleValue"
    }

    for name, definition in sorted(schemas.items()):
        if name in primitive_wrappers:
            continue

        pyi_content.append("\n@dataclass")
        class_lines = [f"class {name}(BaseDataClassModel):"]
        docstring = _generate_model_docstring(name, definition)
        class_lines.append(docstring)

        required_fields = definition.get("required", [])

        properties = {}
        if 'allOf' in definition:
            for item in definition['allOf']:
                if 'properties' in item: properties.update(item['properties'])
        else:
            properties = definition.get("properties", {})

        if not properties:
            class_lines.append("    ...")
        else:
            for prop_name, prop_details in sorted(properties.items()):
                if prop_name in ["note", "rowNumber", "error", "_links"]: continue
                is_required = prop_name in required_fields
                type_hint = _map_schema_type_to_python_type(prop_details, is_required)
                class_lines.append(f"    {prop_name}: {type_hint} = ...")

        pyi_content.extend(class_lines)

    return "\n".join(pyi_content) + "\n"


def generate_client_stubs(schema: Dict[str, Any]) -> str:
    paths = schema.get("paths", {})
    tags_to_ops: Dict[str, Dict] = {}
    for path, path_info in paths.items():
        for http_method, details in path_info.items():
            tag = details.get("tags", [None])[0]
            if not tag or http_method not in ['get', 'put', 'post', 'delete']: continue
            if tag not in tags_to_ops: tags_to_ops[tag] = {}
            op_id = details.get("operationId", "")
            if op_id: tags_to_ops[tag][op_id] = (path, http_method, details)

    pyi_content = [
        "from __future__ import annotations",
        "from typing import Any, Union, List, Dict",
        "from .core import BaseService, BaseDataClassModel",
        "from .odata import QueryOptions",
        "from . import models\n"
    ]

    for tag, operations in sorted(tags_to_ops.items()):
        service_class_name = f"{tag}Service"
        pyi_content.append(f"\nclass {service_class_name}(BaseService):")

        has_put_file_method = any("PutFile" in op_id for op_id in operations)

        if not operations:
            pyi_content.append("    ...")
        else:
            for op_id, (path, http_method, details) in sorted(operations.items()):
                name_part = op_id.split('_', 1)[-1]
                s1 = re.sub('(.)([A-Z][a-z]+)', r'\1_\2', name_part)
                method_name = re.sub('([a-z0-9])([A-Z])', r'\1_\2', s1).lower()
                method_name = method_name.replace('__', '_')

                docstring_content = _generate_service_docstring(tag, op_id, details)
                docstring = textwrap.indent(docstring_content, '        ')

                method_signature = ""
                if "InvokeAction" in op_id:
                    ref = details.get("requestBody", {}).get("content", {}).get("application/json", {}).get("schema", {}).get("$ref", "")
                    model_name = ref.split("/")[-1]
                    method_signature = f"    def {method_name}(self, invocation: models.{model_name}, api_version: str | None = None) -> Any:\n        \"\"\"\n{docstring}\n        \"\"\"\n        ..."
                elif "PutEntity" in op_id:
                    ref = details.get("requestBody", {}).get("content", {}).get("application/json", {}).get("schema", {}).get("$ref", "")
                    model_name = ref.split("/")[-1]
                    method_signature = f"    def {method_name}(self, data: Union[dict, models.{model_name}], options: QueryOptions | None = None, api_version: str | None = None) -> Any:\n        \"\"\"\n{docstring}\n        \"\"\"\n        ..."
                elif "PutFile" in op_id:
                    method_signature = f"    def {method_name}(self, entity_id: str, filename: str, data: bytes, comment: str | None = None, api_version: str | None = None) -> None:\n        \"\"\"\n{docstring}\n        \"\"\"\n        ..."
                elif "GetById" in op_id or "GetByKeys" in op_id:
                    method_signature = f"    def {method_name}(self, entity_id: str, options: QueryOptions | None = None, api_version: str | None = None) -> Any:\n        \"\"\"\n{docstring}\n        \"\"\"\n        ..."
                elif "GetList" in op_id:
                    method_signature = f"    def {method_name}(self, options: QueryOptions | None = None, api_version: str | None = None) -> Any:\n        \"\"\"\n{docstring}\n        \"\"\"\n        ..."
                elif "DeleteById" in op_id or "DeleteByKeys" in op_id:
                    method_signature = f"    def {method_name}(self, entity_id: str, api_version: str | None = None) -> None:\n        \"\"\"\n{docstring}\n        \"\"\"\n        ..."
                elif "GetAdHocSchema" in op_id:
                    method_signature = f"    def {method_name}(self, api_version: str | None = None) -> Any:\n        \"\"\"\n{docstring}\n        \"\"\"\n        ..."

                if method_signature:
                    pyi_content.append(method_signature)

            if has_put_file_method:
                get_files_docstring_content = _generate_service_docstring(tag, "", {}, is_get_files=True)
                get_files_docstring = textwrap.indent(get_files_docstring_content, '        ')
                get_files_signature = (
                    f"    def get_files(self, entity_id: str, api_version: str | None = None) -> List[Dict[str, Any]]:\n"
                    f"        \"\"\"\n{get_files_docstring}\n        \"\"\"\n"
                    f"        ..."
                )
                pyi_content.append(get_files_signature)


    pyi_content.append("\nclass AcumaticaClient:")
    for tag in sorted(tags_to_ops.keys()):
        service_class_name = f"{tag}Service"
        attr_name = ''.join(['_' + i.lower() if i.isupper() else i for i in tag]).lstrip('_') + 's'
        pyi_content.append(f"    {attr_name}: {service_class_name}")
    pyi_content.append("    models: models\n")

    return "\n".join(pyi_content)


def main():
    parser = argparse.ArgumentParser(description="Generate .pyi stub files for easy-acumatica.")
    parser.add_argument("--url", help="Base URL of the Acumatica instance.")
    parser.add_argument("--username", help="Username for authentication.")
    parser.add_argument("--password", help="Password for authentication.")
    parser.add_argument("--tenant", help="The tenant to connect to.")
    parser.add_argument("--endpoint-version", default="24.200.001", help="The API endpoint version to use.")
    parser.add_argument("--endpoint-name", default="Default", help="The API endpoint name.")
    args = parser.parse_args()

    if not args.url:
        args.url = input("Enter Acumatica URL: ")
    if not args.tenant:
        args.tenant = input("Enter Tenant: ")
    if not args.username:
        args.username = input("Enter Username: ")
    if not args.password:
        args.password = getpass.getpass("Enter Password: ")

    client = AcumaticaClient(
        base_url=args.url,
        username=args.username,
        password=args.password,
        tenant=args.tenant,
        endpoint_name = args.endpoint_name,
        endpoint_version = args.endpoint_version,

    )
    schema = client._fetch_schema(args.endpoint_name, args.endpoint_version)
    model_pyi = generate_model_stubs(schema)
    Path("src/easy_acumatica/models.pyi").write_text(model_pyi, encoding='utf-8')
    print("✅ Success! Model stubs written to src/easy_acumatica/models.pyi")

    client_pyi = generate_client_stubs(schema)
    Path("src/easy_acumatica/client.pyi").write_text(client_pyi, encoding='utf-8')
    print("✅ Success! Client stubs written to src/easy_acumatica/client.pyi")

if __name__ == "__main__":
    main()
