import yaml
import os


# generate age curve yamls for all states except NY and VT
def create_age_curve_yamls():
    # Define all state data
    state_data = {
        "default": {
            "base": 0.765,
            "values": {
                0: 0.765,
                15: 0.833,
                16: 0.859,
                17: 0.885,
                18: 0.913,
                19: 0.941,
                20: 0.970,
                21: 1.000,
                22: 1.000,
                23: 1.000,
                24: 1.000,
                25: 1.004,
                26: 1.024,
                27: 1.048,
                28: 1.087,
                29: 1.119,
                30: 1.135,
                31: 1.159,
                32: 1.183,
                33: 1.198,
                34: 1.214,
                35: 1.222,
                36: 1.230,
                37: 1.238,
                38: 1.246,
                39: 1.262,
                40: 1.278,
                41: 1.302,
                42: 1.325,
                43: 1.357,
                44: 1.397,
                45: 1.444,
                46: 1.500,
                47: 1.563,
                48: 1.635,
                49: 1.706,
                50: 1.786,
                51: 1.865,
                52: 1.952,
                53: 2.040,
                54: 2.135,
                55: 2.230,
                56: 2.333,
                57: 2.437,
                58: 2.548,
                59: 2.603,
                60: 2.714,
                61: 2.810,
                62: 2.873,
                63: 2.952,
                64: 3.000,
            },
        },
        "AL": {
            "base": 0.635,
            "values": {
                0: 0.635,
                15: 0.635,
                16: 0.635,
                17: 0.635,
                18: 0.635,
                19: 0.635,
                20: 0.635,
                21: 1.000,
                22: 1.000,
                23: 1.000,
                24: 1.000,
                25: 1.004,
                26: 1.024,
                27: 1.048,
                28: 1.087,
                29: 1.119,
                30: 1.135,
                31: 1.159,
                32: 1.183,
                33: 1.198,
                34: 1.214,
                35: 1.222,
                36: 1.230,
                37: 1.238,
                38: 1.246,
                39: 1.262,
                40: 1.278,
                41: 1.302,
                42: 1.325,
                43: 1.357,
                44: 1.397,
                45: 1.444,
                46: 1.500,
                47: 1.563,
                48: 1.635,
                49: 1.706,
                50: 1.786,
                51: 1.865,
                52: 1.952,
                53: 2.040,
                54: 2.135,
                55: 2.230,
                56: 2.333,
                57: 2.437,
                58: 2.548,
                59: 2.603,
                60: 2.714,
                61: 2.810,
                62: 2.873,
                63: 2.952,
                64: 3.000,
            },
        },
        "DC": {
            "base": 0.654,
            "values": {
                0: 0.654,
                15: 0.654,
                16: 0.654,
                17: 0.654,
                18: 0.654,
                19: 0.654,
                20: 0.654,
                21: 0.727,
                22: 0.727,
                23: 0.727,
                24: 0.727,
                25: 0.727,
                26: 0.727,
                27: 0.727,
                28: 0.744,
                29: 0.760,
                30: 0.779,
                31: 0.799,
                32: 0.817,
                33: 0.836,
                34: 0.856,
                35: 0.876,
                36: 0.896,
                37: 0.916,
                38: 0.927,
                39: 0.938,
                40: 0.975,
                41: 1.013,
                42: 1.053,
                43: 1.094,
                44: 1.137,
                45: 1.181,
                46: 1.227,
                47: 1.275,
                48: 1.325,
                49: 1.377,
                50: 1.431,
                51: 1.487,
                52: 1.545,
                53: 1.605,
                54: 1.668,
                55: 1.733,
                56: 1.801,
                57: 1.871,
                58: 1.944,
                59: 2.020,
                60: 2.099,
                61: 2.181,
                62: 2.181,
                63: 2.181,
                64: 2.181,
            },
        },
        "MA": {
            "base": 0.751,
            "values": {
                0: 0.751,
                15: 0.751,
                16: 0.751,
                17: 0.751,
                18: 0.751,
                19: 0.751,
                20: 0.751,
                21: 1.183,
                22: 1.183,
                23: 1.183,
                24: 1.183,
                25: 1.183,
                26: 1.183,
                27: 1.220,
                28: 1.250,
                29: 1.275,
                30: 1.287,
                31: 1.305,
                32: 1.323,
                33: 1.334,
                34: 1.346,
                35: 1.352,
                36: 1.358,
                37: 1.363,
                38: 1.369,
                39: 1.381,
                40: 1.393,
                41: 1.410,
                42: 1.427,
                43: 1.450,
                44: 1.478,
                45: 1.511,
                46: 1.550,
                47: 1.593,
                48: 1.641,
                49: 1.688,
                50: 1.741,
                51: 1.792,
                52: 1.847,
                53: 1.902,
                54: 1.961,
                55: 2.019,
                56: 2.080,
                57: 2.142,
                58: 2.206,
                59: 2.280,
                60: 2.365,
                61: 2.365,
                62: 2.365,
                63: 2.365,
                64: 2.365,
            },
        },
        "MN": {
            "base": 0.890,
            "values": {
                0: 0.890,
                15: 0.890,
                16: 0.890,
                17: 0.890,
                18: 0.890,
                19: 0.890,
                20: 0.890,
                21: 1.000,
                22: 1.000,
                23: 1.000,
                24: 1.000,
                25: 1.004,
                26: 1.024,
                27: 1.048,
                28: 1.087,
                29: 1.119,
                30: 1.135,
                31: 1.159,
                32: 1.183,
                33: 1.198,
                34: 1.214,
                35: 1.222,
                36: 1.230,
                37: 1.238,
                38: 1.246,
                39: 1.262,
                40: 1.278,
                41: 1.302,
                42: 1.325,
                43: 1.357,
                44: 1.397,
                45: 1.444,
                46: 1.500,
                47: 1.563,
                48: 1.635,
                49: 1.706,
                50: 1.786,
                51: 1.865,
                52: 1.952,
                53: 2.040,
                54: 2.135,
                55: 2.230,
                56: 2.333,
                57: 2.437,
                58: 2.548,
                59: 2.603,
                60: 2.714,
                61: 2.810,
                62: 2.873,
                63: 2.952,
                64: 3.000,
            },
        },
        "MS": {
            "base": 0.635,
            "values": {
                0: 0.635,
                15: 0.635,
                16: 0.635,
                17: 0.635,
                18: 0.635,
                19: 0.635,
                20: 0.635,
                21: 1.000,
                22: 1.000,
                23: 1.000,
                24: 1.000,
                25: 1.004,
                26: 1.024,
                27: 1.048,
                28: 1.087,
                29: 1.119,
                30: 1.135,
                31: 1.159,
                32: 1.183,
                33: 1.198,
                34: 1.214,
                35: 1.222,
                36: 1.230,
                37: 1.238,
                38: 1.246,
                39: 1.262,
                40: 1.278,
                41: 1.302,
                42: 1.325,
                43: 1.357,
                44: 1.397,
                45: 1.444,
                46: 1.500,
                47: 1.563,
                48: 1.635,
                49: 1.706,
                50: 1.786,
                51: 1.865,
                52: 1.952,
                53: 2.040,
                54: 2.135,
                55: 2.230,
                56: 2.333,
                57: 2.437,
                58: 2.548,
                59: 2.603,
                60: 2.714,
                61: 2.810,
                62: 2.873,
                63: 2.952,
                64: 3.000,
            },
        },
        "OR": {
            "base": 0.635,
            "values": {
                0: 0.635,
                15: 0.635,
                16: 0.635,
                17: 0.635,
                18: 0.635,
                19: 0.635,
                20: 0.635,
                21: 1.000,
                22: 1.000,
                23: 1.000,
                24: 1.000,
                25: 1.004,
                26: 1.024,
                27: 1.048,
                28: 1.087,
                29: 1.119,
                30: 1.135,
                31: 1.159,
                32: 1.183,
                33: 1.198,
                34: 1.214,
                35: 1.222,
                36: 1.230,
                37: 1.238,
                38: 1.246,
                39: 1.262,
                40: 1.278,
                41: 1.302,
                42: 1.325,
                43: 1.357,
                44: 1.397,
                45: 1.444,
                46: 1.500,
                47: 1.563,
                48: 1.635,
                49: 1.706,
                50: 1.786,
                51: 1.865,
                52: 1.952,
                53: 2.040,
                54: 2.135,
                55: 2.230,
                56: 2.333,
                57: 2.437,
                58: 2.548,
                59: 2.603,
                60: 2.714,
                61: 2.810,
                62: 2.873,
                63: 2.952,
                64: 3.000,
            },
        },
        "UT": {
            "base": 0.793,
            "values": {
                0: 0.793,
                15: 0.793,
                16: 0.793,
                17: 0.793,
                18: 0.793,
                19: 0.793,
                20: 0.793,
                21: 1.000,
                22: 1.050,
                23: 1.113,
                24: 1.191,
                25: 1.298,
                26: 1.363,
                27: 1.390,
                28: 1.390,
                29: 1.390,
                30: 1.390,
                31: 1.390,
                32: 1.390,
                33: 1.390,
                34: 1.390,
                35: 1.390,
                36: 1.390,
                37: 1.404,
                38: 1.425,
                39: 1.450,
                40: 1.479,
                41: 1.516,
                42: 1.562,
                43: 1.616,
                44: 1.681,
                45: 1.748,
                46: 1.818,
                47: 1.891,
                48: 1.966,
                49: 2.045,
                50: 2.127,
                51: 2.212,
                52: 2.300,
                53: 2.392,
                54: 2.488,
                55: 2.588,
                56: 2.691,
                57: 2.799,
                58: 2.911,
                59: 3.000,
                60: 3.000,
                61: 3.000,
                62: 3.000,
                63: 3.000,
                64: 3.000,
            },
        },
    }

    # Custom YAML dumper class
    class NoAliasDumper(yaml.SafeDumper):
        def ignore_aliases(self, data):
            return True

    # Remove quotes from numbers and format them properly
    def custom_str_presenter(dumper, data):
        if data.startswith("0000-01-01"):
            return dumper.represent_scalar(
                "tag:yaml.org,2002:str", data, style=""
            )
        return dumper.represent_scalar("tag:yaml.org,2002:str", data)

    NoAliasDumper.add_representer(str, custom_str_presenter)

    # Process each state
    for state_name, data in state_data.items():
        base_value = data["base"]
        values = data["values"]

        # Normalize values
        normalized_values = {
            age: value / base_value for age, value in values.items()
        }

        # Create YAML structure
        yaml_data = {
            "description": f"Age curve factors for ACA premium pricing in {state_name.replace('_', ' ').title()}, normalized to age 0.",
            "metadata": {
                "type": "single_amount",
                "period": "year",
                "amount_unit": "/1",
                "label": f"ACA premium age curve factors - {state_name.replace('_', ' ').title()}",
                "reference": [
                    {
                        "title": "CMS Market Rating Reforms State Specific Age Curve Variations",
                        "href": "https://www.cms.gov/cciio/programs-and-initiatives/health-insurance-market-reforms/downloads/statespecagecrv053117.pdf",
                    }
                ],
            },
            "brackets": [],
        }

        # Find threshold points (where values change)
        prev_value = None
        for age, value in sorted(normalized_values.items()):
            if prev_value != value:
                bracket = {
                    "threshold": {"0000-01-01": age},
                    "amount": {"0000-01-01": round(value, 4)},
                }
                yaml_data["brackets"].append(bracket)
                prev_value = value

        # Save to YAML file
        filename = f"{state_name}.yaml"
        with open(filename, "w") as f:
            yaml.dump(
                yaml_data,
                f,
                Dumper=NoAliasDumper,
                sort_keys=False,
                default_flow_style=False,
            )

        print(f"Created {filename}")


if __name__ == "__main__":
    create_age_curve_yamls()
    print("\nAll YAML files have been created successfully!")
