"""Crop foreground transform."""

import functools
from typing import Any, Dict, List, Sequence

import torch
from monai.config import type_definitions
from monai.transforms.croppad import array as monai_croppad_transforms
from monai.utils.enums import PytorchPadMode
from torchvision import tv_tensors
from torchvision.transforms import v2
from typing_extensions import override

from eva.vision.data import tv_tensors as eva_tv_tensors


class CropForeground(v2.Transform):
    """Crop an image using a bounding box.

    The bounding box is generated by selecting foreground using select_fn
    at channels channel_indices. margin is added in each spatial dimension
    of the bounding box. The typical usage is to help training and evaluation
    if the valid part is small in the whole medical image.
    """

    def __init__(
        self,
        threshold: float = 0.0,
        channel_indices: type_definitions.IndexSelection | None = None,
        margin: Sequence[int] | int = 0,
        allow_smaller: bool = True,
        return_coords: bool = False,
        k_divisible: Sequence[int] | int = 1,
        mode: str = PytorchPadMode.CONSTANT,
        **pad_kwargs,
    ) -> None:
        """Initializes the transform.

        Args:
            threshold: function to select expected foreground.
            channel_indices: if defined, select foreground only on the specified channels
                of image. if None, select foreground on the whole image.
            margin: add margin value to spatial dims of the bounding box, if only 1 value provided,
                use it for all dims.
            allow_smaller: when computing box size with `margin`, whether to allow the image edges
                to be smaller than the final box edges. If `False`, part of a padded output box
                might be outside of the original image, if `True`, the image edges will be used as
                the box edges. Default to `True`.
            return_coords: whether return the coordinates of spatial bounding box for foreground.
            k_divisible: make each spatial dimension to be divisible by k, default to 1.
                if `k_divisible` is an int, the same `k` be applied to all the input spatial
                dimensions.
            mode: available modes for numpy array:{``"constant"``, ``"edge"``, ``"linear_ramp"``,
                ``"maximum"``, ``"mean"``, ``"median"``, ``"minimum"``, ``"reflect"``,
                ``"symmetric"``, ``"wrap"``, ``"empty"``} available modes for PyTorch Tensor:
                {``"constant"``, ``"reflect"``, ``"replicate"``, ``"circular"``}. One of the listed
                string values or a user supplied function. Defaults to ``"constant"``.
                See also: https://numpy.org/doc/1.18/reference/generated/numpy.pad.html
                https://pytorch.org/docs/stable/generated/torch.nn.functional.pad.html
            pad_kwargs: other arguments for the `np.pad` or `torch.pad` function.
                note that `np.pad` treats channel dimension as the first dimension.
        """
        super().__init__()

        self._foreground_crop = monai_croppad_transforms.CropForeground(
            select_fn=functools.partial(_threshold_fn, threshold=threshold),
            channel_indices=channel_indices,
            margin=margin,
            allow_smaller=allow_smaller,
            return_coords=return_coords,
            k_divisible=k_divisible,
            mode=mode,
            lazy=False,
            **pad_kwargs,
        )

    def _get_params(self, flat_inputs: List[Any]) -> Dict[str, Any]:
        volume = next(inpt for inpt in flat_inputs if isinstance(inpt, eva_tv_tensors.Volume))
        box_start, box_end = self._foreground_crop.compute_bounding_box(volume)
        return {"box_start": box_start, "box_end": box_end}

    @functools.singledispatchmethod
    @override
    def _transform(self, inpt: Any, params: Dict[str, Any]) -> Any:
        return inpt

    @_transform.register(tv_tensors.Image)
    @_transform.register(eva_tv_tensors.Volume)
    @_transform.register(tv_tensors.Mask)
    def _(self, inpt: Any, params: Dict[str, Any]) -> Any:
        inpt_foreground_cropped = self._foreground_crop.crop_pad(
            inpt, params["box_start"], params["box_end"]
        )
        return tv_tensors.wrap(inpt_foreground_cropped, like=inpt)


def _threshold_fn(image: torch.Tensor, threshold: int | float = 0.3) -> torch.Tensor:
    """Applies a thresholding operation to an image tensor.

    Pixels greater than the threshold are set to True, while others are False.

    Args:
        image: Input image tensor with pixel values.
        threshold: Threshold value.

    Returns:
        A binary mask tensor of the same shape as `image`,
        where True represents pixels above the threshold.
    """
    return image > threshold
