# Jentic SDK [Beta]

Jentic SDK is a comprehensive library for discovery and execution of APIs and workflows.

The Jentic SDK is backed by the data in the [Jentic Public API](https://github.com/jentic/jentic-public-api) repository.

## Core API & Use Cases

The main entry point is the `Jentic` class.

### LLM Tool Definition & Execution

A typical agent loop with tool use looks like this:

```python
from jentic import Jentic

class MyAgent:
    def __init__(self):
        self.jentic = Jentic()
        self.jentic_tools = self.jentic.generate_llm_tool_definitions("anthropic")

    async def process_message(self, user_message):

        # Pass your tool definitions here, generated by the Jentic SDK.
        response = self.client.messages.create(
            model='claude-3-5-sonnet-latest',  
            messages=messages,
            tools=self.jentic_tools, 
        )
        
        while response.stop_reason == "tool_use":
            tool_use = next(block for block in response.content if block.type == "tool_use")
            tool_name = tool_use.name
            tool_input = tool_use.input

            # Execute the tool
            tool_result = await self.jentic.run_llm_tool(
                tool_name,
                tool_input 
            )
```

## Components

### agent_runtime

A library for generating, managing, and executing LLM-compatible tools from Arazzo workflows and OpenAPI operations.

Features:
- Generate dynamic tool definitions for OpenAI and Anthropic LLMs from project workflows and API operations
- Execute workflows and operations as tools via a unified interface
- Provide runtime classes for tool management (`AgentToolManager`), tool specification (`LLMToolSpecManager`), and execution (`TaskExecutor`)
- Return standardized results for workflow and operation execution

### api

A client for the Jentic API Knowledge Hub.

Set `JENTIC_API_URL` to set the base URL for the API hub client for local testing or testing against the development environment. 

## Installation

### For Development

To install the package in development mode:

```bash
# From the current directory
pip install -e .
```

Then you can import it in your projects:

```python
import jentic
```

You can also import it to other projects by referencing the package directory by specifying the following in your `pyproject.toml`:

```toml
dependencies = [
    "jentic @ file:///path/to/jentic/sdk"
]
```

### For Production

```bash
pip install jentic
```

### Get Your Jentic UUID

To use the Jentic SDK, you must first obtain a Jentic UUID. The easiest way is using the Jentic CLI. You can _optionally_ include an email address for higher rate limits and for early access to new features.

```sh
pip install jentic
jentic register --email '<your_email>'
```

This will print your UUID and an export command to set it in your environment:

```sh
export JENTIC_UUID=<your-jentic-uuid>
```

Alternatively, you can use curl to register and obtain your UUID:

```sh
curl -X POST https://api.jentic.com/api/v1/auth/register \
     -H "Content-Type: application/json" \
     -d '{"email": "<your_email>"}'
```

### Testing

```bash
# Run unit tests for jentic
pdm run test

# Run integration tests for dev environment
pdm run integration-dev

# Run integration tests for production environment
pdm run integration
```

#### Integration Test Configuration

Integration tests require environment variables to be set in environment-specific files. An example file is provided that you need to copy and configure:

```bash
# For development testing
cp tests/integration/.env.example tests/integration/.env.dev

# For production testing
cp tests/integration/.env.example tests/integration/.env.prod
```

After copying, edit these files to include your API credentials and configuration:

1. For development testing: `tests/integration/.env.dev`
2. For production testing: `tests/integration/.env.prod`

Example configuration:

```
# Required UUIDs for Discord API testing
DISCORD_GET_MY_USER_OPERATION_UUID="your_operation_uuid_here"
DISCORD_GET_USER_DETAILS_WORKFLOW_UUID="your_workflow_uuid_here"

# Base URL for Jentic API (Uncomment for Dev)
JENTIC_API_URL=https://directory-api.qa1.eu-west-1.jenticdev.net

# Your Discord bot token
DISCORD_BOTTOKEN="your_bot_token_here"
```

The integration tests validate:
1. Loading operation and workflow execution information
2. Executing Discord operations and workflows
3. Searching API capabilities
4. Generating LLM tool definitions
5. Running LLM tools including both operations and workflows

### Linting & Formatting

```bash
# Run all linting for jentic
pdm run lint
```

## License

Proprietary - Jentic