# -*- coding: utf-8 -*-
from distutils.core import setup

packages = \
['subdivisions', 'subdivisions.builders']

package_data = \
{'': ['*']}

install_requires = \
['arrow', 'asbool', 'boto3', 'loguru', 'python-dotenv', 'stringcase']

setup_kwargs = {
    'name': 'sub-divisions',
    'version': '1.0.2',
    'description': 'AWS EventBridge based PubSub Library',
    'long_description': '# Welcome to Subdivisions\n\n[![PyPI](https://img.shields.io/pypi/v/sub.divisions)](https://pypi.org/project/sub.divisions/)\n[![Publish](https://github.com/chrismaille/subdivisions/workflows/publish/badge.svg)](https://github.com/chrismaille/subdivisions/actions)\n[![PyPI - Python Version](https://img.shields.io/pypi/pyversions/subdivisions)](https://www.python.org)\n[![Conventional Commits](https://img.shields.io/badge/Conventional%20Commits-1.0.0-yellow.svg)](https://conventionalcommits.org)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n[![pre-commit](https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white)](https://github.com/pre-commit/pre-commit)\n\n[AWS Eventbridge backed PubSub solution](https://www.youtube.com/watch?v=EYYdQB0mkEU)\n\n### Install in Project\n\n1. Install subdivisions\n\n```shell\n$ pip install sub-divisions\n```\n\n### Configure Project\n\n1. On every project which you will send or receive messages, create a\n    `pyproject.toml` with the following configuration:\n\n```toml\n[tool.subdivisions]\nsource_name = "ProjectName"     # For Eventbridge Schema Discovery\naws_account = ""                # AWS Account Id\naws_user = ""                   # AWS User with sufficient permissions\n```\n\nThe `source_name` are used to inform the `Source` field in Eventbridge\nmessages. This source will be used on Eventbridge discovery schemas.\n\nThe `aws_account` is the AWS account which will be configured for\nEventbridge, KMS, SNS and SQS artifacts.\n\nThe `aws_user` is the user which we will use for create the KMS PubSub\nKey. To avoid conflicts use the *AWS Secret Key* and *AWS Secret Id*\nfrom the same user. This account must have the minimum permissions:\n\n* Allow all Eventbridge actions\n* Allow all SQS actions\n* Allow all SNS actions\n* Allow all KMS actions\n\n### Usage\n#### Send Messages\n```python\nfrom enum import unique, Enum\nfrom subdivisions.client import SubClient\n\n@unique\nclass MyProjectEvents(Enum):\n    MY_EXAMPLE_EVENT = "my_example_event"\n\nclient = SubClient()\nclient.topic = MyProjectEvents.MY_EXAMPLE_EVENT\nclient.send({"foo": "bar"})\n```\n\n#### Receive Messages\n```python\nfrom enum import unique, Enum\nfrom subdivisions.client import SubClient\n\n@unique\nclass MyProjectEvents(Enum):\n    MY_EXAMPLE_EVENT = "my_example_event"\n\nclient = SubClient()\nclient.topic = MyProjectEvents.MY_EXAMPLE_EVENT\nmessages = client.get_messages()    # use the ``from_dead_letter=True` to receive Dead Letter messages\n# Process messages\nclient.delete_received_messages()\n```\n### AWS Credentials\n\nSubdivisions will use AWS default environment variables. If you need to define another credentials, use the following variables:\n\n```env\nSUBDIVISIONS_USE_AWS_ENV_VARS="false"\nSUBDIVISIONS_AWS_ACCESS_KEY_ID="your id"\nSUBDIVISIONS_AWS_SECRET_ACCESS_KEY="your key"\nSUBDIVISIONS_AWS_SESSION_TOKEN="your token" # optional\n```\n\n### Configuration\n\nConfigure subdivisions options in `pyproject.toml` file, inside `[tool.subdivisions]` table:\n\n```toml\n# pyproject.toml\n[tool.subdivisions]\naws_region = "us-east-1"            # AWS Region\naws_account = ""                    # AWS Account for configuration/use\naws_user = ""                       # AWS User with sufficient permissions\naws_sqs_policy = ""                 # AWS SQS Policy (default in policies.py)\naws_sns_policy = ""                 # AWS SNS Policy (default in policies.py)\naws_kms_policy = ""                 # AWS KMS Policy (default in policies.py)\npub_key = "PubSubKey"               # KMS PubSubKey (must be created first)\nsqs_tags = []                       # SQS tags for new queues. Example [{"foo": "bar"}]\nqueue_prefix = ""                   # Prefix for new SQS queues\nqueue_suffix = ""                   # Suffix for new SQS queues\nqueue_max_receive_count = 1000      # SQS MaxReceiveCount setting\nsns_prefix = ""                     # Prefix for new SNS topics\nsns_suffix = ""                     # Suffix for new SNS topics\nsns_tags = []                       # SNS tags for new topics. Example [{"foo": "bar"}]\nevent_prefix = ""                   # Prefix for new Eventbride rules\nevent_suffix = ""                   # Suffix for new Eventbride rules\nevent_tags = []                     # Eventbridge tags for new rules. Example [{"foo": "bar"}]\nevent_bus = "default"               # Eventbridge Bus\nsource_name = ""                    # Eventbridge default source name. No default, must inform\nauto_create_new_topic = true        # Auto create new topic if not exists in Eventbridge\nauto_remove_from_queue = false      # Acknowledge first messages on receive\nuse_aws_env_vars = true             # Use AWS default env vars. If false append "SUBDIVISION_" on env vars. Example: "SUBDIVISION_AWS_ACCESS_KEY_ID"\ndefault_prefix = "pubsub"           # Default prefix for all sns, sqs and rule created\ndefault_suffix = ""                 # Default suffix for all sns, sqs and rule created\n```\n\nAll options above can be configured in environment variables. Just append `SUBDIVISIONS_` on name. Example: `SUBDIVISIONS_SOURCE_NAME="my_project"`\n',
    'author': 'Chris Maillefaud',
    'author_email': 'chrismaillefaud@gmail.com',
    'url': 'https://github.com/chrismaille/subdivisions',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
