"""Test suite for GenieService.

Tests for the GenieService class covering all service layer operations with >95% coverage.
Follows TDD Red-Green-Refactor approach with failing tests first.

Test Categories:
- Unit tests: Individual service method testing
- Integration tests: Docker Compose integration and container management
- Mock tests: Container operations and filesystem interactions  
- Security tests: Workspace validation and path security
- Container lifecycle tests: Real docker-compose integration patterns

GenieService Methods Tested:
1. serve_genie() - Start Genie all-in-one container on port 48886
2. stop_genie() - Stop Genie container cleanly
3. restart_genie() - Restart Genie container  
4. show_genie_logs() - Display Genie container logs
5. get_genie_status() - Get comprehensive Genie container status
6. _validate_genie_environment() - Validate Genie workspace environment
7. _setup_genie_postgres() - Setup internal PostgreSQL for Genie
8. _generate_genie_credentials() - Generate secure Genie credentials
"""

import tempfile
from pathlib import Path
from unittest.mock import Mock, patch

import pytest

# Note: GenieService doesn't exist yet - these tests will fail initially (RED phase)
try:
    from cli.core.genie_service import GenieService
except ImportError:
    # Expected during RED phase - create mock class for testing
    class GenieService:
        def __init__(self):
            pass


class TestGenieService:
    """Test suite for GenieService class with comprehensive coverage."""

    @pytest.fixture
    def mock_compose_manager(self):
        """Mock DockerComposeManager for testing container operations."""
        with patch("cli.core.genie_service.DockerComposeManager") as mock_compose_class:
            mock_compose = Mock()
            mock_compose_class.return_value = mock_compose
            yield mock_compose

    @pytest.fixture
    def temp_workspace(self):
        """Create temporary workspace directory for testing."""
        with tempfile.TemporaryDirectory() as temp_dir:
            workspace = Path(temp_dir)
            # Create required files for valid Genie workspace
            (workspace / "docker-compose-genie.yml").write_text("""
version: '3.8'
services:
  genie-server:
    image: automagik-hive:genie
    container_name: hive-genie-server
    ports:
      - "48886:48886"
    environment:
      - HIVE_DATABASE_URL=postgresql+psycopg://genie:genie@localhost:5432/hive_genie
      - HIVE_API_PORT=48886
    volumes:
      - ./data/postgres-genie:/var/lib/postgresql/data
    networks:
      - genie_network
networks:
  genie_network:
    driver: bridge
""")
            (workspace / ".env.genie").write_text("""
POSTGRES_USER=test_genie_user
POSTGRES_PASSWORD=test_genie_pass
POSTGRES_DB=hive_genie
HIVE_API_PORT=48886
HIVE_API_KEY=genie_test_key_123
""")
            # Create data and logs directories
            (workspace / "data" / "postgres-genie").mkdir(parents=True, exist_ok=True)
            (workspace / "logs").mkdir(exist_ok=True)
            yield str(workspace)

    def test_genie_service_initialization(self, mock_compose_manager):
        """Test GenieService initializes with correct configuration."""
        service = GenieService()

        # Should fail initially - GenieService class not implemented yet
        assert hasattr(service, "compose_manager")
        assert service.genie_port == 48886
        assert service.genie_compose_file == "docker-compose-genie.yml"

    def test_serve_genie_success(self, mock_compose_manager, temp_workspace):
        """Test successful Genie server start."""
        # Mock container not running initially
        mock_status = Mock()
        mock_status.name = "STOPPED"
        mock_compose_manager.get_service_status.return_value = mock_status
        mock_compose_manager.start_service.return_value = True

        service = GenieService()
        result = service.serve_genie(temp_workspace)

        # Should fail initially - serve_genie method not implemented
        assert result is True
        mock_compose_manager.start_service.assert_called_once_with(
            "genie-server", temp_workspace
        )

    def test_serve_genie_already_running(self, mock_compose_manager, temp_workspace):
        """Test serve when Genie is already running."""
        # Mock container already running
        mock_status = Mock()
        mock_status.name = "RUNNING"
        mock_compose_manager.get_service_status.return_value = mock_status

        service = GenieService()
        result = service.serve_genie(temp_workspace)

        # Should fail initially - already running check not implemented
        assert result is True
        # Should not call start_service if already running
        mock_compose_manager.start_service.assert_not_called()

    def test_serve_genie_container_start_failure(self, mock_compose_manager, temp_workspace):
        """Test serve when container fails to start."""
        mock_status = Mock()
        mock_status.name = "STOPPED"
        mock_compose_manager.get_service_status.return_value = mock_status
        mock_compose_manager.start_service.return_value = False

        service = GenieService()
        result = service.serve_genie(temp_workspace)

        # Should fail initially - error handling not implemented
        assert result is False
        assert mock_compose_manager.start_service.called

    def test_serve_genie_invalid_workspace(self, mock_compose_manager):
        """Test serve with invalid workspace path."""
        service = GenieService()

        # Should fail initially - workspace validation not implemented
        with pytest.raises(Exception):  # Could be SecurityError or other validation error
            service.serve_genie("/invalid/workspace/path")

    def test_stop_genie_success(self, mock_compose_manager, temp_workspace):
        """Test successful Genie server stop."""
        mock_compose_manager.stop_service.return_value = True

        service = GenieService()
        result = service.stop_genie(temp_workspace)

        # Should fail initially - stop_genie method not implemented
        assert result is True
        mock_compose_manager.stop_service.assert_called_once_with(
            "genie-server", temp_workspace
        )

    def test_stop_genie_failure(self, mock_compose_manager, temp_workspace):
        """Test Genie server stop failure."""
        mock_compose_manager.stop_service.return_value = False

        service = GenieService()
        result = service.stop_genie(temp_workspace)

        # Should fail initially - error handling not implemented
        assert result is False
        assert mock_compose_manager.stop_service.called

    def test_stop_genie_not_running(self, mock_compose_manager, temp_workspace):
        """Test stop when Genie is not running."""
        mock_status = Mock()
        mock_status.name = "STOPPED"
        mock_compose_manager.get_service_status.return_value = mock_status
        mock_compose_manager.stop_service.return_value = True

        service = GenieService()
        result = service.stop_genie(temp_workspace)

        # Should fail initially - not running check not implemented
        assert result is True
        # Should still call stop_service for cleanup
        assert mock_compose_manager.stop_service.called

    def test_restart_genie_success(self, mock_compose_manager, temp_workspace):
        """Test successful Genie server restart."""
        mock_compose_manager.restart_service.return_value = True

        service = GenieService()
        result = service.restart_genie(temp_workspace)

        # Should fail initially - restart_genie method not implemented
        assert result is True
        mock_compose_manager.restart_service.assert_called_once_with(
            "genie-server", temp_workspace
        )

    def test_restart_genie_failure(self, mock_compose_manager, temp_workspace):
        """Test Genie server restart failure."""
        mock_compose_manager.restart_service.return_value = False

        service = GenieService()
        result = service.restart_genie(temp_workspace)

        # Should fail initially - error handling not implemented
        assert result is False
        assert mock_compose_manager.restart_service.called

    def test_show_genie_logs_success(self, mock_compose_manager, temp_workspace):
        """Test successful Genie logs display."""
        mock_compose_manager.get_service_logs.return_value = "Genie server log line 1\nGenie server log line 2"

        service = GenieService()
        result = service.show_genie_logs(temp_workspace, tail=50)

        # Should fail initially - show_genie_logs method not implemented
        assert result is True
        mock_compose_manager.get_service_logs.assert_called_once_with(
            "genie-server", 50, temp_workspace
        )

    def test_show_genie_logs_failure(self, mock_compose_manager, temp_workspace):
        """Test Genie logs display failure."""
        mock_compose_manager.get_service_logs.return_value = None

        service = GenieService()
        result = service.show_genie_logs(temp_workspace, tail=20)

        # Should fail initially - error handling not implemented
        assert result is False
        assert mock_compose_manager.get_service_logs.called

    def test_show_genie_logs_custom_tail(self, mock_compose_manager, temp_workspace):
        """Test Genie logs with custom tail parameter."""
        mock_compose_manager.get_service_logs.return_value = "Recent logs"

        service = GenieService()
        result = service.show_genie_logs(temp_workspace, tail=100)

        # Should fail initially - custom tail handling not implemented
        assert result is True
        mock_compose_manager.get_service_logs.assert_called_once_with(
            "genie-server", 100, temp_workspace
        )

    def test_get_genie_status_success(self, mock_compose_manager, temp_workspace):
        """Test successful Genie status retrieval."""
        # Mock comprehensive status response
        mock_services_status = {
            "genie-server": Mock(status=Mock(name="RUNNING"),
                               container_id="abc123",
                               ports=["0.0.0.0:48886->48886/tcp"])
        }
        mock_compose_manager.get_all_services_status.return_value = mock_services_status

        service = GenieService()
        result = service.get_genie_status(temp_workspace)

        # Should fail initially - get_genie_status method not implemented
        assert isinstance(result, dict)
        assert "genie-server" in result
        assert mock_compose_manager.get_all_services_status.called

    def test_get_genie_status_container_not_found(self, mock_compose_manager, temp_workspace):
        """Test status when Genie container doesn't exist."""
        mock_compose_manager.get_all_services_status.return_value = {}

        service = GenieService()
        result = service.get_genie_status(temp_workspace)

        # Should fail initially - empty status handling not implemented
        assert isinstance(result, dict)
        assert len(result) >= 0  # Should handle empty status gracefully

    def test_get_genie_status_with_health_check(self, mock_compose_manager, temp_workspace):
        """Test status includes health check information."""
        mock_services_status = {
            "genie-server": Mock(status=Mock(name="RUNNING"),
                               health_status="healthy")
        }
        mock_compose_manager.get_all_services_status.return_value = mock_services_status

        service = GenieService()
        result = service.get_genie_status(temp_workspace)

        # Should fail initially - health check integration not implemented
        assert isinstance(result, dict)
        assert mock_compose_manager.get_all_services_status.called


class TestGenieServiceValidation:
    """Test GenieService workspace validation and security."""

    @pytest.fixture
    def mock_security_utils(self):
        """Mock security utilities for testing."""
        with patch("cli.core.genie_service.secure_resolve_workspace") as mock_resolve:
            with patch("cli.core.genie_service.secure_subprocess_call") as mock_subprocess:
                yield mock_resolve, mock_subprocess

    def test_validate_genie_environment_valid_workspace(self, temp_workspace):
        """Test validation with valid Genie workspace."""
        service = GenieService()

        # Should fail initially - _validate_genie_environment method not implemented
        result = service._validate_genie_environment(Path(temp_workspace))
        assert result is True

    def test_validate_genie_environment_missing_compose_file(self):
        """Test validation with missing docker-compose-genie.yml."""
        with tempfile.TemporaryDirectory() as temp_dir:
            workspace = Path(temp_dir)

            service = GenieService()
            # Should fail initially - compose file validation not implemented
            result = service._validate_genie_environment(workspace)
            assert result is False

    def test_validate_genie_environment_missing_env_file(self):
        """Test validation with missing .env.genie file."""
        with tempfile.TemporaryDirectory() as temp_dir:
            workspace = Path(temp_dir)
            # Create compose file but not env file
            (workspace / "docker-compose-genie.yml").write_text("version: '3.8'")

            service = GenieService()
            # Should fail initially - env file validation not implemented
            result = service._validate_genie_environment(workspace)
            assert result is False

    def test_validate_genie_environment_invalid_workspace_path(self):
        """Test validation with invalid workspace path."""
        service = GenieService()

        # Should fail initially - path validation not implemented
        with pytest.raises(Exception):  # Could be SecurityError or FileNotFoundError
            service._validate_genie_environment(Path("/nonexistent/path"))

    def test_security_error_handling(self, mock_security_utils):
        """Test security error handling in service methods."""
        mock_resolve, mock_subprocess = mock_security_utils
        mock_resolve.side_effect = Exception("Security validation failed")

        service = GenieService()

        # Should fail initially - security error handling not implemented
        with pytest.raises(Exception):
            service.serve_genie("../../invalid/path")


class TestGenieServiceContainerOperations:
    """Test Genie service container-specific operations."""

    @pytest.fixture
    def mock_docker_operations(self):
        """Mock Docker operations for container testing."""
        with patch("cli.core.genie_service.DockerComposeManager") as mock_compose_class:
            mock_compose = Mock()
            mock_compose_class.return_value = mock_compose
            yield mock_compose

    def test_genie_all_in_one_container_management(self, mock_docker_operations, temp_workspace):
        """Test management of all-in-one container (PostgreSQL + FastAPI)."""
        # Mock all-in-one container status
        mock_status = Mock()
        mock_status.name = "RUNNING"
        mock_docker_operations.get_service_status.return_value = mock_status

        service = GenieService()
        result = service.serve_genie(temp_workspace)

        # Should fail initially - all-in-one container logic not implemented
        assert result is True
        assert mock_docker_operations.get_service_status.called

    def test_genie_container_port_48886_validation(self, mock_docker_operations, temp_workspace):
        """Test Genie container uses correct port 48886."""
        service = GenieService()

        # Should fail initially - port validation not implemented
        assert service.genie_port == 48886

        # Test port is used in container operations
        service.serve_genie(temp_workspace)
        assert mock_docker_operations.start_service.called

    def test_genie_container_network_isolation(self, mock_docker_operations, temp_workspace):
        """Test Genie container network isolation."""
        mock_services_status = {
            "genie-server": Mock(
                networks=["hive_genie_network"],
                status=Mock(name="RUNNING")
            )
        }
        mock_docker_operations.get_all_services_status.return_value = mock_services_status

        service = GenieService()
        result = service.get_genie_status(temp_workspace)

        # Should fail initially - network isolation check not implemented
        assert isinstance(result, dict)
        assert mock_docker_operations.get_all_services_status.called

    def test_genie_container_volume_persistence(self, mock_docker_operations, temp_workspace):
        """Test Genie container volume persistence."""
        service = GenieService()

        # Check volume directories exist
        data_dir = Path(temp_workspace) / "data" / "postgres-genie"
        logs_dir = Path(temp_workspace) / "logs"

        assert data_dir.exists()
        assert logs_dir.exists()

        # Should fail initially - volume validation not implemented
        result = service.serve_genie(temp_workspace)
        assert mock_docker_operations.start_service.called

    def test_genie_container_supervisor_integration(self, mock_docker_operations, temp_workspace):
        """Test Genie container supervisor process management."""
        # Mock supervisor status in all-in-one container
        mock_services_status = {
            "genie-server": Mock(
                processes=["postgresql", "fastapi", "supervisor"],
                status=Mock(name="RUNNING")
            )
        }
        mock_docker_operations.get_all_services_status.return_value = mock_services_status

        service = GenieService()
        result = service.get_genie_status(temp_workspace)

        # Should fail initially - supervisor integration not implemented
        assert isinstance(result, dict)
        assert mock_docker_operations.get_all_services_status.called

    def test_genie_container_health_check(self, mock_docker_operations, temp_workspace):
        """Test Genie container health check integration."""
        # Mock health check response
        mock_docker_operations.get_service_logs.return_value = (
            "PostgreSQL ready\n"
            "FastAPI server started on port 48886\n"
            "Health check: HEALTHY"
        )

        service = GenieService()
        result = service.show_genie_logs(temp_workspace)

        # Should fail initially - health check integration not implemented
        assert result is True
        assert mock_docker_operations.get_service_logs.called


class TestGenieServiceCredentialManagement:
    """Test Genie service credential and environment management."""

    def test_generate_genie_credentials(self, temp_workspace):
        """Test Genie credential generation."""
        service = GenieService()

        # Should fail initially - _generate_genie_credentials method not implemented
        result = service._generate_genie_credentials(temp_workspace)
        assert result is True

        # Check credentials were generated
        env_file = Path(temp_workspace) / ".env.genie"
        if env_file.exists():
            env_content = env_file.read_text()
            assert "POSTGRES_USER=" in env_content
            assert "POSTGRES_PASSWORD=" in env_content
            assert "HIVE_API_KEY=" in env_content

    def test_setup_genie_postgres_credentials(self, temp_workspace):
        """Test PostgreSQL credential setup for Genie."""
        service = GenieService()

        # Should fail initially - _setup_genie_postgres method not implemented
        result = service._setup_genie_postgres(temp_workspace)
        assert result is True

    def test_genie_api_key_generation(self, temp_workspace):
        """Test Genie API key generation."""
        service = GenieService()

        # Should fail initially - API key generation not implemented
        api_key = service._generate_genie_api_key(temp_workspace)
        assert api_key is not None
        assert isinstance(api_key, str)
        assert len(api_key) > 20  # Should be reasonably long

    def test_secure_credential_storage(self, temp_workspace):
        """Test secure storage of Genie credentials."""
        service = GenieService()

        # Should fail initially - secure storage not implemented
        service._generate_genie_credentials(temp_workspace)

        env_file = Path(temp_workspace) / ".env.genie"
        if env_file.exists():
            # Check file permissions are secure
            file_mode = oct(env_file.stat().st_mode)[-3:]
            assert file_mode == "600" or file_mode == "644"  # Secure permissions

    def test_credential_validation(self, temp_workspace):
        """Test Genie credential validation."""
        service = GenieService()

        # Should fail initially - credential validation not implemented
        result = service._validate_genie_credentials(temp_workspace)
        assert result in [True, False]  # Should return boolean


class TestGenieServiceErrorHandling:
    """Test GenieService error handling and edge cases."""

    def test_workspace_not_exists_error(self):
        """Test handling when workspace doesn't exist."""
        service = GenieService()

        # Should fail initially - workspace existence check not implemented
        with pytest.raises(Exception):  # Could be FileNotFoundError or custom error
            service.serve_genie("/nonexistent/workspace")

    def test_docker_compose_file_corrupted(self):
        """Test handling when docker-compose-genie.yml is corrupted."""
        with tempfile.TemporaryDirectory() as temp_dir:
            workspace = Path(temp_dir)
            # Create corrupted compose file
            (workspace / "docker-compose-genie.yml").write_text("invalid: yaml: content:")

            service = GenieService()
            # Should fail initially - compose file validation not implemented
            with pytest.raises(Exception):
                service.serve_genie(str(workspace))

    def test_container_start_timeout(self):
        """Test handling when container takes too long to start."""
        with patch("cli.core.genie_service.DockerComposeManager") as mock_compose_class:
            mock_compose = Mock()
            mock_compose.start_service.side_effect = Exception("Container start timeout")
            mock_compose_class.return_value = mock_compose

            service = GenieService()
            # Should fail initially - timeout handling not implemented
            with pytest.raises(Exception):
                service.serve_genie("/tmp/test_workspace")

    def test_insufficient_permissions_error(self):
        """Test handling when insufficient permissions for Docker operations."""
        with patch("cli.core.genie_service.DockerComposeManager") as mock_compose_class:
            mock_compose = Mock()
            mock_compose.start_service.side_effect = PermissionError("Docker permission denied")
            mock_compose_class.return_value = mock_compose

            service = GenieService()
            # Should fail initially - permission error handling not implemented
            with pytest.raises(PermissionError):
                service.serve_genie("/tmp/test_workspace")

    def test_port_already_in_use_error(self):
        """Test handling when port 48886 is already in use."""
        with patch("cli.core.genie_service.DockerComposeManager") as mock_compose_class:
            mock_compose = Mock()
            mock_compose.start_service.side_effect = Exception("Port 48886 already in use")
            mock_compose_class.return_value = mock_compose

            service = GenieService()
            # Should fail initially - port conflict handling not implemented
            with pytest.raises(Exception):
                service.serve_genie("/tmp/test_workspace")

    def test_disk_space_insufficient_error(self):
        """Test handling when insufficient disk space for container volumes."""
        service = GenieService()

        # Should fail initially - disk space check not implemented
        # This would typically be caught during container startup
        result = service.serve_genie("/tmp/test_workspace")
        assert result in [True, False]  # Should handle gracefully


class TestGenieServiceCrossPlatform:
    """Test GenieService cross-platform compatibility."""

    @pytest.fixture
    def mock_platform_detection(self):
        """Mock platform detection for cross-platform testing."""
        with patch("platform.system") as mock_system:
            yield mock_system

    def test_genie_service_on_windows(self, mock_platform_detection, temp_workspace):
        """Test GenieService on Windows platform."""
        mock_platform_detection.return_value = "Windows"

        service = GenieService()
        # Should fail initially - Windows path handling not implemented
        result = service.serve_genie(temp_workspace)
        assert result in [True, False]

    def test_genie_service_on_linux(self, mock_platform_detection, temp_workspace):
        """Test GenieService on Linux platform."""
        mock_platform_detection.return_value = "Linux"

        service = GenieService()
        # Should fail initially - Linux-specific optimizations not implemented
        result = service.serve_genie(temp_workspace)
        assert result in [True, False]

    def test_genie_service_on_macos(self, mock_platform_detection, temp_workspace):
        """Test GenieService on macOS platform."""
        mock_platform_detection.return_value = "Darwin"

        service = GenieService()
        # Should fail initially - macOS Docker Desktop integration not implemented
        result = service.serve_genie(temp_workspace)
        assert result in [True, False]

    def test_path_handling_cross_platform(self, temp_workspace):
        """Test path handling works across platforms."""
        service = GenieService()

        # Test various path formats
        test_paths = [
            temp_workspace,
            str(Path(temp_workspace)),
            Path(temp_workspace).resolve(),
        ]

        for test_path in test_paths:
            # Should fail initially - cross-platform path normalization not implemented
            try:
                result = service._validate_genie_environment(Path(str(test_path)))
                assert result in [True, False]
            except Exception:
                pass  # Expected during RED phase
