"""Integration tests for Genie CLI commands end-to-end workflow.

Tests the complete Genie container lifecycle from CLI commands through service layer
to actual Docker container operations. Follows TDD Red-Green-Refactor approach.

Integration Test Categories:
- End-to-end workflow: Complete Genie container lifecycle testing
- CLI to container integration: Commands → Service → Docker integration
- Container networking: Port 48886 validation and health checks
- Real Docker Compose integration: Actual docker-compose-genie.yml testing
- Cross-component integration: CLI + Service + Container orchestration
- Performance testing: Container startup/shutdown timing validation

Genie Integration Scenarios:
1. Complete serve workflow: CLI command → Service → Container start → Health check
2. Complete stop workflow: CLI command → Service → Container stop → Cleanup
3. Complete restart workflow: Stop + Start with state persistence
4. Logs integration: Container logs → Service → CLI display
5. Status integration: Container state → Service → CLI formatted output
6. Error propagation: Container errors → Service → CLI error handling
"""

import tempfile
import time
from pathlib import Path
from unittest.mock import Mock, patch

import pytest

from cli.commands.genie import (
    GenieCommands,
    genie_serve_cmd,
    genie_status_cmd,
    genie_stop_cmd,
)


class TestGenieEndToEndWorkflow:
    """Test complete end-to-end Genie workflow integration."""

    @pytest.fixture
    def integration_workspace(self):
        """Create realistic workspace for integration testing."""
        with tempfile.TemporaryDirectory() as temp_dir:
            workspace = Path(temp_dir)

            # Create complete Genie workspace structure
            (workspace / "docker-compose-genie.yml").write_text("""
version: '3.8' 

services:
  genie-server:
    build:
      context: .
      dockerfile: Dockerfile.genie
      target: genie-production
    container_name: hive-genie-server
    restart: unless-stopped
    ports:
      - "48886:48886"
    environment:
      - HIVE_DATABASE_URL=postgresql+psycopg://genie:genie@localhost:5432/hive_genie
      - RUNTIME_ENV=prd  
      - HIVE_LOG_LEVEL=info
      - HIVE_API_HOST=0.0.0.0
      - HIVE_API_PORT=48886
      - POSTGRES_USER=genie
      - POSTGRES_PASSWORD=genie
      - POSTGRES_DB=hive_genie
    volumes:
      - ./data/postgres-genie:/var/lib/postgresql/data
      - genie_app_logs:/app/logs
      - genie_supervisor_logs:/var/log/supervisor
    networks:
      - genie_network
    healthcheck:
      test: |
        pg_isready -U genie -d hive_genie &&
        curl -f http://localhost:48886/api/v1/health
      interval: 30s
      timeout: 15s
      retries: 3
      start_period: 90s

networks:
  genie_network:
    driver: bridge
    name: hive_genie_network

volumes:
  genie_app_logs:
    driver: local
    name: hive_genie_app_logs
  genie_supervisor_logs:
    driver: local
    name: hive_genie_supervisor_logs
""")

            # Create environment file
            (workspace / ".env.genie").write_text("""
POSTGRES_USER=integration_genie_user
POSTGRES_PASSWORD=integration_genie_pass_123
POSTGRES_DB=hive_genie
HIVE_API_PORT=48886
HIVE_API_KEY=genie_integration_test_key_xyz789
RUNTIME_ENV=integration
HIVE_LOG_LEVEL=debug
""")

            # Create required directories
            (workspace / "data" / "postgres-genie").mkdir(parents=True, exist_ok=True)
            (workspace / "logs").mkdir(exist_ok=True)

            # Create Dockerfile.genie stub for testing
            (workspace / "Dockerfile.genie").write_text("""
FROM agnohq/pgvector:16 as genie-production
# Genie all-in-one container
EXPOSE 48886
CMD ["supervisord", "-c", "/etc/supervisor/conf.d/supervisord.conf"]
""")

            yield str(workspace)

    @pytest.fixture
    def mock_docker_compose_integration(self):
        """Mock Docker Compose for realistic integration testing."""
        with patch("cli.core.genie_service.DockerComposeManager") as mock_compose_class:
            mock_compose = Mock()

            # Mock realistic container lifecycle
            container_state = {"running": False, "container_id": "genie-abc123"}

            def mock_start_service(service, workspace):
                container_state["running"] = True
                time.sleep(0.1)  # Simulate startup time
                return True

            def mock_stop_service(service, workspace):
                container_state["running"] = False
                time.sleep(0.05)  # Simulate shutdown time
                return True

            def mock_get_status(service, workspace):
                status = Mock()
                status.name = "RUNNING" if container_state["running"] else "STOPPED"
                return status

            def mock_get_all_services_status(workspace):
                return {
                    "genie-server": Mock(
                        status=Mock(name="RUNNING" if container_state["running"] else "STOPPED"),
                        container_id=container_state["container_id"],
                        ports=["0.0.0.0:48886->48886/tcp"] if container_state["running"] else [],
                        health_status="healthy" if container_state["running"] else "unhealthy"
                    )
                }

            def mock_get_logs(service, tail, workspace):
                if container_state["running"]:
                    return "Genie server startup complete\nHealth check: OK\nListening on port 48886"
                return "Container not running"

            mock_compose.start_service = mock_start_service
            mock_compose.stop_service = mock_stop_service
            mock_compose.restart_service = lambda s, w: mock_stop_service(s, w) and mock_start_service(s, w)
            mock_compose.get_service_status = mock_get_status
            mock_compose.get_all_services_status = mock_get_all_services_status
            mock_compose.get_service_logs = mock_get_logs

            mock_compose_class.return_value = mock_compose
            yield mock_compose, container_state

    def test_complete_genie_serve_workflow(self, integration_workspace, mock_docker_compose_integration):
        """Test complete Genie serve workflow from CLI to container."""
        mock_compose, container_state = mock_docker_compose_integration

        # Test CLI command calls service layer
        commands = GenieCommands()
        result = commands.serve(integration_workspace)

        # Should fail initially - integration not implemented
        assert result is True
        assert container_state["running"] is True
        assert mock_compose.start_service.called

        # Verify container is accessible on port 48886
        # (In real test, would check actual port accessibility)
        status = commands.status(integration_workspace)
        assert status is True

    def test_complete_genie_stop_workflow(self, integration_workspace, mock_docker_compose_integration):
        """Test complete Genie stop workflow from CLI to container."""
        mock_compose, container_state = mock_docker_compose_integration

        # Start container first
        commands = GenieCommands()
        commands.serve(integration_workspace)
        assert container_state["running"] is True

        # Test stop workflow
        result = commands.stop(integration_workspace)

        # Should fail initially - stop integration not implemented
        assert result is True
        assert container_state["running"] is False
        assert mock_compose.stop_service.called

    def test_complete_genie_restart_workflow(self, integration_workspace, mock_docker_compose_integration):
        """Test complete Genie restart workflow with state persistence."""
        mock_compose, container_state = mock_docker_compose_integration

        # Start container first
        commands = GenieCommands()
        commands.serve(integration_workspace)
        original_container_id = container_state["container_id"]

        # Test restart workflow
        result = commands.restart(integration_workspace)

        # Should fail initially - restart integration not implemented
        assert result is True
        assert container_state["running"] is True
        assert mock_compose.restart_service.called

        # Container should maintain state (same ID in this mock)
        assert container_state["container_id"] == original_container_id

    def test_genie_logs_integration_workflow(self, integration_workspace, mock_docker_compose_integration):
        """Test Genie logs integration from container to CLI display."""
        mock_compose, container_state = mock_docker_compose_integration

        # Start container to generate logs
        commands = GenieCommands()
        commands.serve(integration_workspace)

        # Test logs integration
        result = commands.logs(integration_workspace, tail=50)

        # Should fail initially - logs integration not implemented
        assert result is True
        assert mock_compose.get_service_logs.called

        # Test logs when container is stopped
        commands.stop(integration_workspace)
        result = commands.logs(integration_workspace, tail=10)
        assert result is True  # Should handle stopped container gracefully

    def test_genie_status_integration_workflow(self, integration_workspace, mock_docker_compose_integration):
        """Test Genie status integration with formatted CLI output."""
        mock_compose, container_state = mock_docker_compose_integration

        commands = GenieCommands()

        # Test status when stopped
        result = commands.status(integration_workspace)
        assert result is True

        # Start container and test status
        commands.serve(integration_workspace)
        result = commands.status(integration_workspace)

        # Should fail initially - status integration not implemented
        assert result is True
        assert mock_compose.get_all_services_status.called

        # Status should show port 48886 information
        status_info = mock_compose.get_all_services_status(integration_workspace)
        assert "48886" in str(status_info)


class TestGenieContainerNetworkingIntegration:
    """Test Genie container networking and port integration."""

    @pytest.fixture
    def mock_network_validation(self):
        """Mock network validation for integration testing."""
        with patch("socket.socket") as mock_socket:
            with patch("urllib.request.urlopen") as mock_urlopen:
                yield mock_socket, mock_urlopen

    def test_port_48886_accessibility_integration(self, integration_workspace, mock_network_validation):
        """Test port 48886 is accessible after Genie container start."""
        mock_socket, mock_urlopen = mock_network_validation

        # Mock successful port connection
        mock_socket_instance = Mock()
        mock_socket_instance.connect.return_value = None
        mock_socket.return_value.__enter__.return_value = mock_socket_instance

        with patch("cli.core.genie_service.DockerComposeManager"):
            commands = GenieCommands()
            result = commands.serve(integration_workspace)

            # Should fail initially - port accessibility check not implemented
            assert result is True

            # In real integration, would test actual port connection
            # commands._validate_port_accessibility(48886)

    def test_genie_health_endpoint_integration(self, integration_workspace, mock_network_validation):
        """Test Genie health endpoint integration via HTTP."""
        mock_socket, mock_urlopen = mock_network_validation

        # Mock successful health check response
        mock_response = Mock()
        mock_response.read.return_value = b'{"status": "healthy", "services": {"postgres": "ok", "api": "ok"}}'
        mock_urlopen.return_value.__enter__.return_value = mock_response

        with patch("cli.core.genie_service.DockerComposeManager"):
            commands = GenieCommands()
            commands.serve(integration_workspace)

            # Should fail initially - health endpoint integration not implemented
            # In real integration test:
            # health_status = commands._check_health_endpoint()
            # assert health_status["status"] == "healthy"

    def test_genie_network_isolation_integration(self, integration_workspace):
        """Test Genie container network isolation integration."""
        with patch("cli.core.genie_service.DockerComposeManager") as mock_compose_class:
            mock_compose = Mock()
            mock_compose.get_all_services_status.return_value = {
                "genie-server": Mock(
                    networks=["hive_genie_network"],
                    network_mode="bridge",
                    isolated=True
                )
            }
            mock_compose_class.return_value = mock_compose

            commands = GenieCommands()
            result = commands.status(integration_workspace)

            # Should fail initially - network isolation validation not implemented
            assert result is True
            assert mock_compose.get_all_services_status.called


class TestGenieRealDockerComposeIntegration:
    """Test integration with actual docker-compose-genie.yml file."""

    def test_docker_compose_file_validation_integration(self, integration_workspace):
        """Test docker-compose-genie.yml file validation integration."""
        compose_file = Path(integration_workspace) / "docker-compose-genie.yml"
        assert compose_file.exists()

        with patch("cli.core.genie_service.DockerComposeManager"):
            commands = GenieCommands()

            # Should fail initially - compose file validation not implemented
            result = commands.serve(integration_workspace)
            assert result is True

    def test_docker_compose_environment_integration(self, integration_workspace):
        """Test docker-compose environment variable integration."""
        env_file = Path(integration_workspace) / ".env.genie"
        assert env_file.exists()

        env_content = env_file.read_text()
        assert "HIVE_API_PORT=48886" in env_content
        assert "POSTGRES_USER=" in env_content

        with patch("cli.core.genie_service.DockerComposeManager"):
            commands = GenieCommands()
            result = commands.serve(integration_workspace)

            # Should fail initially - environment integration not implemented
            assert result is True

    def test_docker_compose_volume_integration(self, integration_workspace):
        """Test docker-compose volume mount integration."""
        data_dir = Path(integration_workspace) / "data" / "postgres-genie"
        logs_dir = Path(integration_workspace) / "logs"

        assert data_dir.exists()
        assert logs_dir.exists()

        with patch("cli.core.genie_service.DockerComposeManager"):
            commands = GenieCommands()
            result = commands.serve(integration_workspace)

            # Should fail initially - volume mount validation not implemented
            assert result is True

    def test_docker_compose_health_check_integration(self, integration_workspace):
        """Test docker-compose health check configuration integration."""
        compose_file = Path(integration_workspace) / "docker-compose-genie.yml"
        compose_content = compose_file.read_text()

        # Verify health check is configured
        assert "healthcheck:" in compose_content
        assert "pg_isready" in compose_content
        assert "curl -f http://localhost:48886/api/v1/health" in compose_content

        with patch("cli.core.genie_service.DockerComposeManager"):
            commands = GenieCommands()
            result = commands.serve(integration_workspace)

            # Should fail initially - health check integration not implemented
            assert result is True


class TestGenieCLIIntegrationFunctions:
    """Test CLI integration functions end-to-end."""

    def test_genie_serve_cmd_integration(self, integration_workspace):
        """Test genie_serve_cmd CLI function integration."""
        with patch("cli.commands.genie.GenieCommands") as mock_commands_class:
            mock_commands = Mock()
            mock_commands.serve.return_value = True
            mock_commands_class.return_value = mock_commands

            # Test CLI function integration
            result = genie_serve_cmd(integration_workspace)

            # Should fail initially - CLI integration not implemented
            assert result == 0
            mock_commands.serve.assert_called_once_with(integration_workspace)

    def test_genie_stop_cmd_integration(self, integration_workspace):
        """Test genie_stop_cmd CLI function integration."""
        with patch("cli.commands.genie.GenieCommands") as mock_commands_class:
            mock_commands = Mock()
            mock_commands.stop.return_value = True
            mock_commands_class.return_value = mock_commands

            result = genie_stop_cmd(integration_workspace)

            # Should fail initially - CLI integration not implemented
            assert result == 0
            mock_commands.stop.assert_called_once_with(integration_workspace)

    def test_genie_status_cmd_integration(self, integration_workspace):
        """Test genie_status_cmd CLI function integration."""
        with patch("cli.commands.genie.GenieCommands") as mock_commands_class:
            mock_commands = Mock()
            mock_commands.status.return_value = True
            mock_commands_class.return_value = mock_commands

            result = genie_status_cmd(integration_workspace)

            # Should fail initially - CLI integration not implemented
            assert result == 0
            mock_commands.status.assert_called_once_with(integration_workspace)

    def test_cli_error_propagation_integration(self, integration_workspace):
        """Test error propagation from service to CLI integration."""
        with patch("cli.commands.genie.GenieCommands") as mock_commands_class:
            mock_commands = Mock()
            mock_commands.serve.side_effect = Exception("Container startup failed")
            mock_commands_class.return_value = mock_commands

            # Should fail initially - error propagation not implemented
            with pytest.raises(Exception):
                genie_serve_cmd(integration_workspace)


class TestGeniePerformanceIntegration:
    """Test Genie container performance and timing integration."""

    def test_container_startup_timing_integration(self, integration_workspace):
        """Test Genie container startup timing integration."""
        with patch("cli.core.genie_service.DockerComposeManager") as mock_compose_class:
            mock_compose = Mock()

            # Simulate realistic startup timing
            def slow_start_service(service, workspace):
                time.sleep(0.2)  # Simulate container startup time
                return True

            mock_compose.start_service = slow_start_service
            mock_compose_class.return_value = mock_compose

            commands = GenieCommands()
            start_time = time.time()
            result = commands.serve(integration_workspace)
            end_time = time.time()

            # Should fail initially - timing integration not implemented
            assert result is True
            assert (end_time - start_time) >= 0.2  # Should include startup delay

    def test_container_shutdown_timing_integration(self, integration_workspace):
        """Test Genie container shutdown timing integration."""
        with patch("cli.core.genie_service.DockerComposeManager") as mock_compose_class:
            mock_compose = Mock()

            def slow_stop_service(service, workspace):
                time.sleep(0.1)  # Simulate graceful shutdown time
                return True

            mock_compose.stop_service = slow_stop_service
            mock_compose_class.return_value = mock_compose

            commands = GenieCommands()
            start_time = time.time()
            result = commands.stop(integration_workspace)
            end_time = time.time()

            # Should fail initially - shutdown timing not implemented
            assert result is True
            assert (end_time - start_time) >= 0.1  # Should include shutdown delay

    def test_container_resource_usage_integration(self, integration_workspace):
        """Test Genie container resource usage integration."""
        with patch("cli.core.genie_service.DockerComposeManager") as mock_compose_class:
            mock_compose = Mock()
            mock_compose.get_all_services_status.return_value = {
                "genie-server": Mock(
                    memory_usage="512MB / 2GB",
                    cpu_usage="25%",
                    status=Mock(name="RUNNING")
                )
            }
            mock_compose_class.return_value = mock_compose

            commands = GenieCommands()
            result = commands.status(integration_workspace)

            # Should fail initially - resource monitoring not implemented
            assert result is True
            assert mock_compose.get_all_services_status.called


class TestGenieIntegrationErrorHandling:
    """Test integration error handling and recovery scenarios."""

    def test_container_crash_recovery_integration(self, integration_workspace):
        """Test Genie container crash recovery integration."""
        with patch("cli.core.genie_service.DockerComposeManager") as mock_compose_class:
            mock_compose = Mock()

            # Simulate container crash during operation
            call_count = 0
            def crash_then_recover(service, workspace):
                nonlocal call_count
                call_count += 1
                if call_count == 1:
                    raise Exception("Container crashed unexpectedly")
                return True

            mock_compose.start_service = crash_then_recover
            mock_compose_class.return_value = mock_compose

            commands = GenieCommands()

            # First attempt should fail
            with pytest.raises(Exception):
                commands.serve(integration_workspace)

            # Second attempt should succeed (recovery)
            # Should fail initially - crash recovery not implemented
            result = commands.serve(integration_workspace)
            assert result is True

    def test_network_connectivity_failure_integration(self, integration_workspace):
        """Test network connectivity failure integration."""
        with patch("cli.core.genie_service.DockerComposeManager") as mock_compose_class:
            mock_compose = Mock()
            mock_compose.start_service.side_effect = Exception("Network unreachable")
            mock_compose_class.return_value = mock_compose

            commands = GenieCommands()

            # Should fail initially - network error handling not implemented
            with pytest.raises(Exception):
                commands.serve(integration_workspace)

    def test_disk_space_exhaustion_integration(self, integration_workspace):
        """Test disk space exhaustion integration."""
        with patch("cli.core.genie_service.DockerComposeManager") as mock_compose_class:
            mock_compose = Mock()
            mock_compose.start_service.side_effect = Exception("No space left on device")
            mock_compose_class.return_value = mock_compose

            commands = GenieCommands()

            # Should fail initially - disk space error handling not implemented
            with pytest.raises(Exception):
                commands.serve(integration_workspace)

    def test_concurrent_access_integration(self, integration_workspace):
        """Test concurrent Genie access integration."""
        with patch("cli.core.genie_service.DockerComposeManager") as mock_compose_class:
            mock_compose = Mock()
            mock_compose.start_service.return_value = True
            mock_compose.get_service_status.return_value = Mock(name="RUNNING")
            mock_compose_class.return_value = mock_compose

            commands1 = GenieCommands()
            commands2 = GenieCommands()

            # Start from first instance
            result1 = commands1.serve(integration_workspace)
            assert result1 is True

            # Try to start from second instance (should detect already running)
            result2 = commands2.serve(integration_workspace)

            # Should fail initially - concurrent access handling not implemented
            assert result2 is True  # Should handle gracefully
