from math import ceil


def generate_div2_list(input_size, n_min, max_repeats=2) -> list:
    """
    Generates a list of integers starting from `n` and repeatedly dividing by 2 until `n_min` is reached.
    Each integer is repeated a number of times, starting from 1 and increasing by 1 with each division,
    up to a maximum of 4 repeats.

    Args:
        input_size (int): The starting integer.
        n_min (int): The minimum integer value to stop the division.
        max_repeats (int): The maximum number of times an integer can be repeated. Default is 2.

    Returns:
        list: A list of integers with the described pattern.

    Examples:
        >>> from spotpython.hyperparameters.architecture import generate_div2_list
            for n in range(5, 21):
                print(generate_div2_list(input_size=n, n_min=5))
            [5]
            [6]
            [7]
            [8]
            [9]
            [10, 5, 5]
            [11, 5, 5]
            [12, 6, 6]
            [13, 6, 6]
            [14, 7, 7]
            [15, 7, 7]
            [16, 8, 8]
            [17, 8, 8]
            [18, 9, 9]
            [19, 9, 9]
            [20, 10, 10, 5, 5]
    """
    result = []
    current = input_size
    repeats = 1
    while current >= n_min:
        result.extend([current] * min(repeats, max_repeats))
        current = current // 2
        repeats = repeats + 1
    # if result is an empty list, add n_min to it
    if not result:
        result.append(n_min)
    return result


def get_hidden_sizes(_L_in, l1, max_n=10) -> list:
    """
    Generates a list of hidden sizes for a neural network with a given input size.
    Starting with size l1, the list is generated by dividing the input size by 2 until the minimum size is reached.

    Args:
        _L_in (int):
            input size.
        l1 (int):
            number of neurons in the first hidden layer.
        max_n (int):
            maximum number of hidden sizes to generate. Default is 10.

    Returns:
        (list):
            list of hidden sizes.

    Examples:
        >>> from spotpython.hyperparameters.architecture import get_hidden_sizes
            _L_in = 10
            l1 = 20
            n = 4
            get_hidden_sizes(_L_in, l1, max_n=n)
            [20, 10, 10, 5]
    """
    if l1 < 4:
        l1 = 4
    n_low = _L_in // 4
    n_high = max(l1, 2 * n_low)
    hidden_sizes = generate_div2_list(n_high, n_low)
    # keep only the first n values of hidden_sizes list
    if len(hidden_sizes) > max_n:
        hidden_sizes = hidden_sizes[:max_n]
    return hidden_sizes


def get_three_layers(_L_in, l1) -> list:
    """
    Calculate three layers based on input values.

    Args:
        _L_in (float): The input value to be multiplied.
        l1 (float): The multiplier for the layers.

    Returns:
        list: A list containing three calculated layers [a, b, c] where:
            - a = 3 * l1 * _L_in
            - b = 2 * l1 * _L_in
            - c = l1 * _L_in

    Examples:
        >>> from spotpython.hyperparameters.architecture import get_three_layers
            _L_in = 10
            l1 = 20
            get_three_layers(_L_in, l1)
            [600, 400, 200]
    """
    a = 2 * l1 * _L_in
    b = l1 * _L_in
    c = ceil(l1 / 2) * _L_in
    return [a, b, a, b, b, c, c]
