from __future__ import annotations

import copy
import dataclasses
from typing import TYPE_CHECKING
from unittest.mock import ANY, MagicMock, call

import pytest

from randovania.game.game_enum import RandovaniaGame
from randovania.game_description.assignment import PickupTarget
from randovania.game_description.db.hint_node import HintNode
from randovania.game_description.db.node_identifier import NodeIdentifier
from randovania.game_description.hint import (
    HintDarkTemple,
    HintItemPrecision,
    JokeHint,
    LocationHint,
    PrecisionPair,
    RedTempleHint,
    SpecificHintPrecision,
)
from randovania.game_description.pickup.pickup_entry import StartingPickupBehavior
from randovania.game_description.resources.pickup_index import PickupIndex
from randovania.games.prime2.generator.hint_distributor import EchoesHintDistributor
from randovania.generator.pre_fill_params import PreFillParams
from randovania.layout.base.standard_pickup_state import StandardPickupState

if TYPE_CHECKING:
    from randovania.game_description.hint_features import HintFeature
    from randovania.games.prime2.layout.echoes_configuration import EchoesConfiguration


@pytest.mark.parametrize("is_multiworld", [False, True])
async def test_add_default_hints_to_patches(echoes_game_description, echoes_game_patches, is_multiworld):
    # Setup
    layout_configuration = MagicMock()
    layout_configuration.game = RandovaniaGame.METROID_PRIME_ECHOES
    rng = MagicMock()
    hint_distributor = EchoesHintDistributor()

    def precision(loc: str) -> HintFeature:
        return echoes_game_description.hint_feature_database[loc]

    def _light_suit_location_hint(number: int):
        return LocationHint(
            PrecisionPair(precision("specific_hint_2mos"), HintItemPrecision.DETAILED, include_owner=False),
            PickupIndex(number),
        )

    def _guardian_hint(number: int):
        return LocationHint(
            PrecisionPair(precision("specific_hint_guardian"), HintItemPrecision.DETAILED, include_owner=False),
            PickupIndex(number),
        )

    def _keybearer_hint(number: int):
        return LocationHint(
            PrecisionPair(precision("specific_hint_keybearer"), SpecificHintPrecision(0.65, 0.1), include_owner=True),
            PickupIndex(number),
        )

    base_expected = {
        # Keybearer
        "Temple Grounds/Landing Site/Keybearer Corpse (M-Dhe)": _keybearer_hint(11),
        "Temple Grounds/Industrial Site/Keybearer Corpse (J-Fme)": _keybearer_hint(15),
        "Temple Grounds/Storage Cavern A/Keybearer Corpse (D-Isl)": _keybearer_hint(19),
        # Agon
        "Agon Wastes/Central Mining Station/Keybearer Corpse (J-Stl)": _keybearer_hint(45),
        "Agon Wastes/Main Reactor/Keybearer Corpse (B-Stl)": _keybearer_hint(53),
        # Torvus
        "Torvus Bog/Torvus Lagoon/Keybearer Corpse (S-Dly)": _keybearer_hint(68),
        "Torvus Bog/Catacombs/Keybearer Corpse (G-Sch)": _keybearer_hint(91),
        # Sanctuary
        "Sanctuary Fortress/Sanctuary Entrance/Keybearer Corpse (S-Jrs)": _keybearer_hint(117),
        "Sanctuary Fortress/Dynamo Works/Keybearer Corpse (C-Rch)": _keybearer_hint(106),
        # Locations with hints
        "Sanctuary Fortress/Sanctuary Energy Controller/Lore Scan": _light_suit_location_hint(24),
        "Sanctuary Fortress/Main Gyro Chamber/Lore Scan": _guardian_hint(43),
        "Sanctuary Fortress/Watch Station/Lore Scan": _guardian_hint(79),
        "Sanctuary Fortress/Main Research/Lore Scan": _guardian_hint(115),
        # Dark Temple hints
        "Sanctuary Fortress/Hall of Combat Mastery/Lore Scan": RedTempleHint(dark_temple=HintDarkTemple.AGON_WASTES),
        "Sanctuary Fortress/Sanctuary Entrance/Lore Scan": RedTempleHint(dark_temple=HintDarkTemple.TORVUS_BOG),
        "Torvus Bog/Catacombs/Lore Scan": RedTempleHint(dark_temple=HintDarkTemple.SANCTUARY_FORTRESS),
        # Jokes
        "Torvus Bog/Gathering Hall/Lore Scan": JokeHint(),
        "Torvus Bog/Training Chamber/Lore Scan": JokeHint(),
    }

    expected = {NodeIdentifier.from_string(ident_s): hint for ident_s, hint in base_expected.items()}

    # Run
    result = await hint_distributor.assign_pre_filler_hints(
        echoes_game_patches,
        prefill=PreFillParams(
            rng=rng,
            configuration=layout_configuration,
            game=echoes_game_description,
            is_multiworld=is_multiworld,
        ),
    )

    # Assert
    rng.shuffle.assert_has_calls([call(ANY), call(ANY)])
    assert result.hints == expected


@pytest.fixture
def echoes_configuration_everything_shuffled(default_echoes_configuration) -> EchoesConfiguration:
    old_pickups_config = default_echoes_configuration.standard_pickup_configuration
    pickups_state = copy.copy(old_pickups_config.pickups_state)

    for pickup in old_pickups_config.pickups_state:
        if pickup.starting_condition == StartingPickupBehavior.MUST_BE_STARTING:
            continue
        pickups_state[pickup] = StandardPickupState(
            num_shuffled_pickups=1, included_ammo=tuple(1 for ammo in pickup.ammo)
        )

    return dataclasses.replace(
        default_echoes_configuration,
        standard_pickup_configuration=dataclasses.replace(
            old_pickups_config,
            pickups_state=pickups_state,
        ),
        ammo_pickup_configuration=dataclasses.replace(
            default_echoes_configuration.ammo_pickup_configuration,
            pickups_state={
                ammo: dataclasses.replace(ammo_state, pickup_count=1)
                for ammo, ammo_state in default_echoes_configuration.ammo_pickup_configuration.pickups_state.items()
            },
        ),
    )


@pytest.mark.parametrize(
    ("target_pickup", "target_player", "features", "in_spoiler", "expected"),
    [
        ("Amber Translator", 0, set(), True, True),
        ("Energy Tank", 0, {"energy_tank"}, False, False),
        ("Unlimited Missiles", 0, {"cheat"}, True, False),
        ("Beam Ammo Expansion", 0, set(), False, False),
    ],
)
def test_echoes_more_interesting_pickups(
    target_pickup: str,
    target_player: int,
    features: set[str],
    in_spoiler: bool,
    expected: bool,
    echoes_game_description,
):
    # Setup
    hint_node = echoes_game_description.region_list.typed_node_by_identifier(
        NodeIdentifier.create("Great Temple", "Main Energy Controller", "Lore Scan"),
        HintNode,
    )
    hint_distributor = EchoesHintDistributor()

    target = MagicMock(spec=PickupTarget)
    target.pickup.name = target_pickup
    target.pickup.show_in_credits_spoiler = in_spoiler
    target.pickup.has_hint_feature = lambda feat: feat in features
    target.player = target_player

    # Run
    result = hint_distributor.is_pickup_more_interesting(target, 0, hint_node)

    # Assert
    assert result == expected


@pytest.mark.parametrize(
    ("target_pickup", "target_player", "features", "expected"),
    [
        ("Violet Translator", 0, set(), False),
        ("Amber Translator", 0, set(), True),
        ("Violet Translator", 1, set(), True),
        ("Sky Temple Key 1", 0, {"key"}, False),
        ("Energy Tank", 0, {"energy_tank"}, True),
    ],
)
def test_echoes_interesting_pickups(
    target_pickup: str, target_player: int, features: set[str], expected: bool, echoes_game_description
):
    # Setup
    hint_node = echoes_game_description.region_list.typed_node_by_identifier(
        NodeIdentifier.create("Great Temple", "Main Energy Controller", "Lore Scan"),
        HintNode,
    )
    hint_distributor = EchoesHintDistributor()

    target = MagicMock(spec=PickupTarget)
    target.pickup.name = target_pickup
    target.pickup.has_hint_feature = lambda feat: feat in features
    target.player = target_player

    # Run
    result = hint_distributor.is_pickup_interesting(target, 0, hint_node)

    # Assert
    assert result == expected
