# Claude Instructions for Memorizz Project

This file contains specific instructions for Claude Code to help with common development tasks for the Memorizz project.

## Project Overview

Memorizz is a Python library for AI agent memory management with MongoDB integration and semantic caching capabilities. The project uses semantic versioning and is published to PyPI.

## Development Commands

### Linting and Type Checking
When making code changes, always run these commands before committing:
```bash
# Add the appropriate linting commands here once identified
# Example: flake8, black, mypy, etc.
```

### Testing
```bash
# Add test commands here once test framework is identified
# Example: pytest, python -m unittest, etc.
```

## PyPI Deployment Process

Use this process when deploying a new version to PyPI:

### 1. Version Update
Update the version number in `pyproject.toml`:
```toml
[project]
name = "memorizz"
version = "X.X.X"  # Update this version number
```

### 2. Clean and Build
```bash
# Clean previous builds
rm -rf dist/

# Install build dependencies
pip install build twine

# Build source distribution and wheel
python -m build
```

### 3. Git Operations
```bash
# Commit version changes
git add pyproject.toml
git commit -m "Bump version to X.X.X for PyPI release"

# Create and push annotated tag
git tag -a vX.X.X -m "Release version X.X.X"
git push origin vX.X.X
git push origin main
```

### 4. PyPI Upload
```bash
# Upload to production PyPI (requires API token)
twine upload dist/*
```

**Note**: You'll need to provide your PyPI API token when prompted, or set up `~/.pypirc` with your credentials:
```ini
[pypi]
username = __token__
password = your-api-token-here
```

## Project Structure Notes

- **Main package**: Located in `src/memorizz/`
- **Examples**: Located in `examples/` directory with Jupyter notebooks
- **Memory types**: Defined in `src/memorizz/enums/memory_type.py`
- **MongoDB provider**: Located in `src/memorizz/memory_provider/mongodb/`
- **Semantic cache**: Located in `src/memorizz/short_term_memory/semantic_cache.py`

## Key Features to Remember

- **Semantic Cache**: Vector-based query-response caching with configurable similarity thresholds
- **Memory Types**: CONVERSATION_MEMORY, WORKFLOW_MEMORY, LONG_TERM_MEMORY, SHORT_TERM_MEMORY, PERSONAS, TOOLBOX, SHARED_MEMORY, MEMAGENT, SUMMARIES, SEMANTIC_CACHE
- **Scoping**: Supports LOCAL (agent-specific) and GLOBAL (cross-agent) cache scopes
- **MongoDB Integration**: Uses MongoDB Atlas with vector search capabilities

## Semantic Cache Configuration Example

```python
from memorizz.short_term_memory.semantic_cache import SemanticCacheConfig
from memorizz.enums import SemanticCacheScope

config = SemanticCacheConfig(
    similarity_threshold=0.85,  # 0.0-1.0 scale
    max_cache_size=1000,       # Maximum entries
    ttl_hours=24.0,            # Time-to-live
    scope=SemanticCacheScope.LOCAL,  # LOCAL or GLOBAL
    enable_memory_provider_sync=True
)
```

## Important Reminders

- Always test changes before deploying
- Semantic versioning: MAJOR.MINOR.PATCH
- Check that all dependencies are properly listed in `pyproject.toml`
- Ensure MongoDB configurations are properly handled
- Semantic cache requires embedding providers (OpenAI, VoyageAI, etc.)

## Common Issues

- **Embedding Provider**: Make sure embedding providers are configured correctly
- **MongoDB Atlas**: Vector search indexes must be created for semantic functionality
- **Dependencies**: Ensure all required packages are installed and compatible

## Repository Information

- **GitHub**: https://github.com/RichmondAlake/memorizz
- **PyPI**: https://pypi.org/project/memorizz/
- **Main Branch**: `main`
- **License**: MIT