"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GCEPersistentDiskPersistentVolume = exports.AzureDiskPersistentVolume = exports.AwsElasticBlockStorePersistentVolume = exports.PersistentVolumeReclaimPolicy = exports.PersistentVolume = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
const base = require("./base");
const k8s = require("./imports/k8s");
const pvc = require("./pvc");
const volume = require("./volume");
class ImportedPersistentVolume extends constructs_1.Construct {
    constructor(scope, id, name) {
        super(scope, id);
        this.resourceType = 'persistentvolumes';
        this._name = name;
    }
    get name() {
        return this._name;
    }
    get apiVersion() {
        return k8s.KubePersistentVolume.GVK.apiVersion;
    }
    get apiGroup() {
        return '';
    }
    get kind() {
        return k8s.KubePersistentVolume.GVK.kind;
    }
    get resourceName() {
        return this.name;
    }
}
/**
 * A PersistentVolume (PV) is a piece of storage in the cluster that has been
 * provisioned by an administrator or dynamically provisioned using Storage Classes.
 * It is a resource in the cluster just like a node is a cluster resource.
 * PVs are volume plugins like Volumes, but have a lifecycle independent of any
 * individual Pod that uses the PV. This API object captures the details of the
 * implementation of the storage, be that NFS, iSCSI, or a
 * cloud-provider-specific storage system.
 */
class PersistentVolume extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.resourceType = 'persistentvolumes';
        this.storage = props.storage;
        this.mode = props.volumeMode ?? pvc.PersistentVolumeMode.FILE_SYSTEM;
        this.storageClassName = props.storageClassName;
        this._accessModes = props.accessModes;
        this.mountOptions = props.mountOptions;
        this.reclaimPolicy = props.reclaimPolicy ?? PersistentVolumeReclaimPolicy.RETAIN;
        if (props.claim) {
            this.bind(props.claim);
        }
        this.apiObject = new k8s.KubePersistentVolume(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
    }
    /**
     * Imports a pv from the cluster as a reference.
     */
    static fromPersistentVolumeName(scope, id, volumeName) {
        return new ImportedPersistentVolume(scope, id, volumeName);
    }
    /**
     * Access modes requirement of this claim.
     */
    get accessModes() {
        return this._accessModes ? [...this._accessModes] : undefined;
    }
    /**
     * PVC this volume is bound to. Undefined means this volume is not yet
     * claimed by any PVC.
     */
    get claim() {
        return this._claim;
    }
    /**
     * Reserve a `PersistentVolume` by creating a `PersistentVolumeClaim`
     * that is wired to claim this volume.
     *
     * Note that this method will throw in case the volume is already claimed.
     *
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/#reserving-a-persistentvolume
     */
    reserve() {
        const claim = new pvc.PersistentVolumeClaim(this, `${this.name}PVC`, {
            metadata: { name: `pvc-${this.name}`, namespace: this.metadata.namespace },
            // the storage classes must match, otherwise the claim
            // will use the default class (or no class at all), which may be different than the class
            // of this volume. note that other requirements are not needed since
            // when they are not defined, any volume satisifies them.
            storageClassName: this.storageClassName,
        });
        this.bind(claim);
        claim.bind(this);
        return claim;
    }
    /**
     * Bind a volume to a specific claim.
     * Note that you must also bind the claim to the volume.
     *
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/#binding
     *
     * @param claim The PVC to bind to.
     */
    bind(claim) {
        if (this._claim && this._claim.name !== claim.name) {
            throw new Error(`Cannot bind volume '${this.name}' to claim '${claim.name}' since it is already bound to claim '${this._claim.name}'`);
        }
        this._claim = claim;
    }
    asVolume() {
        const claim = this.reserve();
        return volume.Volume.fromPersistentVolumeClaim(this, 'Volume', claim);
    }
    /**
     * @internal
     */
    _toKube() {
        const storage = this.storage ? k8s.Quantity.fromString(this.storage.toGibibytes() + 'Gi') : undefined;
        return {
            claimRef: this._claim ? { name: this._claim?.name } : undefined,
            accessModes: this.accessModes?.map(a => a.toString()),
            capacity: storage ? { storage } : undefined,
            mountOptions: this.mountOptions?.map(o => o),
            storageClassName: this.storageClassName,
            persistentVolumeReclaimPolicy: this.reclaimPolicy,
            volumeMode: this.mode,
        };
    }
}
exports.PersistentVolume = PersistentVolume;
_a = JSII_RTTI_SYMBOL_1;
PersistentVolume[_a] = { fqn: "cdk8s-plus-25.PersistentVolume", version: "2.22.19" };
/**
 * Reclaim Policies.
 */
var PersistentVolumeReclaimPolicy;
(function (PersistentVolumeReclaimPolicy) {
    /**
     * The Retain reclaim policy allows for manual reclamation of the resource.
     * When the PersistentVolumeClaim is deleted, the PersistentVolume still exists and the
     * volume is considered "released". But it is not yet available for another claim
     * because the previous claimant's data remains on the volume.
     * An administrator can manually reclaim the volume with the following steps:
     *
     * 1. Delete the PersistentVolume. The associated storage asset in external
     *    infrastructure (such as an AWS EBS, GCE PD, Azure Disk, or Cinder volume) still exists after the PV is deleted.
     * 2. Manually clean up the data on the associated storage asset accordingly.
     * 3. Manually delete the associated storage asset.
     *
     * If you want to reuse the same storage asset, create a new PersistentVolume
     * with the same storage asset definition.
     */
    PersistentVolumeReclaimPolicy["RETAIN"] = "Retain";
    /**
     * For volume plugins that support the Delete reclaim policy, deletion removes both the
     * PersistentVolume object from Kubernetes, as well as the associated storage asset in
     * the external infrastructure, such as an AWS EBS, GCE PD, Azure Disk, or Cinder volume.
     * Volumes that were dynamically provisioned inherit the reclaim policy of their StorageClass, which defaults to Delete.
     * The administrator should configure the StorageClass according to users' expectations; otherwise,
     * the PV must be edited or patched after it is created
     */
    PersistentVolumeReclaimPolicy["DELETE"] = "Delete";
})(PersistentVolumeReclaimPolicy = exports.PersistentVolumeReclaimPolicy || (exports.PersistentVolumeReclaimPolicy = {}));
/**
 * Represents an AWS Disk resource that is attached to a kubelet's host machine and
 * then exposed to the pod.
 *
 * @see https://kubernetes.io/docs/concepts/storage/volumes#awselasticblockstore
 */
class AwsElasticBlockStorePersistentVolume extends PersistentVolume {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.volumeId = props.volumeId;
        this.readOnly = props.readOnly ?? false;
        this.fsType = props.fsType ?? 'ext4';
        this.partition = props.partition;
    }
    /**
     * @internal
     */
    _toKube() {
        const spec = super._toKube();
        return {
            ...spec,
            awsElasticBlockStore: {
                volumeId: this.volumeId,
                fsType: this.fsType,
                partition: this.partition,
                readOnly: this.readOnly,
            },
        };
    }
}
exports.AwsElasticBlockStorePersistentVolume = AwsElasticBlockStorePersistentVolume;
_b = JSII_RTTI_SYMBOL_1;
AwsElasticBlockStorePersistentVolume[_b] = { fqn: "cdk8s-plus-25.AwsElasticBlockStorePersistentVolume", version: "2.22.19" };
/**
 * AzureDisk represents an Azure Data Disk mount on the host and bind mount to the pod.
 */
class AzureDiskPersistentVolume extends PersistentVolume {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.diskName = props.diskName;
        this.diskUri = props.diskUri;
        this.cachingMode = props.cachingMode ?? volume.AzureDiskPersistentVolumeCachingMode.NONE;
        this.fsType = props.fsType ?? 'ext4';
        this.azureKind = props.kind ?? volume.AzureDiskPersistentVolumeKind.SHARED;
        this.readOnly = props.readOnly ?? false;
    }
    /**
     * @internal
     *
     * @see https://github.com/kubernetes/examples/blob/master/staging/volumes/azure_disk/README.md
     */
    _toKube() {
        const spec = super._toKube();
        return {
            ...spec,
            azureDisk: {
                diskName: this.diskName,
                diskUri: this.diskUri,
                cachingMode: this.cachingMode,
                fsType: this.fsType,
                kind: this.azureKind,
                readOnly: this.readOnly,
            },
        };
    }
}
exports.AzureDiskPersistentVolume = AzureDiskPersistentVolume;
_c = JSII_RTTI_SYMBOL_1;
AzureDiskPersistentVolume[_c] = { fqn: "cdk8s-plus-25.AzureDiskPersistentVolume", version: "2.22.19" };
/**
 * GCEPersistentDisk represents a GCE Disk resource that is attached to a kubelet's host machine
 * and then exposed to the pod. Provisioned by an admin.
 *
 * @see https://kubernetes.io/docs/concepts/storage/volumes#gcepersistentdisk
 */
class GCEPersistentDiskPersistentVolume extends PersistentVolume {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.pdName = props.pdName;
        this.readOnly = props.readOnly ?? false;
        this.fsType = props.fsType ?? 'ext4';
        this.partition = props.partition;
    }
    /**
     * @internal
     *
     * @see https://github.com/kubernetes/examples/blob/master/staging/volumes/azure_disk/README.md
     */
    _toKube() {
        const spec = super._toKube();
        return {
            ...spec,
            gcePersistentDisk: {
                pdName: this.pdName,
                fsType: this.fsType,
                partition: this.partition,
                readOnly: this.readOnly,
            },
        };
    }
}
exports.GCEPersistentDiskPersistentVolume = GCEPersistentDiskPersistentVolume;
_d = JSII_RTTI_SYMBOL_1;
GCEPersistentDiskPersistentVolume[_d] = { fqn: "cdk8s-plus-25.GCEPersistentDiskPersistentVolume", version: "2.22.19" };
//# sourceMappingURL=data:application/json;base64,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