"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WatchDynamoTable = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const constructs_1 = require("constructs");
const metrics_1 = require("./monitoring/aws/dynamodb/metrics");
const DEFAULT_PERCENT = 80;
class WatchDynamoTable extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const table = props.table;
        this.watchful = props.watchful;
        this.metrics = new metrics_1.DynamoDbMetricFactory();
        const cfnTable = table.node.defaultChild;
        const billingMode = cfnTable.billingMode ?? dynamodb.BillingMode.PROVISIONED;
        switch (billingMode) {
            case dynamodb.BillingMode.PAY_PER_REQUEST:
                this.createWidgetsForPayPerRequestTable(props.title, table);
                break;
            case dynamodb.BillingMode.PROVISIONED:
                this.createWidgetsForProvisionedTable(props.title, table, props.readCapacityThresholdPercent, props.writeCapacityThresholdPercent);
                break;
        }
    }
    /**
     * Create widgets for tables with billingMode=PROVISIONED
     * Include alarms when capacity is over 80% of the provisioned value
     */
    createWidgetsForProvisionedTable(title, table, readCapacityThresholdPercent, writeCapacityThresholdPercent) {
        const cfnTable = table.node.defaultChild;
        const metrics = this.metrics.metricConsumedCapacityUnits(table.tableName);
        const readCapacityMetric = metrics.read;
        const writeCapacityMetric = metrics.write;
        const throughput = cfnTable.provisionedThroughput;
        this.watchful.addAlarm(this.createDynamoCapacityAlarm('read', readCapacityMetric, throughput.readCapacityUnits, readCapacityThresholdPercent));
        this.watchful.addAlarm(this.createDynamoCapacityAlarm('write', writeCapacityMetric, throughput.writeCapacityUnits, writeCapacityThresholdPercent));
        this.watchful.addSection(title, {
            links: [{ title: 'Amazon DynamoDB Console', url: linkForDynamoTable(table) }],
        });
        this.watchful.addWidgets(this.createDynamoCapacityGraph('Read', readCapacityMetric, throughput.readCapacityUnits, readCapacityThresholdPercent), this.createDynamoCapacityGraph('Write', writeCapacityMetric, throughput.writeCapacityUnits, writeCapacityThresholdPercent));
    }
    /**
     * Create widgets for tables with billingMode=PAY_PER_REQUEST
     * Include consumed capacity metrics
     */
    createWidgetsForPayPerRequestTable(title, table) {
        const metrics = this.metrics.metricConsumedCapacityUnits(table.tableName);
        const readCapacityMetric = metrics.read;
        const writeCapacityMetric = metrics.write;
        this.watchful.addSection(title, {
            links: [{ title: 'Amazon DynamoDB Console', url: linkForDynamoTable(table) }],
        });
        this.watchful.addWidgets(this.createDynamoPPRGraph('Read', readCapacityMetric), this.createDynamoPPRGraph('Write', writeCapacityMetric));
    }
    createDynamoCapacityGraph(type, metric, provisioned, percent = DEFAULT_PERCENT) {
        return new cloudwatch.GraphWidget({
            title: `${type} Capacity Units/${metric.period.toMinutes()}min`,
            width: 12,
            stacked: true,
            left: [metric],
            leftAnnotations: [
                {
                    label: 'Provisioned',
                    value: provisioned * metric.period.toSeconds(),
                    color: '#58D68D',
                },
                {
                    color: '#FF3333',
                    label: `Alarm on ${percent}%`,
                    value: calculateUnits(provisioned, percent, metric.period),
                },
            ],
        });
    }
    createDynamoPPRGraph(type, metric) {
        return new cloudwatch.GraphWidget({
            title: `${type} Capacity Units/${metric.period.toMinutes()}min`,
            width: 12,
            stacked: true,
            left: [metric],
        });
    }
    createDynamoCapacityAlarm(type, metric, provisioned, percent = DEFAULT_PERCENT) {
        const periodMinutes = 5;
        const threshold = calculateUnits(provisioned, percent, aws_cdk_lib_1.Duration.minutes(periodMinutes));
        const metricWithPeriod = metric.with({
            statistic: 'sum',
            period: aws_cdk_lib_1.Duration.minutes(periodMinutes),
        });
        const alarm = metricWithPeriod.createAlarm(this, `CapacityAlarm:${type}`, {
            alarmDescription: `at ${threshold}% of ${type} capacity`,
            threshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
        });
        return alarm;
    }
}
exports.WatchDynamoTable = WatchDynamoTable;
_a = JSII_RTTI_SYMBOL_1;
WatchDynamoTable[_a] = { fqn: "cdk-watchful.WatchDynamoTable", version: "0.6.417" };
function linkForDynamoTable(table, tab = 'overview') {
    return `https://console.aws.amazon.com/dynamodb/home?region=${table.stack.region}#tables:selected=${table.tableName};tab=${tab}`;
}
function calculateUnits(provisioned, percent, period) {
    return provisioned * ((percent === undefined ? DEFAULT_PERCENT : percent) / 100) * period.toSeconds();
}
//# sourceMappingURL=data:application/json;base64,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