"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WatchEcsService = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const constructs_1 = require("constructs");
const metrics_1 = require("./monitoring/aws/ecs/metrics");
class WatchEcsService extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.watchful = props.watchful;
        if (props.ec2Service) {
            this.ecsService = props.ec2Service;
            this.serviceName = props.ec2Service.serviceName;
            this.clusterName = props.ec2Service.cluster.clusterName;
        }
        else if (props.fargateService) {
            this.ecsService = props.fargateService;
            this.serviceName = props.fargateService.serviceName;
            this.clusterName = props.fargateService.cluster.clusterName;
        }
        else {
            throw new Error('No service provided to monitor.');
        }
        this.targetGroup = props.targetGroup;
        this.targetGroupName = this.targetGroup.targetGroupFullName;
        this.loadBalancerName = this.targetGroup.firstLoadBalancerFullName;
        this.metrics = new metrics_1.EcsMetricFactory();
        this.watchful.addSection(props.title, {
            links: [
                { title: 'ECS Service', url: linkForEcsService(this.ecsService) },
            ],
        });
        const { cpuUtilizationMetric, cpuUtilizationAlarm } = this.createCpuUtilizationMonitor(props.cpuMaximumThresholdPercent);
        const { memoryUtilizationMetric, memoryUtilizationAlarm } = this.createMemoryUtilizationMonitor(props.memoryMaximumThresholdPercent);
        const { targetResponseTimeMetric, targetResponseTimeAlarm } = this.createTargetResponseTimeMonitor(props.targetResponseTimeThreshold);
        const { healthyHostsMetric, unhealthyHostsMetric } = this.createHostCountMetrics();
        const { requestsMetric, requestsAlarm } = this.createRequestsMonitor(props.requestsThreshold);
        const { http2xxMetric, http3xxMetric, http4xxMetric, http5xxMetric } = this.createHttpRequestsMetrics();
        const { requestsErrorRateMetric, requestsErrorRateAlarm } = this.requestsErrorRate(props.requestsErrorRateThreshold);
        this.watchful.addWidgets(new cloudwatch.GraphWidget({
            title: `CPUUtilization/${cpuUtilizationMetric.period.toMinutes()}min`,
            width: 12,
            left: [cpuUtilizationMetric],
            leftAnnotations: [cpuUtilizationAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `MemoryUtilization/${memoryUtilizationMetric.period.toMinutes()}min`,
            width: 12,
            left: [memoryUtilizationMetric],
            leftAnnotations: [memoryUtilizationAlarm.toAnnotation()],
        }));
        this.watchful.addWidgets(new cloudwatch.SingleValueWidget({
            title: 'Healthy Hosts',
            height: 6,
            width: 6,
            metrics: [healthyHostsMetric],
        }), new cloudwatch.SingleValueWidget({
            title: 'UnHealthy Hosts',
            height: 6,
            width: 6,
            metrics: [unhealthyHostsMetric],
        }), new cloudwatch.GraphWidget({
            title: `TargetResponseTime/${targetResponseTimeMetric.period.toMinutes()}min`,
            width: 6,
            left: [targetResponseTimeMetric],
            leftAnnotations: [targetResponseTimeAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `Requests/${requestsMetric.period.toMinutes()}min`,
            width: 6,
            left: [requestsMetric],
            leftAnnotations: [requestsAlarm.toAnnotation()],
        }));
        this.watchful.addWidgets(new cloudwatch.GraphWidget({
            title: 'HTTP Requests Overview',
            width: 12,
            left: [http2xxMetric, http3xxMetric, http4xxMetric, http5xxMetric],
        }), new cloudwatch.GraphWidget({
            title: `HTTP Requests Error rate/${requestsErrorRateMetric.period.toMinutes()}min`,
            width: 12,
            left: [requestsErrorRateMetric],
            leftAnnotations: [requestsErrorRateAlarm.toAnnotation()],
        }));
    }
    createCpuUtilizationMonitor(cpuMaximumThresholdPercent = 0) {
        const cpuUtilizationMetric = this.metrics.metricCpuUtilizationAverage(this.clusterName, this.serviceName);
        const cpuUtilizationAlarm = cpuUtilizationMetric.createAlarm(this, 'cpuUtilizationAlarm', {
            alarmDescription: 'cpuUtilizationAlarm',
            threshold: cpuMaximumThresholdPercent,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(cpuUtilizationAlarm);
        return { cpuUtilizationMetric, cpuUtilizationAlarm };
    }
    createMemoryUtilizationMonitor(memoryMaximumThresholdPercent = 0) {
        const memoryUtilizationMetric = this.metrics.metricMemoryUtilizationAverage(this.clusterName, this.serviceName);
        const memoryUtilizationAlarm = memoryUtilizationMetric.createAlarm(this, 'memoryUtilizationAlarm', {
            alarmDescription: 'memoryUtilizationAlarm',
            threshold: memoryMaximumThresholdPercent,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(memoryUtilizationAlarm);
        return { memoryUtilizationMetric, memoryUtilizationAlarm };
    }
    createTargetResponseTimeMonitor(targetResponseTimeThreshold = 0) {
        const targetResponseTimeMetric = this.metrics.metricTargetResponseTime(this.targetGroupName, this.loadBalancerName).avg;
        const targetResponseTimeAlarm = targetResponseTimeMetric.createAlarm(this, 'targetResponseTimeAlarm', {
            alarmDescription: 'targetResponseTimeAlarm',
            threshold: targetResponseTimeThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(targetResponseTimeAlarm);
        return { targetResponseTimeMetric, targetResponseTimeAlarm };
    }
    createRequestsMonitor(requestsThreshold = 0) {
        const requestsMetric = this.metrics.metricRequestCount(this.targetGroupName, this.loadBalancerName);
        const requestsAlarm = requestsMetric.createAlarm(this, 'requestsAlarm', {
            alarmDescription: 'requestsAlarm',
            threshold: requestsThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(requestsAlarm);
        return { requestsMetric, requestsAlarm };
    }
    createHttpRequestsMetrics() {
        const metrics = this.metrics.metricHttpStatusCodeCount(this.targetGroupName, this.loadBalancerName);
        const http2xxMetric = metrics.count2XX;
        const http3xxMetric = metrics.count3XX;
        const http4xxMetric = metrics.count4XX;
        const http5xxMetric = metrics.count5XX;
        return { http2xxMetric, http3xxMetric, http4xxMetric, http5xxMetric };
    }
    createHostCountMetrics() {
        const healthyHostsMetric = this.metrics.metricMinHealthyHostCount(this.targetGroupName, this.loadBalancerName);
        const unhealthyHostsMetric = this.metrics.metricMaxUnhealthyHostCount(this.targetGroupName, this.loadBalancerName);
        return { healthyHostsMetric, unhealthyHostsMetric };
    }
    requestsErrorRate(requestsErrorRateThreshold = 0) {
        const requestsErrorRateMetric = this.metrics.metricHttpErrorStatusCodeRate(this.targetGroupName, this.loadBalancerName);
        const requestsErrorRateAlarm = requestsErrorRateMetric.createAlarm(this, 'requestsErrorRateAlarm', {
            alarmDescription: 'requestsErrorRateAlarm',
            threshold: requestsErrorRateThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(requestsErrorRateAlarm);
        return { requestsErrorRateMetric, requestsErrorRateAlarm };
    }
}
exports.WatchEcsService = WatchEcsService;
_a = JSII_RTTI_SYMBOL_1;
WatchEcsService[_a] = { fqn: "cdk-watchful.WatchEcsService", version: "0.6.417" };
function linkForEcsService(ecsService) {
    return `https://console.aws.amazon.com/ecs/home?region=${ecsService.stack.region}#/clusters/${ecsService.cluster.clusterName}/services/${ecsService.serviceName}/details`;
}
//# sourceMappingURL=data:application/json;base64,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