"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WatchLambdaFunction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const constructs_1 = require("constructs");
const metrics_1 = require("./monitoring/aws/lambda/metrics");
const DEFAULT_DURATION_THRESHOLD_PERCENT = 80;
class WatchLambdaFunction extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const cfnFunction = props.fn.node.defaultChild;
        const timeoutSec = cfnFunction.timeout || 3;
        this.watchful = props.watchful;
        this.fn = props.fn;
        this.metrics = new metrics_1.LambdaMetricFactory();
        this.watchful.addSection(props.title, {
            links: [
                { title: 'AWS Lambda Console', url: linkForLambdaFunction(this.fn) },
                { title: 'CloudWatch Logs', url: linkForLambdaLogs(this.fn) },
            ],
        });
        const { errorsMetric, errorsAlarm } = this.createErrorsMonitor(props.errorsPerMinuteThreshold);
        const { throttlesMetric, throttlesAlarm } = this.createThrottlesMonitor(props.throttlesPerMinuteThreshold);
        const { durationMetric, durationAlarm } = this.createDurationMonitor(timeoutSec, props.durationThresholdPercent);
        const invocationsMetric = this.metrics.metricInvocations(this.fn.functionName);
        this.watchful.addWidgets(new cloudwatch.GraphWidget({
            title: `Invocations/${invocationsMetric.period.toMinutes()}min`,
            width: 6,
            left: [invocationsMetric],
        }), new cloudwatch.GraphWidget({
            title: `Errors/${errorsMetric.period.toMinutes()}min`,
            width: 6,
            left: [errorsMetric],
            leftAnnotations: [errorsAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `Throttles/${throttlesMetric.period.toMinutes()}min`,
            width: 6,
            left: [throttlesMetric],
            leftAnnotations: [throttlesAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `Duration/${durationMetric.period.toMinutes()}min`,
            width: 6,
            left: [durationMetric],
            leftAnnotations: [durationAlarm.toAnnotation()],
        }));
    }
    createErrorsMonitor(errorsPerMinuteThreshold = 0) {
        const fn = this.fn;
        const errorsMetric = this.metrics.metricErrors(fn.functionName);
        const errorsAlarm = errorsMetric.createAlarm(this, 'ErrorsAlarm', {
            alarmDescription: `Over ${errorsPerMinuteThreshold} errors per minute`,
            threshold: errorsPerMinuteThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(errorsAlarm);
        return { errorsMetric, errorsAlarm };
    }
    createThrottlesMonitor(throttlesPerMinuteThreshold = 0) {
        const fn = this.fn;
        const throttlesMetric = this.metrics.metricThrottles(fn.functionName);
        const throttlesAlarm = throttlesMetric.createAlarm(this, 'ThrottlesAlarm', {
            alarmDescription: `Over ${throttlesPerMinuteThreshold} throttles per minute`,
            threshold: throttlesPerMinuteThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(throttlesAlarm);
        return { throttlesMetric, throttlesAlarm };
    }
    createDurationMonitor(timeoutSec, durationPercentThreshold = DEFAULT_DURATION_THRESHOLD_PERCENT) {
        const fn = this.fn;
        const durationMetric = this.metrics.metricDuration(fn.functionName).p99;
        const durationThresholdSec = Math.floor(durationPercentThreshold / 100 * timeoutSec);
        const durationAlarm = durationMetric.createAlarm(this, 'DurationAlarm', {
            alarmDescription: `p99 latency >= ${durationThresholdSec}s (${durationPercentThreshold}%)`,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            threshold: durationThresholdSec * 1000,
            evaluationPeriods: 3,
        });
        this.watchful.addAlarm(durationAlarm);
        return { durationMetric, durationAlarm };
    }
}
exports.WatchLambdaFunction = WatchLambdaFunction;
_a = JSII_RTTI_SYMBOL_1;
WatchLambdaFunction[_a] = { fqn: "cdk-watchful.WatchLambdaFunction", version: "0.6.417" };
function linkForLambdaFunction(fn, tab = 'graph') {
    return `https://console.aws.amazon.com/lambda/home?region=${fn.stack.region}#/functions/${fn.functionName}?tab=${tab}`;
}
function linkForLambdaLogs(fn) {
    return `https://console.aws.amazon.com/cloudwatch/home?region=${fn.stack.region}#logEventViewer:group=/aws/lambda/${fn.functionName}`;
}
//# sourceMappingURL=data:application/json;base64,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