"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WatchRdsAurora = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const constructs_1 = require("constructs");
const metrics_1 = require("./monitoring/aws/rds/metrics");
class WatchRdsAurora extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.watchful = props.watchful;
        this.cluster = props.cluster;
        this.metrics = new metrics_1.RdsAuroraMetricFactory();
        this.watchful.addSection(props.title, {
            links: [
                { title: 'AWS RDS Cluster', url: linkForRDS(this.cluster) },
            ],
        });
        const { cpuUtilizationMetric, cpuUtilizationAlarm } = this.createCpuUtilizationMonitor(props.cpuMaximumThresholdPercent);
        const { dbConnectionsMetric, dbConnectionsAlarm } = this.createDbConnectionsMonitor(props.dbConnectionsMaximumThreshold);
        const { dbReplicaLagMetric, dbReplicaLagAlarm } = this.createDbReplicaLagMonitor(props.dbReplicaLagMaximumThreshold);
        const { dbBufferCacheHitRatioMetric, dbBufferCacheHitRatioAlarm } = this.createDbBufferCacheMonitor(props.dbBufferCacheMinimumThreshold);
        const { dbInsertThroughputMetric, dbUpdateThroughputMetric, dbSelectThroughputMetric, dbDeleteThroughputMetric } = this.createDbDmlThroughputMonitor(props.dbThroughputMaximumThreshold);
        this.watchful.addWidgets(new cloudwatch.GraphWidget({
            title: `CPUUtilization/${cpuUtilizationMetric.period.toMinutes()}min`,
            width: 6,
            left: [cpuUtilizationMetric],
            leftAnnotations: [cpuUtilizationAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `DB Connections/${dbConnectionsMetric.period.toMinutes()}min`,
            width: 6,
            left: [dbConnectionsMetric],
            leftAnnotations: [dbConnectionsAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `DB Replica Lag/${dbReplicaLagMetric.period.toMinutes()}min`,
            width: 6,
            left: [dbReplicaLagMetric],
            leftAnnotations: [dbReplicaLagAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `DB BufferCache Hit Ratio/${dbBufferCacheHitRatioMetric.period.toMinutes()}min`,
            width: 6,
            left: [dbBufferCacheHitRatioMetric],
            leftAnnotations: [dbBufferCacheHitRatioAlarm.toAnnotation()],
        }));
        this.watchful.addWidgets(new cloudwatch.GraphWidget({
            title: 'RDS DML Overview',
            width: 24,
            left: [dbInsertThroughputMetric, dbUpdateThroughputMetric, dbSelectThroughputMetric, dbDeleteThroughputMetric],
        }));
    }
    createCpuUtilizationMonitor(cpuMaximumThresholdPercent = 80) {
        const cpuUtilizationMetric = this.metrics.metricCpuUtilization(this.cluster.clusterIdentifier);
        const cpuUtilizationAlarm = cpuUtilizationMetric.createAlarm(this, 'CpuUtilizationAlarm', {
            alarmDescription: 'cpuUtilizationAlarm',
            threshold: cpuMaximumThresholdPercent,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        return { cpuUtilizationMetric, cpuUtilizationAlarm };
    }
    createDbConnectionsMonitor(dbConnectionsMaximumThreshold = 0) {
        const dbConnectionsMetric = this.metrics.metricDbConnections(this.cluster.clusterIdentifier);
        const dbConnectionsAlarm = dbConnectionsMetric.createAlarm(this, 'DbConnectionsAlarm', {
            alarmDescription: 'dbConnectionsAlarm',
            threshold: dbConnectionsMaximumThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        return { dbConnectionsMetric, dbConnectionsAlarm };
    }
    createDbReplicaLagMonitor(dbReplicaLagMaximumThreshold = 0) {
        const dbReplicaLagMetric = this.metrics.metricReplicaLag(this.cluster.clusterIdentifier);
        const dbReplicaLagAlarm = dbReplicaLagMetric.createAlarm(this, 'DbReplicaLagAlarm', {
            alarmDescription: 'dbConnectionsAlarm',
            threshold: dbReplicaLagMaximumThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        return { dbReplicaLagMetric, dbReplicaLagAlarm };
    }
    createDbBufferCacheMonitor(dbBufferCacheMinimumThreshold = 0) {
        const dbBufferCacheHitRatioMetric = this.metrics.metricBufferCacheHitRatio(this.cluster.clusterIdentifier);
        const dbBufferCacheHitRatioAlarm = dbBufferCacheHitRatioMetric.createAlarm(this, 'DbBufferCacheHitRatioAlarm', {
            alarmDescription: 'dbConnectionsAlarm',
            threshold: dbBufferCacheMinimumThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        return { dbBufferCacheHitRatioMetric, dbBufferCacheHitRatioAlarm };
    }
    createDbDmlThroughputMonitor(dbThroughputMaximumThreshold = 0) {
        // @ts-ignore
        const AlarmThreshold = dbThroughputMaximumThreshold;
        return this.metrics.metricDmlThroughput(this.cluster.clusterIdentifier);
    }
}
exports.WatchRdsAurora = WatchRdsAurora;
_a = JSII_RTTI_SYMBOL_1;
WatchRdsAurora[_a] = { fqn: "cdk-watchful.WatchRdsAurora", version: "0.6.417" };
function linkForRDS(cluster) {
    return `https://console.aws.amazon.com/rds/home?region=${cluster.stack.region}#database:id=${cluster.clusterIdentifier};is-cluster=true`;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmRzLWF1cm9yYS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9yZHMtYXVyb3JhLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEseURBQXlEO0FBRXpELDJDQUF1QztBQUV2QywwREFBc0U7QUE4Q3RFLE1BQWEsY0FBZSxTQUFRLHNCQUFTO0lBTTNDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBMEI7UUFDbEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7UUFDL0IsSUFBSSxDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzdCLElBQUksQ0FBQyxPQUFPLEdBQUcsSUFBSSxnQ0FBc0IsRUFBRSxDQUFDO1FBRTVDLElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUU7WUFDcEMsS0FBSyxFQUFFO2dCQUNMLEVBQUUsS0FBSyxFQUFFLGlCQUFpQixFQUFFLEdBQUcsRUFBRSxVQUFVLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxFQUFFO2FBQzVEO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxFQUFFLG9CQUFvQixFQUFFLG1CQUFtQixFQUFFLEdBQUcsSUFBSSxDQUFDLDJCQUEyQixDQUFDLEtBQUssQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1FBQ3pILE1BQU0sRUFBRSxtQkFBbUIsRUFBRSxrQkFBa0IsRUFBRSxHQUFHLElBQUksQ0FBQywwQkFBMEIsQ0FBQyxLQUFLLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN6SCxNQUFNLEVBQUUsa0JBQWtCLEVBQUUsaUJBQWlCLEVBQUUsR0FBRyxJQUFJLENBQUMseUJBQXlCLENBQUMsS0FBSyxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDckgsTUFBTSxFQUFFLDJCQUEyQixFQUFFLDBCQUEwQixFQUFFLEdBQUcsSUFBSSxDQUFDLDBCQUEwQixDQUFDLEtBQUssQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBRXpJLE1BQU0sRUFBRSx3QkFBd0IsRUFBRSx3QkFBd0IsRUFBRSx3QkFBd0IsRUFBRSx3QkFBd0IsRUFBRSxHQUM1RyxJQUFJLENBQUMsNEJBQTRCLENBQUMsS0FBSyxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFFMUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQ3RCLElBQUksVUFBVSxDQUFDLFdBQVcsQ0FBQztZQUN6QixLQUFLLEVBQUUsa0JBQWtCLG9CQUFvQixDQUFDLE1BQU0sQ0FBQyxTQUFTLEVBQUUsS0FBSztZQUNyRSxLQUFLLEVBQUUsQ0FBQztZQUNSLElBQUksRUFBRSxDQUFDLG9CQUFvQixDQUFDO1lBQzVCLGVBQWUsRUFBRSxDQUFDLG1CQUFtQixDQUFDLFlBQVksRUFBRSxDQUFDO1NBQ3RELENBQUMsRUFDRixJQUFJLFVBQVUsQ0FBQyxXQUFXLENBQUM7WUFDekIsS0FBSyxFQUFFLGtCQUFrQixtQkFBbUIsQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFLEtBQUs7WUFDcEUsS0FBSyxFQUFFLENBQUM7WUFDUixJQUFJLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQztZQUMzQixlQUFlLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxZQUFZLEVBQUUsQ0FBQztTQUNyRCxDQUFDLEVBQ0YsSUFBSSxVQUFVLENBQUMsV0FBVyxDQUFDO1lBQ3pCLEtBQUssRUFBRSxrQkFBa0Isa0JBQWtCLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBRSxLQUFLO1lBQ25FLEtBQUssRUFBRSxDQUFDO1lBQ1IsSUFBSSxFQUFFLENBQUMsa0JBQWtCLENBQUM7WUFDMUIsZUFBZSxFQUFFLENBQUMsaUJBQWlCLENBQUMsWUFBWSxFQUFFLENBQUM7U0FDcEQsQ0FBQyxFQUNGLElBQUksVUFBVSxDQUFDLFdBQVcsQ0FBQztZQUN6QixLQUFLLEVBQUUsNEJBQTRCLDJCQUEyQixDQUFDLE1BQU0sQ0FBQyxTQUFTLEVBQUUsS0FBSztZQUN0RixLQUFLLEVBQUUsQ0FBQztZQUNSLElBQUksRUFBRSxDQUFDLDJCQUEyQixDQUFDO1lBQ25DLGVBQWUsRUFBRSxDQUFDLDBCQUEwQixDQUFDLFlBQVksRUFBRSxDQUFDO1NBQzdELENBQUMsQ0FDSCxDQUFDO1FBQ0YsSUFBSSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQ3RCLElBQUksVUFBVSxDQUFDLFdBQVcsQ0FBQztZQUN6QixLQUFLLEVBQUUsa0JBQWtCO1lBQ3pCLEtBQUssRUFBRSxFQUFFO1lBQ1QsSUFBSSxFQUFFLENBQUMsd0JBQXdCLEVBQUUsd0JBQXdCLEVBQUUsd0JBQXdCLEVBQUUsd0JBQXdCLENBQUM7U0FDL0csQ0FBQyxDQUNILENBQUM7SUFDSixDQUFDO0lBRU8sMkJBQTJCLENBQUMsMEJBQTBCLEdBQUcsRUFBRTtRQUNqRSxNQUFNLG9CQUFvQixHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsb0JBQW9CLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBQy9GLE1BQU0sbUJBQW1CLEdBQUcsb0JBQW9CLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSxxQkFBcUIsRUFBRTtZQUN4RixnQkFBZ0IsRUFBRSxxQkFBcUI7WUFDdkMsU0FBUyxFQUFFLDBCQUEwQjtZQUNyQyxrQkFBa0IsRUFBRSxVQUFVLENBQUMsa0JBQWtCLENBQUMsc0JBQXNCO1lBQ3hFLGlCQUFpQixFQUFFLENBQUM7U0FDckIsQ0FBQyxDQUFDO1FBQ0gsT0FBTyxFQUFFLG9CQUFvQixFQUFFLG1CQUFtQixFQUFFLENBQUM7SUFDdkQsQ0FBQztJQUVPLDBCQUEwQixDQUFDLDZCQUE2QixHQUFHLENBQUM7UUFDbEUsTUFBTSxtQkFBbUIsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLG1CQUFtQixDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsaUJBQWlCLENBQUMsQ0FBQztRQUM3RixNQUFNLGtCQUFrQixHQUFHLG1CQUFtQixDQUFDLFdBQVcsQ0FBQyxJQUFJLEVBQUUsb0JBQW9CLEVBQUU7WUFDckYsZ0JBQWdCLEVBQUUsb0JBQW9CO1lBQ3RDLFNBQVMsRUFBRSw2QkFBNkI7WUFDeEMsa0JBQWtCLEVBQUUsVUFBVSxDQUFDLGtCQUFrQixDQUFDLHNCQUFzQjtZQUN4RSxpQkFBaUIsRUFBRSxDQUFDO1NBQ3JCLENBQUMsQ0FBQztRQUNILE9BQU8sRUFBRSxtQkFBbUIsRUFBRSxrQkFBa0IsRUFBRSxDQUFDO0lBQ3JELENBQUM7SUFFTyx5QkFBeUIsQ0FBQyw0QkFBNEIsR0FBRyxDQUFDO1FBQ2hFLE1BQU0sa0JBQWtCLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFDekYsTUFBTSxpQkFBaUIsR0FBRyxrQkFBa0IsQ0FBQyxXQUFXLENBQUMsSUFBSSxFQUFFLG1CQUFtQixFQUFFO1lBQ2xGLGdCQUFnQixFQUFFLG9CQUFvQjtZQUN0QyxTQUFTLEVBQUUsNEJBQTRCO1lBQ3ZDLGtCQUFrQixFQUFFLFVBQVUsQ0FBQyxrQkFBa0IsQ0FBQyxzQkFBc0I7WUFDeEUsaUJBQWlCLEVBQUUsQ0FBQztTQUNyQixDQUFDLENBQUM7UUFDSCxPQUFPLEVBQUUsa0JBQWtCLEVBQUUsaUJBQWlCLEVBQUUsQ0FBQztJQUNuRCxDQUFDO0lBRU8sMEJBQTBCLENBQUMsNkJBQTZCLEdBQUcsQ0FBQztRQUNsRSxNQUFNLDJCQUEyQixHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMseUJBQXlCLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBQzNHLE1BQU0sMEJBQTBCLEdBQUcsMkJBQTJCLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSw0QkFBNEIsRUFBRTtZQUM3RyxnQkFBZ0IsRUFBRSxvQkFBb0I7WUFDdEMsU0FBUyxFQUFFLDZCQUE2QjtZQUN4QyxrQkFBa0IsRUFBRSxVQUFVLENBQUMsa0JBQWtCLENBQUMsbUJBQW1CO1lBQ3JFLGlCQUFpQixFQUFFLENBQUM7U0FDckIsQ0FBQyxDQUFDO1FBQ0gsT0FBTyxFQUFFLDJCQUEyQixFQUFFLDBCQUEwQixFQUFFLENBQUM7SUFDckUsQ0FBQztJQUNPLDRCQUE0QixDQUFDLDRCQUE0QixHQUFHLENBQUM7UUFDbkUsYUFBYTtRQUNiLE1BQU0sY0FBYyxHQUFHLDRCQUE0QixDQUFDO1FBQ3BELE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDMUUsQ0FBQzs7QUE3R0gsd0NBOEdDOzs7QUFFRCxTQUFTLFVBQVUsQ0FBQyxPQUE0QjtJQUM5QyxPQUFPLGtEQUFrRCxPQUFPLENBQUMsS0FBSyxDQUFDLE1BQU0sZ0JBQWdCLE9BQU8sQ0FBQyxpQkFBaUIsa0JBQWtCLENBQUM7QUFDM0ksQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNsb3Vkd2F0Y2ggZnJvbSAnYXdzLWNkay1saWIvYXdzLWNsb3Vkd2F0Y2gnO1xuaW1wb3J0ICogYXMgcmRzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1yZHMnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBJV2F0Y2hmdWwgfSBmcm9tICcuL2FwaSc7XG5pbXBvcnQgeyBSZHNBdXJvcmFNZXRyaWNGYWN0b3J5IH0gZnJvbSAnLi9tb25pdG9yaW5nL2F3cy9yZHMvbWV0cmljcyc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgV2F0Y2hSZHNBdXJvcmFPcHRpb25zIHtcbiAgLyoqXG4gICAgICogVGhyZXNob2xkIGZvciB0aGUgQ3B1IE1heGltdW0gdXRpbGl6YXRpb25cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IDgwXG4gICAgICovXG4gIHJlYWRvbmx5IGNwdU1heGltdW1UaHJlc2hvbGRQZXJjZW50PzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaHJlc2hvbGQgZm9yIHRoZSBNYXhpbXVtIERiIENvbm5lY3Rpb25zLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIDAuXG4gICAqL1xuICByZWFkb25seSBkYkNvbm5lY3Rpb25zTWF4aW11bVRocmVzaG9sZD86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhyZXNob2xkIGZvciB0aGUgTWF4aW11bSBEYiBSZXBsaWNhTGFnLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIDAuXG4gICAqL1xuICByZWFkb25seSBkYlJlcGxpY2FMYWdNYXhpbXVtVGhyZXNob2xkPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaHJlc2hvbGQgZm9yIHRoZSBNYXhpbXVtIERiIFRocm91Z2hwdXQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gMC5cbiAgICovXG4gIHJlYWRvbmx5IGRiVGhyb3VnaHB1dE1heGltdW1UaHJlc2hvbGQ/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRocmVzaG9sZCBmb3IgdGhlIE1pbmltdW0gRGIgQnVmZmVyIENhY2hlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIDAuXG4gICAqL1xuICByZWFkb25seSBkYkJ1ZmZlckNhY2hlTWluaW11bVRocmVzaG9sZD86IG51bWJlcjtcblxufVxuXG5leHBvcnQgaW50ZXJmYWNlIFdhdGNoUmRzQXVyb3JhUHJvcHMgZXh0ZW5kcyBXYXRjaFJkc0F1cm9yYU9wdGlvbnMge1xuICByZWFkb25seSB0aXRsZTogc3RyaW5nO1xuICByZWFkb25seSB3YXRjaGZ1bDogSVdhdGNoZnVsO1xuICByZWFkb25seSBjbHVzdGVyOiByZHMuRGF0YWJhc2VDbHVzdGVyO1xufVxuXG5leHBvcnQgY2xhc3MgV2F0Y2hSZHNBdXJvcmEgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgd2F0Y2hmdWw6IElXYXRjaGZ1bDtcbiAgcHJpdmF0ZSByZWFkb25seSBjbHVzdGVyOiByZHMuRGF0YWJhc2VDbHVzdGVyO1xuICBwcml2YXRlIHJlYWRvbmx5IG1ldHJpY3M6IFJkc0F1cm9yYU1ldHJpY0ZhY3Rvcnk7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFdhdGNoUmRzQXVyb3JhUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy53YXRjaGZ1bCA9IHByb3BzLndhdGNoZnVsO1xuICAgIHRoaXMuY2x1c3RlciA9IHByb3BzLmNsdXN0ZXI7XG4gICAgdGhpcy5tZXRyaWNzID0gbmV3IFJkc0F1cm9yYU1ldHJpY0ZhY3RvcnkoKTtcblxuICAgIHRoaXMud2F0Y2hmdWwuYWRkU2VjdGlvbihwcm9wcy50aXRsZSwge1xuICAgICAgbGlua3M6IFtcbiAgICAgICAgeyB0aXRsZTogJ0FXUyBSRFMgQ2x1c3RlcicsIHVybDogbGlua0ZvclJEUyh0aGlzLmNsdXN0ZXIpIH0sXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgY29uc3QgeyBjcHVVdGlsaXphdGlvbk1ldHJpYywgY3B1VXRpbGl6YXRpb25BbGFybSB9ID0gdGhpcy5jcmVhdGVDcHVVdGlsaXphdGlvbk1vbml0b3IocHJvcHMuY3B1TWF4aW11bVRocmVzaG9sZFBlcmNlbnQpO1xuICAgIGNvbnN0IHsgZGJDb25uZWN0aW9uc01ldHJpYywgZGJDb25uZWN0aW9uc0FsYXJtIH0gPSB0aGlzLmNyZWF0ZURiQ29ubmVjdGlvbnNNb25pdG9yKHByb3BzLmRiQ29ubmVjdGlvbnNNYXhpbXVtVGhyZXNob2xkKTtcbiAgICBjb25zdCB7IGRiUmVwbGljYUxhZ01ldHJpYywgZGJSZXBsaWNhTGFnQWxhcm0gfSA9IHRoaXMuY3JlYXRlRGJSZXBsaWNhTGFnTW9uaXRvcihwcm9wcy5kYlJlcGxpY2FMYWdNYXhpbXVtVGhyZXNob2xkKTtcbiAgICBjb25zdCB7IGRiQnVmZmVyQ2FjaGVIaXRSYXRpb01ldHJpYywgZGJCdWZmZXJDYWNoZUhpdFJhdGlvQWxhcm0gfSA9IHRoaXMuY3JlYXRlRGJCdWZmZXJDYWNoZU1vbml0b3IocHJvcHMuZGJCdWZmZXJDYWNoZU1pbmltdW1UaHJlc2hvbGQpO1xuXG4gICAgY29uc3QgeyBkYkluc2VydFRocm91Z2hwdXRNZXRyaWMsIGRiVXBkYXRlVGhyb3VnaHB1dE1ldHJpYywgZGJTZWxlY3RUaHJvdWdocHV0TWV0cmljLCBkYkRlbGV0ZVRocm91Z2hwdXRNZXRyaWMgfSA9XG4gICAgICAgIHRoaXMuY3JlYXRlRGJEbWxUaHJvdWdocHV0TW9uaXRvcihwcm9wcy5kYlRocm91Z2hwdXRNYXhpbXVtVGhyZXNob2xkKTtcblxuICAgIHRoaXMud2F0Y2hmdWwuYWRkV2lkZ2V0cyhcbiAgICAgIG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHtcbiAgICAgICAgdGl0bGU6IGBDUFVVdGlsaXphdGlvbi8ke2NwdVV0aWxpemF0aW9uTWV0cmljLnBlcmlvZC50b01pbnV0ZXMoKX1taW5gLFxuICAgICAgICB3aWR0aDogNixcbiAgICAgICAgbGVmdDogW2NwdVV0aWxpemF0aW9uTWV0cmljXSxcbiAgICAgICAgbGVmdEFubm90YXRpb25zOiBbY3B1VXRpbGl6YXRpb25BbGFybS50b0Fubm90YXRpb24oKV0sXG4gICAgICB9KSxcbiAgICAgIG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHtcbiAgICAgICAgdGl0bGU6IGBEQiBDb25uZWN0aW9ucy8ke2RiQ29ubmVjdGlvbnNNZXRyaWMucGVyaW9kLnRvTWludXRlcygpfW1pbmAsXG4gICAgICAgIHdpZHRoOiA2LFxuICAgICAgICBsZWZ0OiBbZGJDb25uZWN0aW9uc01ldHJpY10sXG4gICAgICAgIGxlZnRBbm5vdGF0aW9uczogW2RiQ29ubmVjdGlvbnNBbGFybS50b0Fubm90YXRpb24oKV0sXG4gICAgICB9KSxcbiAgICAgIG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHtcbiAgICAgICAgdGl0bGU6IGBEQiBSZXBsaWNhIExhZy8ke2RiUmVwbGljYUxhZ01ldHJpYy5wZXJpb2QudG9NaW51dGVzKCl9bWluYCxcbiAgICAgICAgd2lkdGg6IDYsXG4gICAgICAgIGxlZnQ6IFtkYlJlcGxpY2FMYWdNZXRyaWNdLFxuICAgICAgICBsZWZ0QW5ub3RhdGlvbnM6IFtkYlJlcGxpY2FMYWdBbGFybS50b0Fubm90YXRpb24oKV0sXG4gICAgICB9KSxcbiAgICAgIG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHtcbiAgICAgICAgdGl0bGU6IGBEQiBCdWZmZXJDYWNoZSBIaXQgUmF0aW8vJHtkYkJ1ZmZlckNhY2hlSGl0UmF0aW9NZXRyaWMucGVyaW9kLnRvTWludXRlcygpfW1pbmAsXG4gICAgICAgIHdpZHRoOiA2LFxuICAgICAgICBsZWZ0OiBbZGJCdWZmZXJDYWNoZUhpdFJhdGlvTWV0cmljXSxcbiAgICAgICAgbGVmdEFubm90YXRpb25zOiBbZGJCdWZmZXJDYWNoZUhpdFJhdGlvQWxhcm0udG9Bbm5vdGF0aW9uKCldLFxuICAgICAgfSksXG4gICAgKTtcbiAgICB0aGlzLndhdGNoZnVsLmFkZFdpZGdldHMoXG4gICAgICBuZXcgY2xvdWR3YXRjaC5HcmFwaFdpZGdldCh7XG4gICAgICAgIHRpdGxlOiAnUkRTIERNTCBPdmVydmlldycsXG4gICAgICAgIHdpZHRoOiAyNCxcbiAgICAgICAgbGVmdDogW2RiSW5zZXJ0VGhyb3VnaHB1dE1ldHJpYywgZGJVcGRhdGVUaHJvdWdocHV0TWV0cmljLCBkYlNlbGVjdFRocm91Z2hwdXRNZXRyaWMsIGRiRGVsZXRlVGhyb3VnaHB1dE1ldHJpY10sXG4gICAgICB9KSxcbiAgICApO1xuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVDcHVVdGlsaXphdGlvbk1vbml0b3IoY3B1TWF4aW11bVRocmVzaG9sZFBlcmNlbnQgPSA4MCkge1xuICAgIGNvbnN0IGNwdVV0aWxpemF0aW9uTWV0cmljID0gdGhpcy5tZXRyaWNzLm1ldHJpY0NwdVV0aWxpemF0aW9uKHRoaXMuY2x1c3Rlci5jbHVzdGVySWRlbnRpZmllcik7XG4gICAgY29uc3QgY3B1VXRpbGl6YXRpb25BbGFybSA9IGNwdVV0aWxpemF0aW9uTWV0cmljLmNyZWF0ZUFsYXJtKHRoaXMsICdDcHVVdGlsaXphdGlvbkFsYXJtJywge1xuICAgICAgYWxhcm1EZXNjcmlwdGlvbjogJ2NwdVV0aWxpemF0aW9uQWxhcm0nLFxuICAgICAgdGhyZXNob2xkOiBjcHVNYXhpbXVtVGhyZXNob2xkUGVyY2VudCxcbiAgICAgIGNvbXBhcmlzb25PcGVyYXRvcjogY2xvdWR3YXRjaC5Db21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX1RIUkVTSE9MRCxcbiAgICAgIGV2YWx1YXRpb25QZXJpb2RzOiAzLFxuICAgIH0pO1xuICAgIHJldHVybiB7IGNwdVV0aWxpemF0aW9uTWV0cmljLCBjcHVVdGlsaXphdGlvbkFsYXJtIH07XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZURiQ29ubmVjdGlvbnNNb25pdG9yKGRiQ29ubmVjdGlvbnNNYXhpbXVtVGhyZXNob2xkID0gMCkge1xuICAgIGNvbnN0IGRiQ29ubmVjdGlvbnNNZXRyaWMgPSB0aGlzLm1ldHJpY3MubWV0cmljRGJDb25uZWN0aW9ucyh0aGlzLmNsdXN0ZXIuY2x1c3RlcklkZW50aWZpZXIpO1xuICAgIGNvbnN0IGRiQ29ubmVjdGlvbnNBbGFybSA9IGRiQ29ubmVjdGlvbnNNZXRyaWMuY3JlYXRlQWxhcm0odGhpcywgJ0RiQ29ubmVjdGlvbnNBbGFybScsIHtcbiAgICAgIGFsYXJtRGVzY3JpcHRpb246ICdkYkNvbm5lY3Rpb25zQWxhcm0nLFxuICAgICAgdGhyZXNob2xkOiBkYkNvbm5lY3Rpb25zTWF4aW11bVRocmVzaG9sZCxcbiAgICAgIGNvbXBhcmlzb25PcGVyYXRvcjogY2xvdWR3YXRjaC5Db21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX1RIUkVTSE9MRCxcbiAgICAgIGV2YWx1YXRpb25QZXJpb2RzOiAzLFxuICAgIH0pO1xuICAgIHJldHVybiB7IGRiQ29ubmVjdGlvbnNNZXRyaWMsIGRiQ29ubmVjdGlvbnNBbGFybSB9O1xuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVEYlJlcGxpY2FMYWdNb25pdG9yKGRiUmVwbGljYUxhZ01heGltdW1UaHJlc2hvbGQgPSAwKSB7XG4gICAgY29uc3QgZGJSZXBsaWNhTGFnTWV0cmljID0gdGhpcy5tZXRyaWNzLm1ldHJpY1JlcGxpY2FMYWcodGhpcy5jbHVzdGVyLmNsdXN0ZXJJZGVudGlmaWVyKTtcbiAgICBjb25zdCBkYlJlcGxpY2FMYWdBbGFybSA9IGRiUmVwbGljYUxhZ01ldHJpYy5jcmVhdGVBbGFybSh0aGlzLCAnRGJSZXBsaWNhTGFnQWxhcm0nLCB7XG4gICAgICBhbGFybURlc2NyaXB0aW9uOiAnZGJDb25uZWN0aW9uc0FsYXJtJyxcbiAgICAgIHRocmVzaG9sZDogZGJSZXBsaWNhTGFnTWF4aW11bVRocmVzaG9sZCxcbiAgICAgIGNvbXBhcmlzb25PcGVyYXRvcjogY2xvdWR3YXRjaC5Db21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX1RIUkVTSE9MRCxcbiAgICAgIGV2YWx1YXRpb25QZXJpb2RzOiAzLFxuICAgIH0pO1xuICAgIHJldHVybiB7IGRiUmVwbGljYUxhZ01ldHJpYywgZGJSZXBsaWNhTGFnQWxhcm0gfTtcbiAgfVxuXG4gIHByaXZhdGUgY3JlYXRlRGJCdWZmZXJDYWNoZU1vbml0b3IoZGJCdWZmZXJDYWNoZU1pbmltdW1UaHJlc2hvbGQgPSAwKSB7XG4gICAgY29uc3QgZGJCdWZmZXJDYWNoZUhpdFJhdGlvTWV0cmljID0gdGhpcy5tZXRyaWNzLm1ldHJpY0J1ZmZlckNhY2hlSGl0UmF0aW8odGhpcy5jbHVzdGVyLmNsdXN0ZXJJZGVudGlmaWVyKTtcbiAgICBjb25zdCBkYkJ1ZmZlckNhY2hlSGl0UmF0aW9BbGFybSA9IGRiQnVmZmVyQ2FjaGVIaXRSYXRpb01ldHJpYy5jcmVhdGVBbGFybSh0aGlzLCAnRGJCdWZmZXJDYWNoZUhpdFJhdGlvQWxhcm0nLCB7XG4gICAgICBhbGFybURlc2NyaXB0aW9uOiAnZGJDb25uZWN0aW9uc0FsYXJtJyxcbiAgICAgIHRocmVzaG9sZDogZGJCdWZmZXJDYWNoZU1pbmltdW1UaHJlc2hvbGQsXG4gICAgICBjb21wYXJpc29uT3BlcmF0b3I6IGNsb3Vkd2F0Y2guQ29tcGFyaXNvbk9wZXJhdG9yLkxFU1NfVEhBTl9USFJFU0hPTEQsXG4gICAgICBldmFsdWF0aW9uUGVyaW9kczogMyxcbiAgICB9KTtcbiAgICByZXR1cm4geyBkYkJ1ZmZlckNhY2hlSGl0UmF0aW9NZXRyaWMsIGRiQnVmZmVyQ2FjaGVIaXRSYXRpb0FsYXJtIH07XG4gIH1cbiAgcHJpdmF0ZSBjcmVhdGVEYkRtbFRocm91Z2hwdXRNb25pdG9yKGRiVGhyb3VnaHB1dE1heGltdW1UaHJlc2hvbGQgPSAwKSB7XG4gICAgLy8gQHRzLWlnbm9yZVxuICAgIGNvbnN0IEFsYXJtVGhyZXNob2xkID0gZGJUaHJvdWdocHV0TWF4aW11bVRocmVzaG9sZDtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWNzLm1ldHJpY0RtbFRocm91Z2hwdXQodGhpcy5jbHVzdGVyLmNsdXN0ZXJJZGVudGlmaWVyKTtcbiAgfVxufVxuXG5mdW5jdGlvbiBsaW5rRm9yUkRTKGNsdXN0ZXI6IHJkcy5EYXRhYmFzZUNsdXN0ZXIpIHtcbiAgcmV0dXJuIGBodHRwczovL2NvbnNvbGUuYXdzLmFtYXpvbi5jb20vcmRzL2hvbWU/cmVnaW9uPSR7Y2x1c3Rlci5zdGFjay5yZWdpb259I2RhdGFiYXNlOmlkPSR7Y2x1c3Rlci5jbHVzdGVySWRlbnRpZmllcn07aXMtY2x1c3Rlcj10cnVlYDtcbn1cblxuXG4iXX0=