"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Watchful = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const cloudwatch_actions = require("aws-cdk-lib/aws-cloudwatch-actions");
const sns = require("aws-cdk-lib/aws-sns");
const sns_subscriptions = require("aws-cdk-lib/aws-sns-subscriptions");
const constructs_1 = require("constructs");
const api_gateway_1 = require("./api-gateway");
const aspect_1 = require("./aspect");
const dynamodb_1 = require("./dynamodb");
const ecs_1 = require("./ecs");
const lambda_1 = require("./lambda");
const rds_aurora_1 = require("./rds-aurora");
const state_machine_1 = require("./state-machine");
const section_1 = require("./widget/section");
class Watchful extends constructs_1.Construct {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.createdAlarmCount = 0;
        this.alarmActions = [
            ...(props.alarmActionArns ?? []).map((alarmActionArn) => ({ bind: () => ({ alarmActionArn }) })),
            ...(props.alarmActions ?? []),
        ];
        if ((props.alarmEmail || props.alarmSqs) && !props.alarmSns) {
            this.alarmTopic = new sns.Topic(this, 'AlarmTopic', { displayName: 'Watchful Alarms' });
        }
        if (props.alarmSns) {
            this.alarmTopic = props.alarmSns;
        }
        if (props.alarmEmail && this.alarmTopic) {
            this.alarmTopic.addSubscription(new sns_subscriptions.EmailSubscription(props.alarmEmail));
        }
        if (props.alarmSqs && this.alarmTopic) {
            this.alarmTopic.addSubscription(new sns_subscriptions.SqsSubscription(
            // sqs.Queue.fromQueueArn(this, 'AlarmQueue', props.alarmSqs)
            props.alarmSqs));
        }
        if (props.dashboard === false && props.dashboardName) {
            throw new Error('Dashboard name is provided but dashboard creation is disabled');
        }
        if (props.dashboard !== false) {
            this.dash = new cloudwatch.Dashboard(this, 'Dashboard', { dashboardName: props.dashboardName });
            new aws_cdk_lib_1.CfnOutput(this, 'WatchfulDashboard', {
                value: linkForDashboard(this.dash),
            });
        }
    }
    addWidgets(...widgets) {
        this.dash?.addWidgets(...widgets);
    }
    addAlarm(alarm) {
        const alarmWithAction = hasAlarmAction(alarm) ? alarm : new cloudwatch.CompositeAlarm(this, `Created Alarm ${this.createdAlarmCount++}`, {
            alarmRule: cloudwatch.AlarmRule.fromAlarm(alarm, cloudwatch.AlarmState.ALARM),
        });
        if (this.alarmTopic) {
            alarmWithAction.addAlarmAction(new cloudwatch_actions.SnsAction(this.alarmTopic));
        }
        alarmWithAction.addAlarmAction(...this.alarmActions);
    }
    addSection(title, options = {}) {
        this.addWidgets(new section_1.SectionWidget({
            titleLevel: 1,
            titleMarkdown: title,
            quicklinks: options.links,
        }));
    }
    watchScope(scope, options) {
        const aspect = new aspect_1.WatchfulAspect(this, options);
        aws_cdk_lib_1.Aspects.of(scope).add(aspect);
    }
    watchDynamoTable(title, table, options = {}) {
        return new dynamodb_1.WatchDynamoTable(this, aws_cdk_lib_1.Names.uniqueId(table), {
            title,
            watchful: this,
            table,
            ...options,
        });
    }
    watchApiGateway(title, restApi, options = {}) {
        return new api_gateway_1.WatchApiGateway(this, aws_cdk_lib_1.Names.uniqueId(restApi), {
            title, watchful: this, restApi, ...options,
        });
    }
    watchLambdaFunction(title, fn, options = {}) {
        return new lambda_1.WatchLambdaFunction(this, aws_cdk_lib_1.Names.uniqueId(fn), {
            title, watchful: this, fn, ...options,
        });
    }
    watchStateMachine(title, stateMachine, options = {}) {
        return new state_machine_1.WatchStateMachine(this, aws_cdk_lib_1.Names.uniqueId(stateMachine), {
            title, watchful: this, stateMachine, ...options,
        });
    }
    watchRdsAuroraCluster(title, cluster, options = {}) {
        return new rds_aurora_1.WatchRdsAurora(this, aws_cdk_lib_1.Names.uniqueId(cluster), {
            title, watchful: this, cluster, ...options,
        });
    }
    watchFargateEcs(title, fargateService, targetGroup, options = {}) {
        return new ecs_1.WatchEcsService(this, aws_cdk_lib_1.Names.uniqueId(fargateService), {
            title, watchful: this, fargateService, targetGroup, ...options,
        });
    }
    watchEc2Ecs(title, ec2Service, targetGroup, options = {}) {
        return new ecs_1.WatchEcsService(this, aws_cdk_lib_1.Names.uniqueId(ec2Service), {
            title, watchful: this, ec2Service, targetGroup, ...options,
        });
    }
}
exports.Watchful = Watchful;
_a = JSII_RTTI_SYMBOL_1;
Watchful[_a] = { fqn: "cdk-watchful.Watchful", version: "0.6.417" };
function linkForDashboard(dashboard) {
    const cfnDashboard = dashboard.node.defaultChild;
    return `https://console.aws.amazon.com/cloudwatch/home?region=${dashboard.stack.region}#dashboards:name=${cfnDashboard.ref}`;
}
function hasAlarmAction(alarm) {
    return 'addAlarmAction' in alarm;
}
//# sourceMappingURL=data:application/json;base64,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