import typing

from elevenlabs.speech_to_text.client import SpeechToTextClient as AutogeneratedSpeechToTextClient
from elevenlabs.speech_to_text.client import AsyncSpeechToTextClient as AutogeneratedAsyncSpeechToTextClient
from elevenlabs.realtime import ScribeRealtime


class SpeechToTextClient(AutogeneratedSpeechToTextClient):
    """
    A client to handle ElevenLabs speech-to-text functionality.
    Extends the autogenerated client to include realtime transcription.
    """

    def __init__(self, **kwargs: typing.Any) -> None:
        super().__init__(**kwargs)
        self._realtime: typing.Optional[ScribeRealtime] = None

    @property
    def realtime(self) -> ScribeRealtime:
        """
        Access the realtime speech-to-text transcription API.

        Returns:
            ScribeRealtime instance for creating WebSocket connections

        Example:
            ```python
            from elevenlabs import ElevenLabs, RealtimeEvents, AudioFormat

            client = ElevenLabs(api_key="your-api-key")

            # URL-based streaming
            connection = await client.speech_to_text.realtime.connect({
                "url": "https://stream.example.com/audio.mp3"
            })

            connection.on(RealtimeEvents.PARTIAL_TRANSCRIPT, lambda data: print(data))

            # Manual audio chunks
            connection = await client.speech_to_text.realtime.connect({
                "audio_format": AudioFormat.PCM_16000,
                "sample_rate": 16000
            })
            ```
        """
        if not self._realtime:
            # Extract API key from client wrapper
            api_key = self._client_wrapper.get_headers().get("xi-api-key", "")
            base_url = self._client_wrapper.get_base_url()

            self._realtime = ScribeRealtime(api_key=api_key, base_url=base_url)
        return self._realtime


class AsyncSpeechToTextClient(AutogeneratedAsyncSpeechToTextClient):
    """
    An async client to handle ElevenLabs speech-to-text functionality.
    Extends the autogenerated async client to include realtime transcription.
    """

    def __init__(self, **kwargs: typing.Any) -> None:
        super().__init__(**kwargs)
        self._realtime: typing.Optional[ScribeRealtime] = None

    @property
    def realtime(self) -> ScribeRealtime:
        """
        Access the realtime speech-to-text transcription API.

        Returns:
            ScribeRealtime instance for creating WebSocket connections

        Example:
            ```python
            from elevenlabs import AsyncElevenLabs, RealtimeEvents, AudioFormat

            client = AsyncElevenLabs(api_key="your-api-key")

            # URL-based streaming
            connection = await client.speech_to_text.realtime.connect({
                "url": "https://stream.example.com/audio.mp3"
            })

            connection.on(RealtimeEvents.PARTIAL_TRANSCRIPT, lambda data: print(data))

            # Manual audio chunks
            connection = await client.speech_to_text.realtime.connect({
                "audio_format": AudioFormat.PCM_16000,
                "sample_rate": 16000
            })
            ```
        """
        if not self._realtime:
            # Extract API key from client wrapper
            api_key = self._client_wrapper.get_headers().get("xi-api-key", "")
            base_url = self._client_wrapper.get_base_url()

            self._realtime = ScribeRealtime(api_key=api_key, base_url=base_url)
        return self._realtime

