"""
DataFlow Engine

Main DataFlow class and database management.
"""

import inspect
import logging
import os
import time
from copy import deepcopy
from datetime import datetime
from typing import Any, Dict, List, Optional, Tuple, Type, Union

from kailash.runtime.local import LocalRuntime
from kailash.workflow.builder import WorkflowBuilder

from ..features.bulk import BulkOperations
from ..features.multi_tenant import MultiTenantManager
from ..features.transactions import TransactionManager
from ..migrations.auto_migration_system import AutoMigrationSystem
from ..migrations.schema_state_manager import SchemaStateManager
from ..utils.connection import ConnectionManager
from .config import DatabaseConfig, DataFlowConfig, MonitoringConfig, SecurityConfig
from .nodes import NodeGenerator
from .schema_cache import create_schema_cache  # ADR-001: Schema cache integration

logger = logging.getLogger(__name__)


class DataFlow:
    """Main DataFlow interface."""

    def __init__(
        self,
        database_url: Optional[str] = None,
        config: Optional[DataFlowConfig] = None,
        pool_size: Optional[
            int
        ] = None,  # Changed to Optional to detect when explicitly set
        pool_max_overflow: int = 30,
        pool_recycle: int = 3600,
        echo: bool = False,
        multi_tenant: bool = False,
        encryption_key: Optional[str] = None,
        audit_logging: bool = False,
        cache_enabled: bool = True,
        cache_ttl: int = 3600,
        monitoring: Optional[
            bool
        ] = None,  # Changed to Optional to detect when explicitly set
        slow_query_threshold: float = 1.0,
        debug: bool = False,
        migration_enabled: bool = True,
        auto_migrate: bool = True,  # NEW: Control auto-migration behavior
        existing_schema_mode: bool = False,  # NEW: Safe mode for existing DBs
        enable_model_persistence: bool = True,  # NEW: Enable persistent model registry
        tdd_mode: bool = False,  # NEW: Enable TDD mode for testing
        test_context: Optional[Any] = None,  # NEW: TDD test context
        migration_lock_timeout: int = 30,  # NEW: Migration lock timeout for concurrent safety
        **kwargs,
    ):
        """Initialize DataFlow.

        Args:
            database_url: Database connection URL (uses DATABASE_URL env var if not provided)
            config: DataFlowConfig object with detailed settings
            pool_size: Connection pool size (default 20)
            pool_max_overflow: Maximum overflow connections
            pool_recycle: Time to recycle connections
            echo: Enable SQL logging
            multi_tenant: Enable multi-tenant mode
            encryption_key: Encryption key for sensitive data
            audit_logging: Enable audit logging
            cache_enabled: Enable query caching
            cache_ttl: Cache time-to-live
            monitoring: Enable performance monitoring
            migration_enabled: Enable automatic database migrations (default True)
            auto_migrate: Automatically run migrations on model registration (default True)
            existing_schema_mode: Safe mode for existing databases - validates compatibility (default False)
            **kwargs: Additional configuration options

        Note:
            ARCHITECTURAL FIX: As of this version, migrations are deferred during model registration
            to prevent "Event loop is closed" errors. For proper migration execution:

            # Option 1: Explicit async initialization (recommended)
            db = DataFlow(database_url="sqlite:///app.db")

            @db.model
            class User:
                name: str

            await db.initialize_deferred_migrations()  # Execute migrations async

            # Option 2: Automatic initialization (backward compatibility)
            db.ensure_migrations_initialized()  # Handles event loop properly
        """
        if config:
            # Use the provided config as base but allow kwargs to override
            self.config = deepcopy(config)
            # Override config attributes with kwargs
            if debug is not None:
                self.config.debug = debug
            if "batch_size" in kwargs:
                self.config.batch_size = kwargs["batch_size"]
            if pool_size is not None:
                self.config.pool_size = pool_size
            if pool_max_overflow is not None:
                self.config.max_overflow = pool_max_overflow
            if pool_recycle is not None:
                self.config.pool_recycle = pool_recycle
            if echo is not None:
                self.config.echo = echo
            if monitoring is not None:
                self.config.monitoring = monitoring
            if cache_enabled is not None:
                self.config.enable_query_cache = cache_enabled
            if cache_ttl is not None:
                self.config.cache_ttl = cache_ttl
            if slow_query_threshold is not None:
                self.config.slow_query_threshold = slow_query_threshold
            # ADR-001: Override schema cache configuration from kwargs
            if "schema_cache_enabled" in kwargs:
                self.config.migration.schema_cache_enabled = kwargs[
                    "schema_cache_enabled"
                ]
            if "schema_cache_ttl" in kwargs:
                self.config.migration.schema_cache_ttl = kwargs["schema_cache_ttl"]
            if "schema_cache_max_size" in kwargs:
                self.config.migration.schema_cache_max_size = kwargs[
                    "schema_cache_max_size"
                ]
        else:
            # Validate database_url if provided
            if database_url and not self._is_valid_database_url(database_url):
                raise ValueError(f"Invalid database URL: {database_url}")
            # Create config from environment or parameters
            if database_url is None and all(
                param is None
                for param in [
                    pool_size,
                    pool_max_overflow,
                    pool_recycle,
                    echo,
                    multi_tenant,
                    encryption_key,
                    audit_logging,
                    cache_enabled,
                    cache_ttl,
                    monitoring,
                ]
            ):
                # Zero-config mode - use from_env
                self.config = DataFlowConfig.from_env()
            else:
                # Create structured config from individual parameters
                database_config = DatabaseConfig(
                    url=database_url,
                    pool_size=(
                        pool_size if pool_size is not None else 20
                    ),  # Provide default
                    max_overflow=pool_max_overflow,
                    pool_recycle=pool_recycle,
                    echo=echo,
                )

                monitoring_config = MonitoringConfig(
                    enabled=(
                        monitoring if monitoring is not None else False
                    ),  # Provide default
                    slow_query_threshold=slow_query_threshold,
                )

                security_config = SecurityConfig(
                    multi_tenant=multi_tenant,
                    encrypt_at_rest=encryption_key is not None,
                    audit_enabled=audit_logging,
                )

                # Prepare config parameters
                config_params = {
                    "database": database_config,
                    "monitoring": monitoring_config,
                    "security": security_config,
                    "enable_query_cache": cache_enabled,
                    "cache_ttl": cache_ttl,
                }

                # Add direct parameters that should be passed through
                config_params["debug"] = debug
                if "batch_size" in kwargs:
                    config_params["batch_size"] = kwargs["batch_size"]
                if "cache_max_size" in kwargs:
                    config_params["cache_max_size"] = kwargs["cache_max_size"]
                if "max_retries" in kwargs:
                    config_params["max_retries"] = kwargs["max_retries"]
                if "encryption_enabled" in kwargs:
                    config_params["encryption_enabled"] = kwargs["encryption_enabled"]

                # ADR-001: Pass schema cache configuration kwargs
                if "schema_cache_enabled" in kwargs:
                    config_params["schema_cache_enabled"] = kwargs[
                        "schema_cache_enabled"
                    ]
                if "schema_cache_ttl" in kwargs:
                    config_params["schema_cache_ttl"] = kwargs["schema_cache_ttl"]
                if "schema_cache_max_size" in kwargs:
                    config_params["schema_cache_max_size"] = kwargs[
                        "schema_cache_max_size"
                    ]

                self.config = DataFlowConfig(**config_params)

        # Validate configuration
        if hasattr(self.config, "validate"):
            issues = self.config.validate()
            if issues:
                logger.warning(f"Configuration issues detected: {issues}")

        self._models = {}
        self._registered_models = {}  # Track registered models for compatibility
        self._model_fields = {}  # Store model field information
        self._nodes = {}  # Store generated nodes for testing
        self._tenant_context = None if not self.config.security.multi_tenant else {}

        # Cache AsyncSQLDatabaseNode instances to enable connection pooling across workflows
        # Critical for SQLite :memory: databases which need to share the same connection
        # See: ROOT_CAUSE_ANALYSIS.md in reports/issues/database-url-inheritance/
        self._async_sql_node_cache = {}  # Keyed by database_type

        # Store migration control parameters
        self._auto_migrate = auto_migrate
        self._migration_enabled = migration_enabled
        self._existing_schema_mode = existing_schema_mode
        self._migration_lock_timeout = max(
            1, migration_lock_timeout
        )  # Ensure minimum 1 second

        # ADR-001: Store kwargs for schema cache configuration (needed to detect explicit False)
        self._init_kwargs = kwargs

        # ARCHITECTURAL FIX: Deferred migration queue
        # This solves the "Event loop is closed" issue by separating
        # synchronous model registration from async migration execution
        # Removed deferred migrations - tables now created lazily when first accessed
        # Removed migration tracking - tables are now created lazily

        # Initialize TDD mode first (needed by NodeGenerator and _initialize_database)
        self._tdd_mode = tdd_mode or os.environ.get(
            "DATAFLOW_TDD_MODE", "false"
        ).lower() in (
            "true",
            "yes",
            "1",
            "on",
        )
        self._test_context = test_context
        if self._tdd_mode:
            self._initialize_tdd_mode()

        # Register specialized DataFlow nodes
        self._register_specialized_nodes()

        # Initialize feature modules (NodeGenerator now gets TDD context)
        self._node_generator = NodeGenerator(self)
        self._bulk_operations = BulkOperations(self)
        self._transaction_manager = TransactionManager(self)
        self._connection_manager = ConnectionManager(self)

        if self.config.security.multi_tenant:
            self._multi_tenant_manager = MultiTenantManager(self)
        else:
            self._multi_tenant_manager = None

        # Initialize model registry for multi-application support
        from .model_registry import ModelRegistry

        self._model_registry = ModelRegistry(self)
        self._enable_model_persistence = enable_model_persistence

        # Initialize cache integration if enabled
        self._cache_integration = None
        if self.config.enable_query_cache:
            self._initialize_cache_integration()

        # Initialize migration system if enabled
        self._migration_system = None
        self._schema_state_manager = None
        # Skip migration system initialization if using existing schema without auto-migration
        # existing_schema_mode=True means "use the existing database schema as-is"
        # We only skip if BOTH existing_schema_mode=True AND auto_migrate=False
        if (
            migration_enabled
            and not (self._existing_schema_mode and not self._auto_migrate)
            and not os.environ.get("DATAFLOW_DISABLE_MIGRATIONS", "").lower() == "true"
        ):
            self._initialize_migration_system()
            self._initialize_schema_state_manager()

        # Initialize schema cache (ADR-001)
        # Only enable cache if auto_migrate is enabled (no benefit otherwise)
        # CRITICAL: Respect explicit schema_cache_enabled=False from kwargs
        schema_cache_enabled = (
            self.config.migration.schema_cache_enabled
            and self._auto_migrate
            and self._init_kwargs.get("schema_cache_enabled", True) is not False
        )
        self._schema_cache = create_schema_cache(
            enabled=schema_cache_enabled,
            ttl_seconds=self.config.migration.schema_cache_ttl,
            max_cache_size=self.config.migration.schema_cache_max_size,
            enable_schema_validation=self.config.migration.schema_cache_validation,
            max_failure_count=self.config.migration.migration_max_failures,
            failure_backoff_seconds=self.config.migration.migration_failure_backoff,
        )
        logger.info(
            f"Schema cache initialized: enabled={schema_cache_enabled}, "
            f"ttl={self.config.migration.schema_cache_ttl}s, "
            f"max_size={self.config.migration.schema_cache_max_size}"
        )

        self._initialize_database()

        # Sync models from registry if persistence is enabled
        if self._enable_model_persistence and hasattr(self, "_model_registry"):
            self._sync_models_from_registry()

    async def initialize(self) -> bool:
        """Initialize DataFlow asynchronously.

        This method performs async initialization tasks that cannot be done in __init__.
        It is idempotent and safe to call multiple times.

        Returns:
            bool: True if initialization successful, False otherwise
        """
        try:
            # Validate database connectivity
            if not await self._validate_database_connection():
                logger.error("Database connection validation failed")
                return False

            # Initialize migration system components if enabled and not already done
            if self._migration_enabled and self._migration_system is not None:
                # Ensure migration table exists if we have migration system
                if hasattr(self._migration_system, "_ensure_migration_table"):
                    try:
                        await self._migration_system._ensure_migration_table()
                        logger.info("Migration table verification completed")
                    except Exception as e:
                        logger.warning(f"Migration table setup encountered issue: {e}")
                        # Don't fail initialization for migration table issues in existing_schema_mode
                        if not self._existing_schema_mode:
                            return False

            # Initialize schema state manager if available
            if self._schema_state_manager is not None:
                try:
                    # Schema state manager initialization (if needed)
                    # In existing_schema_mode, this should be very fast
                    logger.info("Schema state manager verified")
                except Exception as e:
                    logger.warning(f"Schema state manager issue: {e}")
                    # Don't fail initialization for schema state issues in existing_schema_mode
                    if not self._existing_schema_mode:
                        return False

            # Verify connection pool is working
            if hasattr(self._connection_manager, "initialize_pool"):
                self._connection_manager.initialize_pool()

            logger.info("DataFlow initialization completed successfully")
            return True

        except Exception as e:
            logger.error(f"DataFlow initialization failed: {e}")
            return False

    async def _validate_database_connection(self) -> bool:
        """Validate that database connection is working.

        Returns:
            bool: True if connection is valid, False otherwise
        """
        try:
            # Get a test connection
            connection = self._get_database_connection()
            if connection is None:
                return False

            # Try a simple query to validate connection
            try:
                if hasattr(connection, "execute"):
                    # For direct database connections
                    cursor = connection.cursor()
                    cursor.execute("SELECT 1")
                    result = cursor.fetchone()
                    cursor.close()
                    connection.close()
                    return result is not None
                else:
                    # For connection wrappers
                    return True
            except Exception as query_error:
                logger.debug(f"Database query test failed: {query_error}")
                if hasattr(connection, "close"):
                    connection.close()
                # In existing_schema_mode, be more lenient with connection issues
                return self._existing_schema_mode

        except Exception as e:
            logger.debug(f"Database connection validation error: {e}")
            # In existing_schema_mode, be more lenient with connection issues
            return self._existing_schema_mode

    def _initialize_cache_integration(self):
        """Initialize cache integration components."""
        try:
            from ..cache import (
                CacheConfig,
                CacheInvalidator,
                CacheKeyGenerator,
                RedisCacheManager,
                create_cache_integration,
            )

            # Create cache configuration
            cache_config = CacheConfig(
                host=getattr(self.config, "cache_host", "localhost"),
                port=getattr(self.config, "cache_port", 6379),
                db=getattr(self.config, "cache_db", 0),
                default_ttl=getattr(self.config, "cache_ttl", 300),
                key_prefix=getattr(self.config, "cache_key_prefix", "dataflow"),
            )

            # Create cache manager
            cache_manager = RedisCacheManager(cache_config)

            # Create key generator
            key_generator = CacheKeyGenerator(
                prefix=cache_config.key_prefix,
                namespace=getattr(self.config, "cache_namespace", None),
            )

            # Create cache invalidator
            invalidator = CacheInvalidator(cache_manager)

            # Create cache integration
            self._cache_integration = create_cache_integration(
                cache_manager, key_generator, invalidator
            )

            logger.info("Cache integration initialized successfully")

        except ImportError:
            logger.warning("Redis not available, cache integration disabled")
        except Exception as e:
            logger.error(f"Failed to initialize cache integration: {e}")
            self._cache_integration = None

    def _initialize_migration_system(self):
        """Initialize the auto-migration system for PostgreSQL only."""
        try:
            # Get real PostgreSQL database connection (async-compatible)
            connection = self._get_async_sql_connection()

            # Alpha release: PostgreSQL only
            database_url = self.config.database.url or ":memory:"
            if "postgresql" in database_url or "postgres" in database_url:
                dialect = "postgresql"
            elif "sqlite" in database_url or database_url == ":memory:":
                dialect = "sqlite"
                # SQLite is fully supported for production with enterprise adapter
            else:
                dialect = "unknown"
                logger.warning(f"Unsupported database dialect in URL: {database_url}")

            # Initialize AutoMigrationSystem with async workflow pattern and lock manager integration
            self._migration_system = AutoMigrationSystem(
                connection_string=self.config.database.get_connection_url(
                    self.config.environment
                ),
                dialect=dialect,
                migrations_dir="migrations",
                dataflow_instance=self,  # Pass DataFlow instance for lock manager integration
                lock_timeout=self._migration_lock_timeout,
            )

            logger.info(f"Migration system initialized successfully for {dialect}")

        except Exception as e:
            logger.error(f"Failed to initialize migration system: {e}")
            self._migration_system = None

    def _initialize_schema_state_manager(self):
        """Initialize the PostgreSQL-optimized schema state management system."""
        try:
            # Get real PostgreSQL database connection (async-compatible)
            connection = self._get_async_sql_connection()

            # Get cache configuration from DataFlow config
            cache_ttl = getattr(
                self.config, "schema_cache_ttl", 300
            )  # 5 minutes default
            cache_max_size = getattr(
                self.config, "schema_cache_max_size", 100
            )  # 100 schemas default

            # Initialize SchemaStateManager with DataFlow instance for WorkflowBuilder pattern
            self._schema_state_manager = SchemaStateManager(
                dataflow_instance=self,
                cache_ttl=cache_ttl,
                cache_max_size=cache_max_size,
            )

            logger.info(
                "PostgreSQL schema state management system initialized successfully"
            )

        except Exception as e:
            logger.error(f"Failed to initialize schema state management system: {e}")
            self._schema_state_manager = None

    def _initialize_tdd_mode(self):
        """Initialize TDD mode configuration."""
        try:
            # Import TDD support if available
            from ..testing.tdd_support import get_test_context, is_tdd_mode

            if self._test_context:
                # Use provided test context
                logger.debug(
                    f"DataFlow using provided TDD test context: {self._test_context.test_id}"
                )
            elif is_tdd_mode():
                # Get current test context from TDD infrastructure
                self._test_context = get_test_context()
                if self._test_context:
                    logger.debug(
                        f"DataFlow using global TDD test context: {self._test_context.test_id}"
                    )
                else:
                    logger.warning("TDD mode enabled but no test context available")

            # Configure for TDD performance
            if self._test_context:
                # Override connection manager for TDD mode
                self._configure_tdd_connection_manager()

                # Disable expensive operations in TDD mode
                self._tdd_optimizations_enabled = True

                logger.info(
                    f"DataFlow TDD mode initialized for test {self._test_context.test_id}"
                )

        except ImportError:
            logger.warning("TDD mode requested but TDD support not available")
            self._tdd_mode = False
        except Exception as e:
            logger.error(f"Failed to initialize TDD mode: {e}")
            self._tdd_mode = False

    def _configure_tdd_connection_manager(self):
        """Configure connection manager for TDD mode."""
        if self._test_context and hasattr(self._test_context, "connection"):
            # Store reference to TDD connection
            self._tdd_connection = self._test_context.connection
            logger.debug("DataFlow configured to use TDD connection")

    async def _get_async_database_connection(self):
        """Get database connection, TDD-aware."""
        if self._tdd_mode and hasattr(self, "_tdd_connection") and self._tdd_connection:
            # Return TDD connection for isolated testing
            return self._tdd_connection
        else:
            # Use regular connection manager
            return self._connection_manager.get_async_connection()

    def _initialize_database(self):
        """Initialize database connection and setup."""
        # Initialize connection pool (unless in TDD mode with existing connection)
        if not (self._tdd_mode and self._test_context):
            self._connection_manager.initialize_pool()

        # In a real implementation, this would:
        # 1. Create SQLAlchemy engine with all config options
        # 2. Setup connection pooling with overflow and recycle
        # 3. Initialize session factory
        # 4. Run migrations if needed
        # 5. Setup monitoring if enabled

    def model(self, cls: Type) -> Type:
        """Decorator to register a model with DataFlow.

        This decorator:
        1. Registers the model with DataFlow
        2. Generates CRUD workflow nodes
        3. Sets up database table mapping
        4. Configures indexes and constraints

        Example:
            @db.model
            class User:
                name: str
                email: str
                active: bool = True
        """
        # Validate model
        model_name = cls.__name__

        # Check for duplicate registration
        if model_name in self._models:
            raise ValueError(f"Model '{model_name}' is already registered")

        # Models without fields are allowed (they might define fields dynamically)

        # Extract model fields from annotations (including inherited)
        fields = {}

        # Collect fields from all parent classes (in method resolution order)
        for base_cls in reversed(cls.__mro__):
            if hasattr(base_cls, "__annotations__"):
                for field_name, field_type in base_cls.__annotations__.items():
                    # Skip private fields (starting with underscore)
                    if field_name.startswith("_"):
                        continue
                    fields[field_name] = {"type": field_type, "required": True}
                    # Check for defaults
                    if hasattr(base_cls, field_name):
                        fields[field_name]["default"] = getattr(base_cls, field_name)
                        fields[field_name]["required"] = False

        # Get model configuration if it exists
        config = {}
        if hasattr(cls, "__dataflow__"):
            config = getattr(cls, "__dataflow__", {})

        # Determine table name - check for __tablename__ override
        table_name = getattr(cls, "__tablename__", None)
        if not table_name:
            table_name = self._class_name_to_table_name(model_name)

        # Register model - store both class and structured info for compatibility
        model_info = {
            "class": cls,
            "fields": fields,
            "config": config,
            "table_name": table_name,
            "registered_at": datetime.now(),
        }

        self._models[model_name] = model_info  # Store structured info
        self._registered_models[model_name] = (
            cls  # Store class for backward compatibility
        )
        self._model_fields[model_name] = fields

        # Persist model in registry for multi-application support
        if self._enable_model_persistence and hasattr(self, "_model_registry"):
            try:
                self._model_registry.register_model(model_name, cls)
            except Exception as e:
                logger.warning(f"Failed to persist model {model_name}: {e}")

        # Auto-detect relationships from schema if available
        self._auto_detect_relationships(model_name, fields)

        # Generate workflow nodes (TDD-aware if in TDD mode)
        self._generate_crud_nodes(model_name, fields)
        self._generate_bulk_nodes(model_name, fields)

        # Add DataFlow attributes
        cls._dataflow = self
        cls._dataflow_meta = {
            "engine": self,
            "model_name": model_name,
            "fields": fields,
            "registered_at": datetime.now(),
        }
        cls._dataflow_config = getattr(cls, "__dataflow__", {})

        # Add multi-tenant support if enabled
        if self.config.security.multi_tenant:
            if "tenant_id" not in fields:
                fields["tenant_id"] = {"type": str, "required": False}
                cls.__annotations__["tenant_id"] = str

        # Add query_builder class method
        def query_builder(cls):
            """Create a QueryBuilder instance for this model."""
            from ..database.query_builder import create_query_builder

            table_name = self._class_name_to_table_name(cls.__name__)
            return create_query_builder(table_name, self.config.database.url)

        # Bind the method as a classmethod
        cls.query_builder = classmethod(query_builder)

        # Tables will be created lazily when first accessed via node operations
        # This eliminates the need for async operations during model registration
        logger.debug(
            f"Model '{model_name}' registered - table will be created lazily on first access"
        )

        return cls

    async def ensure_table_exists(self, model_name: str) -> bool:
        """
        Ensure the table for a model exists, creating it if necessary.

        With schema caching enabled (ADR-001), this method:
        1. Checks cache for table existence (0.001ms)
        2. If cached, returns immediately
        3. If not cached, runs full migration checking (~1500ms)
        4. Updates cache after successful check

        This is called lazily when a node first tries to access a table.

        Args:
            model_name: Name of the model

        Returns:
            bool: True if table exists or was created successfully
        """
        # ADR-001: Check schema cache first
        database_url = self.config.database.url or ":memory:"

        # Calculate schema checksum if validation enabled
        schema_checksum = None
        if self._schema_cache.enable_schema_validation:
            model_info = self._models.get(model_name)
            if model_info:
                schema_checksum = self._calculate_schema_checksum(model_info["fields"])

        # Check cache
        if self._schema_cache.is_table_ensured(
            model_name, database_url, schema_checksum
        ):
            logger.debug(f"Table '{model_name}' found in cache, skipping check")
            return True

        # Early exit conditions (cache these too)
        if not self._auto_migrate or self._existing_schema_mode:
            logger.debug(
                f"Skipping table creation for '{model_name}' "
                f"(auto_migrate={self._auto_migrate}, "
                f"existing_schema_mode={self._existing_schema_mode})"
            )
            # Mark as ensured in cache (skip mode)
            self._schema_cache.mark_table_ensured(
                model_name, database_url, schema_checksum
            )
            return True

        # Get model info
        model_info = self._models.get(model_name)
        if not model_info:
            logger.error(f"Model '{model_name}' not found in registry")
            return False

        fields = model_info["fields"]

        try:
            # Detect database type and route appropriately
            if "postgresql" in database_url or "postgres" in database_url:
                logger.debug(f"Ensuring PostgreSQL table for model {model_name}")
                await self._execute_postgresql_schema_management_async(
                    model_name, fields
                )
            elif (
                "sqlite" in database_url
                or database_url == ":memory:"
                or database_url.endswith(".db")
            ):
                logger.debug(f"Ensuring SQLite table for model {model_name}")
                # For SQLite, use the migration system to ensure table exists
                if self._migration_system is not None:
                    await self._execute_sqlite_migration_system_async(
                        model_name, fields
                    )
                else:
                    logger.warning(
                        f"No migration system available for SQLite model '{model_name}'"
                    )
                    return False
            else:
                # Unknown database type - try PostgreSQL as fallback
                logger.warning(
                    f"Unknown database type for {database_url}, attempting PostgreSQL schema management"
                )
                await self._execute_postgresql_schema_management_async(
                    model_name, fields
                )

            # ADR-001: Mark as successfully ensured in cache
            self._schema_cache.mark_table_ensured(
                model_name, database_url, schema_checksum
            )

            logger.debug(f"Table for model '{model_name}' ensured successfully")
            return True

        except Exception as e:
            logger.error(f"Failed to ensure table exists for model '{model_name}': {e}")

            # ADR-001: Mark as failed in cache
            self._schema_cache.mark_table_failed(model_name, database_url, str(e))

            return False

    def _get_table_status(self, model_name: str) -> str:
        """
        Get the status of a table for a model.

        ADR-001: Now uses schema cache instead of placeholder.

        Returns:
            str: 'exists', 'needs_creation', or 'unknown'
        """
        database_url = self.config.database.url or ":memory:"

        if self._schema_cache.is_table_ensured(model_name, database_url):
            return "exists"
        else:
            return "needs_creation"

    @property
    def has_pending_migrations(self) -> bool:
        """Check if there are any models that might need table creation."""
        # With lazy table creation, there are no "pending" migrations
        # Tables are created on-demand when first accessed
        return False

    def ensure_migrations_initialized(self) -> bool:
        """
        BACKWARD COMPATIBILITY: Ensure migrations are initialized.

        With lazy table creation, no initialization is needed.
        Tables are created automatically when first accessed.

        Returns:
            bool: Always True since no initialization is required
        """
        # With lazy table creation, no need to initialize anything
        # Tables will be created automatically when first accessed
        return True

    async def _execute_sqlite_migration_system_async(
        self, model_name: str, fields: Dict[str, Any]
    ):
        """Execute SQLite migration system asynchronously - ARCHITECTURAL FIX."""

        table_name = self._class_name_to_table_name(model_name)

        # Build expected table schema from model fields
        dict_schema = {table_name: {"columns": self._convert_fields_to_columns(fields)}}

        # Convert to TableDefinition format expected by migration system
        target_schema = self._convert_dict_schema_to_table_definitions(dict_schema)

        try:
            # Execute migration directly in async context - no event loop issues!
            success, migrations = await self._migration_system.auto_migrate(
                target_schema=target_schema,
                dry_run=False,
                interactive=False,  # Non-interactive for automatic execution
                auto_confirm=True,  # Auto-confirm for seamless operation
            )

            if success:
                logger.info(
                    f"SQLite table '{table_name}' ready for model '{model_name}'"
                )
            else:
                logger.warning(
                    f"SQLite migration failed for model '{model_name}': {migrations}"
                )

        except Exception as e:
            logger.error(f"SQLite migration error for model '{model_name}': {e}")
            # Don't fail the entire process - table will be created on-demand

    async def _execute_postgresql_schema_management_async(
        self, model_name: str, fields: Dict[str, Any]
    ):
        """Execute PostgreSQL schema management asynchronously - ARCHITECTURAL FIX."""

        # Handle existing_schema_mode - skip all migration activities
        if self._existing_schema_mode:
            logger.info(
                f"existing_schema_mode=True enabled. Skipping PostgreSQL schema management for model '{model_name}'."
            )
            return

        # Use PostgreSQL-optimized schema state manager if available
        enhanced_success = False
        if self._schema_state_manager is not None:
            try:
                await self._execute_postgresql_enhanced_schema_management_async(
                    model_name, fields
                )
                enhanced_success = True
            except Exception as e:
                logger.warning(
                    f"Enhanced schema management failed for '{model_name}', falling back to migration system: {e}"
                )

        # Fall back to migration system if enhanced failed or unavailable
        if not enhanced_success and self._migration_system is not None:
            await self._execute_postgresql_migration_system_async(model_name, fields)

    async def _execute_postgresql_enhanced_schema_management_async(
        self, model_name: str, fields: Dict[str, Any]
    ):
        """Execute PostgreSQL enhanced schema management asynchronously - ARCHITECTURAL FIX."""

        from ..migrations.schema_state_manager import ModelSchema

        # Build model schema for the specific model being registered
        model_fields = {}
        table_name = self._class_name_to_table_name(model_name)

        for field_name, field_info in fields.items():
            model_fields[field_name] = {
                "type": field_info.get("type", str),
                "required": field_info.get("required", True),
                "default": field_info.get("default"),
            }

        # Use the correct ModelSchema format with tables
        model_schema = ModelSchema(
            tables={
                table_name: {"columns": self._convert_fields_to_columns(model_fields)}
            }
        )

        # The SchemaStateManager doesn't have register_model_schema_async method
        # Instead, use detect_and_plan_migrations for proper schema management
        connection_id = "default"  # Default connection identifier

        try:
            operations, safety = self._schema_state_manager.detect_and_plan_migrations(
                model_schema, connection_id
            )

            # If operations are needed and safe, we should apply them
            # For now, we'll let this fall back to migration system for actual execution
            if operations:
                logger.info(
                    f"Enhanced schema management detected {len(operations)} operations for '{model_name}', falling back to migration system for execution"
                )
                # Raise exception to trigger fallback to migration system
                raise Exception(
                    f"Enhanced schema management requires fallback to migration system for {len(operations)} operations"
                )
            else:
                logger.info(
                    f"PostgreSQL enhanced schema management completed for model '{model_name}' - no operations needed"
                )
        except Exception as e:
            logger.error(
                f"PostgreSQL enhanced schema management error for model '{model_name}': {e}"
            )
            raise  # Re-raise to trigger fallback

    async def _execute_postgresql_migration_system_async(
        self, model_name: str, fields: Dict[str, Any]
    ):
        """Execute PostgreSQL migration system asynchronously - ARCHITECTURAL FIX."""

        table_name = self._class_name_to_table_name(model_name)

        # Build expected table schema from model fields
        dict_schema = {table_name: {"columns": self._convert_fields_to_columns(fields)}}

        # Convert to TableDefinition format expected by migration system
        target_schema = self._convert_dict_schema_to_table_definitions(dict_schema)

        try:
            # Execute migration directly in async context - no event loop issues!
            success, migrations = await self._migration_system.auto_migrate(
                target_schema=target_schema,
                dry_run=False,
                interactive=False,  # Non-interactive for automatic execution
                auto_confirm=True,  # Auto-confirm for seamless operation
            )

            if success:
                logger.info(
                    f"PostgreSQL migration executed successfully for model '{model_name}'"
                )
                if migrations:
                    for migration in migrations:
                        logger.info(
                            f"Applied migration {migration.version} with {len(migration.operations)} operations"
                        )
            else:
                logger.warning(
                    f"PostgreSQL migration was not applied for model '{model_name}'"
                )

        except Exception as e:
            logger.error(f"PostgreSQL migration error for model '{model_name}': {e}")

    async def auto_migrate(
        self,
        dry_run: bool = False,
        interactive: bool = True,
        auto_confirm: bool = False,
        target_schema: Optional[Dict[str, Any]] = None,
    ) -> Tuple[bool, List[Any]]:
        """Run automatic database migration to match registered models.

        This method analyzes the difference between your registered models
        and the current database schema, then applies necessary changes.

        Args:
            dry_run: If True, show what would be changed without applying
            interactive: If True, ask for user confirmation before applying changes
            auto_confirm: If True, automatically confirm all changes (ignores interactive)
            target_schema: Optional specific schema to migrate to (uses registered models if None)

        Returns:
            Tuple of (success: bool, migrations: List[Any])

        Example:
            # Show what would change
            success, migrations = await db.auto_migrate(dry_run=True)

            # Apply changes with confirmation
            success, migrations = await db.auto_migrate()

            # Apply changes automatically (production)
            success, migrations = await db.auto_migrate(auto_confirm=True)
        """
        if self._migration_system is None:
            raise RuntimeError(
                "Auto-migration is not available. Migration system not initialized. "
                "Ensure migration_enabled=True when creating DataFlow instance."
            )

        # If no target schema provided, build it from registered models
        if target_schema is None:
            dict_schema = {}
            for model_name, model_info in self._models.items():
                table_name = model_info["table_name"]
                fields = model_info["fields"]
                dict_schema[table_name] = {
                    "columns": self._convert_fields_to_columns(fields)
                }

            # Convert dictionary schema to TableDefinition format
            target_schema = self._convert_dict_schema_to_table_definitions(dict_schema)

        # Call the migration system
        return await self._migration_system.auto_migrate(
            target_schema=target_schema,
            dry_run=dry_run,
            interactive=interactive,
            auto_confirm=auto_confirm,
        )

    def set_tenant_context(self, tenant_id: str):
        """Set the current tenant context for multi-tenant operations."""
        if self.config.security.multi_tenant:
            self._tenant_context = {"tenant_id": tenant_id}

    def get_models(self) -> Dict[str, Type]:
        """Get all registered models."""
        # Return just the classes for backward compatibility
        return {name: info["class"] for name, info in self._models.items()}

    def get_model_fields(self, model_name: str) -> Dict[str, Any]:
        """Get field information for a model."""
        return self._model_fields.get(model_name, {})

    def get_model_info(self, model_name: str) -> Optional[Dict[str, Any]]:
        """Get comprehensive model information.

        Args:
            model_name: Name of the model

        Returns:
            Dictionary with model information or None if model doesn't exist
        """
        if model_name not in self._models:
            return None

        # Return a copy of the stored model info
        return self._models[model_name].copy()

    def list_models(self) -> List[str]:
        """List all registered model names.

        Returns:
            List of model names
        """
        return list(self._models.keys())

    def _sync_models_from_registry(self):
        """Sync models from persistent registry on startup."""
        try:
            if not self._model_registry._initialized:
                self._model_registry.initialize()

            # Skip model discovery and sync during initialization to prevent excessive operations
            # Model sync can be manually triggered when needed via sync_models() on the registry
            # This prevents the auto-migration excessive migration bug where ALL models are processed
            # during every DataFlow initialization
            logger.debug(
                "Skipping model sync during initialization to prevent excessive database operations"
            )
            logger.debug(
                "Use db.get_model_registry().sync_models() to manually sync models when needed"
            )

        except Exception as e:
            logger.error(f"Failed to initialize model registry: {e}")
            # Continue without model sync - don't fail startup

    # Public API methods for model registry

    def get_model_registry(self):
        """Get the model registry instance for advanced operations.

        Returns:
            ModelRegistry: The model registry instance

        Example:
            >>> registry = db.get_model_registry()
            >>> issues = registry.validate_consistency()
        """
        if not self._enable_model_persistence:
            raise RuntimeError(
                "Model persistence is disabled for this DataFlow instance"
            )
        return self._model_registry

    def validate_model_consistency(self) -> Dict[str, List[str]]:
        """Validate model consistency across all applications.

        Returns:
            Dictionary mapping model names to list of consistency issues

        Example:
            >>> issues = db.validate_model_consistency()
            >>> if issues:
            ...     print("Model inconsistencies found:", issues)
        """
        if not self._enable_model_persistence:
            return {}
        return self._model_registry.validate_consistency()

    def get_model_history(self, model_name: str) -> List[Dict[str, Any]]:
        """Get version history for a specific model.

        Args:
            model_name: Name of the model

        Returns:
            List of version records with fields, options, timestamps

        Example:
            >>> history = db.get_model_history("User")
            >>> for version in history:
            ...     print(f"Version from {version['created_at']}")
        """
        if not self._enable_model_persistence:
            return []
        return self._model_registry.get_model_history(model_name)

    def sync_models(self, force: bool = False) -> Tuple[int, int]:
        """Manually sync models from the registry.

        Args:
            force: Force re-sync even if models already exist

        Returns:
            Tuple of (models_added, models_updated)

        Example:
            >>> added, updated = db.sync_models()
            >>> print(f"Synced {added} new models, updated {updated}")
        """
        if not self._enable_model_persistence:
            return 0, 0
        return self._model_registry.sync_models(force)

    def get_model_checksums(self) -> Dict[str, Dict[str, str]]:
        """Get model checksums for all registered models by application.

        Returns:
            Dictionary mapping model names to application checksums

        Example:
            >>> checksums = db.get_model_checksums()
            >>> print(checksums)
            # {'User': {'app1': 'abc123', 'app2': 'abc123'}}
        """
        if not self._enable_model_persistence:
            return {}

        checksums = {}
        for model_name in self.list_models():
            app_checksums = self._model_registry._get_model_checksums_by_app(model_name)
            if app_checksums:
                checksums[model_name] = app_checksums
        return checksums

    def get_generated_nodes(self, model_name: str) -> Optional[Dict[str, Any]]:
        """Get generated nodes for a model.

        Args:
            model_name: Name of the model

        Returns:
            Dictionary with generated nodes or None if model doesn't exist
        """
        if model_name not in self._models:
            return None

        # Return the nodes that would be generated for this model
        nodes = {}

        # CRUD operations
        nodes["create"] = f"{model_name}CreateNode"
        nodes["read"] = f"{model_name}ReadNode"
        nodes["update"] = f"{model_name}UpdateNode"
        nodes["delete"] = f"{model_name}DeleteNode"
        nodes["list"] = f"{model_name}ListNode"

        # Bulk operations
        nodes["bulk_create"] = f"{model_name}BulkCreateNode"
        nodes["bulk_update"] = f"{model_name}BulkUpdateNode"
        nodes["bulk_delete"] = f"{model_name}BulkDeleteNode"
        nodes["bulk_upsert"] = f"{model_name}BulkUpsertNode"

        return nodes

    def get_connection_pool(self):
        """Get the connection pool for testing.

        Warning:
            This method returns a MockConnectionPool for backward compatibility.
            In v0.7.0+, MockConnectionPool has been moved to tests.fixtures.mock_helpers.
            Consider using real connection pooling in production code.
        """
        # Import from test fixtures
        try:
            from tests.fixtures.mock_helpers import MockConnectionPool
        except ImportError:
            # Fallback for cases where tests module is not available
            import warnings

            warnings.warn(
                "MockConnectionPool could not be imported from tests.fixtures.mock_helpers. "
                "Using inline fallback. This is deprecated and will be removed in v0.8.0.",
                DeprecationWarning,
                stacklevel=2,
            )

            # Inline fallback implementation
            class MockConnectionPool:
                def __init__(self, connection_manager):
                    self.connection_manager = connection_manager
                    self.max_connections = connection_manager._connection_stats.get(
                        "pool_size", 10
                    )

                async def get_metrics(self):
                    return {
                        "connections_created": 1,
                        "connections_reused": 5,
                        "active_connections": 1,
                        "total_connections": self.max_connections,
                    }

                async def get_health_status(self):
                    return {
                        "status": "healthy",
                        "total_connections": self.max_connections,
                        "active_connections": 1,
                    }

        return MockConnectionPool(self._connection_manager)

    def get_connection_info(self) -> Dict[str, Any]:
        """Get connection information.

        Returns:
            Dictionary with connection details
        """
        return {
            "database_url": self.config.database.url or "sqlite:///:memory:",
            "pool_size": self.config.database.pool_size,
            "max_overflow": self.config.database.max_overflow,
            "pool_recycle": self.config.database.pool_recycle,
            "echo": self.config.database.echo,
            "environment": (
                self.config.environment.value
                if hasattr(self.config.environment, "value")
                else str(self.config.environment)
            ),
            "multi_tenant": self.config.security.multi_tenant,
            "monitoring_enabled": getattr(self.config, "monitoring_enabled", False),
        }

    # Public API for feature modules
    @property
    def bulk(self) -> BulkOperations:
        """Access bulk operations."""
        return self._bulk_operations

    @property
    def transactions(self) -> TransactionManager:
        """Access transaction manager."""
        return self._transaction_manager

    @property
    def connection(self) -> ConnectionManager:
        """Access connection manager."""
        return self._connection_manager

    @property
    def tenants(self) -> Optional[MultiTenantManager]:
        """Access multi-tenant manager (if enabled)."""
        return self._multi_tenant_manager

    @property
    def cache(self):
        """Access cache integration (if enabled)."""
        return self._cache_integration

    @property
    def schema_state_manager(self):
        """Access schema state management system (if enabled)."""
        return self._schema_state_manager

    def _inspect_database_schema(self) -> Dict[str, Any]:
        """Internal method to inspect database schema.

        Returns:
            Raw schema information from database inspection.
        """
        # WARNING: This method returns hardcoded mock data in the alpha release
        logger.warning(
            "_inspect_database_schema() returns mock data in DataFlow alpha release. "
            "It does NOT inspect your actual database schema. Use use_real_inspection=True "
            "in discover_schema() for experimental real database introspection."
        )

        # Return hardcoded mock schema for backward compatibility
        return {
            "users": {
                "columns": [
                    {
                        "name": "id",
                        "type": "integer",
                        "primary_key": True,
                        "nullable": False,
                    },
                    {"name": "name", "type": "varchar", "nullable": False},
                    {
                        "name": "email",
                        "type": "varchar",
                        "unique": True,
                        "nullable": False,
                    },
                    {"name": "created_at", "type": "timestamp", "nullable": False},
                ],
                "relationships": {
                    "orders": {"type": "has_many", "foreign_key": "user_id"}
                },
            },
            "orders": {
                "columns": [
                    {
                        "name": "id",
                        "type": "integer",
                        "primary_key": True,
                        "nullable": False,
                    },
                    {"name": "user_id", "type": "integer", "nullable": False},
                    {"name": "total", "type": "decimal", "nullable": False},
                    {"name": "status", "type": "varchar", "default": "pending"},
                ],
                "relationships": {
                    "user": {"type": "belongs_to", "foreign_key": "user_id"}
                },
                "foreign_keys": [
                    {
                        "column_name": "user_id",
                        "foreign_table_name": "users",
                        "foreign_column_name": "id",
                    }
                ],
            },
        }

    def _map_postgresql_type_to_python(self, pg_type: str) -> str:
        """Map PostgreSQL data types to Python types.

        Args:
            pg_type: PostgreSQL data type name

        Returns:
            Python type name as string
        """
        # Comprehensive PostgreSQL type mapping
        TYPE_MAPPING = {
            # Integer types
            "integer": "int",
            "bigint": "int",
            "smallint": "int",
            "serial": "int",
            "bigserial": "int",
            "smallserial": "int",
            # Floating point types
            "numeric": "float",
            "decimal": "float",
            "real": "float",
            "double precision": "float",
            "money": "float",
            # String types
            "character varying": "str",
            "varchar": "str",
            "character": "str",
            "char": "str",
            "text": "str",
            # Boolean
            "boolean": "bool",
            # Date/Time types
            "timestamp": "datetime",
            "timestamp without time zone": "datetime",
            "timestamp with time zone": "datetime",
            "timestamptz": "datetime",
            "date": "date",
            "time": "time",
            "time without time zone": "time",
            "time with time zone": "time",
            "timetz": "time",
            "interval": "timedelta",
            # JSON types
            "json": "dict",
            "jsonb": "dict",
            # Other types
            "uuid": "str",
            "bytea": "bytes",
            "array": "list",
            "inet": "str",
            "cidr": "str",
            "macaddr": "str",
            "tsvector": "str",
            "tsquery": "str",
            "xml": "str",
        }

        # Handle array types
        if pg_type.endswith("[]"):
            return "list"

        # Normalize type name
        normalized_type = pg_type.lower()

        # Return mapped type or default to str
        return TYPE_MAPPING.get(normalized_type, "str")

    async def _inspect_database_schema_real(self) -> Dict[str, Any]:
        """Actually inspect database schema using database-specific system catalogs.

        This method performs real database introspection using PostgreSQL's
        information_schema or SQLite's sqlite_master table.

        Returns:
            Dictionary containing actual database schema information

        Raises:
            ConnectionError: If database connection fails
            QueryError: If schema introspection queries fail
            NotImplementedError: For unsupported databases
        """
        database_url = self.config.database.url or ":memory:"

        # Check database type and route to appropriate inspector
        if "postgresql" in database_url or "postgres" in database_url:
            return await self._inspect_postgresql_schema_real(database_url)
        elif (
            "sqlite" in database_url
            or database_url == ":memory:"
            or database_url.endswith(".db")
        ):
            return await self._inspect_sqlite_schema_real(database_url)
        else:
            # Extract scheme from URL for better error message
            try:
                scheme = (
                    database_url.split("://")[0] if "://" in database_url else "unknown"
                )
            except:
                scheme = "unknown"
            raise NotImplementedError(
                f"Real schema discovery is only supported for PostgreSQL and SQLite in alpha release. "
                f"Database URL uses unsupported scheme: {scheme}"
            )

    async def _inspect_postgresql_schema_real(
        self, database_url: str
    ) -> Dict[str, Any]:
        """Inspect PostgreSQL database schema using information_schema.

        Args:
            database_url: PostgreSQL connection string

        Returns:
            Dictionary containing PostgreSQL schema information
        """

        try:
            # Get PostgreSQL adapter for real introspection
            from ..adapters.postgresql import PostgreSQLAdapter

            adapter = PostgreSQLAdapter(database_url)
            await adapter.create_connection_pool()

            schema = {}

            # Get all tables
            tables_query = adapter.get_tables_query()
            tables_result = await adapter.execute_query(tables_query)

            for table_row in tables_result:
                table_name = table_row["table_name"]

                # Get columns for this table
                columns_query = adapter.get_columns_query(table_name)
                columns_result = await adapter.execute_query(columns_query)

                columns = []
                for col in columns_result:
                    column_info = {
                        "name": col["column_name"],
                        "type": self._normalize_postgresql_type(col["data_type"]),
                        "nullable": col["is_nullable"] == "YES",
                        "primary_key": False,  # Will be updated below
                    }

                    if col["column_default"]:
                        column_info["default"] = col["column_default"]

                    if col["character_maximum_length"]:
                        column_info["max_length"] = col["character_maximum_length"]

                    columns.append(column_info)

                # Get primary keys
                pk_query = """
                    SELECT kcu.column_name
                    FROM information_schema.table_constraints tc
                    JOIN information_schema.key_column_usage kcu
                        ON tc.constraint_name = kcu.constraint_name
                        AND tc.table_schema = kcu.table_schema
                    WHERE tc.constraint_type = 'PRIMARY KEY'
                        AND tc.table_schema = 'public'
                        AND tc.table_name = $1
                """
                pk_result = await adapter.execute_query(pk_query, [table_name])
                pk_columns = {row["column_name"] for row in pk_result}

                # Update primary key flags
                for col in columns:
                    if col["name"] in pk_columns:
                        col["primary_key"] = True

                # Get foreign keys
                fk_query = """
                    SELECT
                        kcu.column_name,
                        ccu.table_name AS foreign_table_name,
                        ccu.column_name AS foreign_column_name,
                        tc.constraint_name
                    FROM information_schema.table_constraints AS tc
                    JOIN information_schema.key_column_usage AS kcu
                        ON tc.constraint_name = kcu.constraint_name
                        AND tc.table_schema = kcu.table_schema
                    JOIN information_schema.constraint_column_usage AS ccu
                        ON ccu.constraint_name = tc.constraint_name
                        AND ccu.table_schema = tc.table_schema
                    WHERE tc.constraint_type = 'FOREIGN KEY'
                        AND tc.table_schema = 'public'
                        AND tc.table_name = $1
                """
                fk_result = await adapter.execute_query(fk_query, [table_name])

                foreign_keys = []
                relationships = {}

                for fk in fk_result:
                    foreign_keys.append(
                        {
                            "column_name": fk["column_name"],
                            "foreign_table_name": fk["foreign_table_name"],
                            "foreign_column_name": fk["foreign_column_name"],
                            "constraint_name": fk["constraint_name"],
                        }
                    )

                    # Create belongs_to relationship
                    rel_name = self._foreign_key_to_relationship_name(fk["column_name"])
                    relationships[rel_name] = {
                        "type": "belongs_to",
                        "target_table": fk["foreign_table_name"],
                        "foreign_key": fk["column_name"],
                        "target_key": fk["foreign_column_name"],
                    }

                # Get indexes
                indexes_query = """
                    SELECT
                        indexname as index_name,
                        indexdef as index_definition
                    FROM pg_indexes
                    WHERE schemaname = 'public'
                        AND tablename = $1
                        AND indexname NOT LIKE '%_pkey'
                """
                indexes_result = await adapter.execute_query(
                    indexes_query, [table_name]
                )

                indexes = []
                for idx in indexes_result:
                    # Parse index definition to extract columns and uniqueness
                    index_def = idx["index_definition"]
                    is_unique = "UNIQUE" in index_def.upper()

                    indexes.append(
                        {
                            "name": idx["index_name"],
                            "unique": is_unique,
                            "definition": index_def,
                        }
                    )

                schema[table_name] = {
                    "columns": columns,
                    "foreign_keys": foreign_keys,
                    "relationships": relationships,
                    "indexes": indexes,
                }

            await adapter.close_connection_pool()

            # Add reverse has_many relationships
            self._add_reverse_relationships_real(schema)

            logger.info(f"Real schema discovery completed. Found {len(schema)} tables.")
            return schema

        except Exception as e:
            logger.error(f"PostgreSQL schema discovery failed: {e}")
            raise

    async def _inspect_sqlite_schema_real(self, database_url: str) -> Dict[str, Any]:
        """Inspect SQLite database schema using sqlite_master table.

        Args:
            database_url: SQLite connection string or ":memory:"

        Returns:
            Dictionary containing SQLite schema information

        Raises:
            NotImplementedError: For in-memory SQLite databases (schema discovery not supported)
        """
        # Check if this is a memory database
        if database_url == ":memory:" or "memory" in database_url.lower():
            raise NotImplementedError(
                "Schema discovery is not supported for in-memory SQLite databases. "
                "Only file-based SQLite databases support schema discovery."
            )

        try:
            # Get SQLite adapter for real introspection
            from ..adapters.sqlite import SQLiteAdapter

            adapter = SQLiteAdapter(database_url)
            await adapter.connect()

            schema = {}

            # Get all tables (excluding SQLite system tables and DataFlow tables)
            tables_query = """
                SELECT name FROM sqlite_master
                WHERE type='table'
                AND name NOT LIKE 'sqlite_%'
                AND name NOT LIKE 'dataflow_%'
                ORDER BY name
            """
            tables_result = await adapter.execute_query(tables_query)

            for table_row in tables_result:
                table_name = table_row["name"]

                # Get columns for this table using PRAGMA table_info
                columns_query = f"PRAGMA table_info({table_name})"
                columns_result = await adapter.execute_query(columns_query)

                columns = []
                for col in columns_result:
                    column_info = {
                        "name": col["name"],
                        "type": self._normalize_sqlite_type(col["type"]),
                        "nullable": not col["notnull"],
                        "primary_key": bool(col["pk"]),
                    }

                    if col["dflt_value"] is not None:
                        column_info["default"] = col["dflt_value"]

                    columns.append(column_info)

                # Get foreign keys using PRAGMA foreign_key_list
                fk_query = f"PRAGMA foreign_key_list({table_name})"
                fk_result = await adapter.execute_query(fk_query)

                foreign_keys = []
                relationships = {}

                for fk in fk_result:
                    foreign_keys.append(
                        {
                            "column_name": fk["from"],
                            "foreign_table_name": fk["table"],
                            "foreign_column_name": fk["to"],
                            "constraint_name": f"fk_{table_name}_{fk['from']}",
                        }
                    )

                    # Create belongs_to relationship
                    rel_name = self._foreign_key_to_relationship_name(fk["from"])
                    relationships[rel_name] = {
                        "type": "belongs_to",
                        "target_table": fk["table"],
                        "foreign_key": fk["from"],
                        "target_key": fk["to"],
                    }

                # Get indexes using PRAGMA index_list and index_info
                indexes_query = f"PRAGMA index_list({table_name})"
                indexes_result = await adapter.execute_query(indexes_query)

                indexes = []
                for idx in indexes_result:
                    # Skip auto-indexes (SQLite internal)
                    if idx["name"].startswith("sqlite_autoindex"):
                        continue

                    indexes.append(
                        {
                            "name": idx["name"],
                            "unique": bool(idx["unique"]),
                            "definition": f"INDEX {idx['name']} ON {table_name}",
                        }
                    )

                schema[table_name] = {
                    "columns": columns,
                    "foreign_keys": foreign_keys,
                    "relationships": relationships,
                    "indexes": indexes,
                }

            await adapter.disconnect()

            # Add reverse has_many relationships
            self._add_reverse_relationships_real(schema)

            logger.info(
                f"SQLite schema discovery completed. Found {len(schema)} tables."
            )
            return schema

        except Exception as e:
            logger.error(f"SQLite schema discovery failed: {e}")
            raise

    def _normalize_sqlite_type(self, sqlite_type: str) -> str:
        """Normalize SQLite data types to standard types."""
        if not sqlite_type:
            return "text"

        # SQLite type mapping to standard types
        type_mapping = {
            "integer": "integer",
            "int": "integer",
            "bigint": "integer",
            "smallint": "integer",
            "tinyint": "integer",
            "real": "float",
            "double": "float",
            "float": "float",
            "numeric": "decimal",
            "decimal": "decimal",
            "text": "text",
            "varchar": "varchar",
            "char": "char",
            "character": "char",
            "blob": "blob",
            "boolean": "boolean",
            "bool": "boolean",
            "date": "date",
            "datetime": "datetime",
            "timestamp": "timestamp",
            "time": "time",
        }

        # Normalize type name (remove parentheses and parameters)
        normalized_type = sqlite_type.lower().split("(")[0].strip()

        # Return mapped type or default to text (SQLite's default)
        return type_mapping.get(normalized_type, "text")

    def _normalize_postgresql_type(self, pg_type: str) -> str:
        """Normalize PostgreSQL data types to standard types."""
        type_mapping = {
            "character varying": "varchar",
            "character": "char",
            "timestamp without time zone": "timestamp",
            "timestamp with time zone": "timestamptz",
            "double precision": "float",
            "bigint": "integer",
            "smallint": "integer",
            "text": "text",
            "boolean": "boolean",
            "numeric": "decimal",
            "jsonb": "jsonb",
            "json": "json",
            "uuid": "uuid",
            "bytea": "bytea",
        }
        return type_mapping.get(pg_type.lower(), pg_type)

    def _add_reverse_relationships_real(self, schema: Dict[str, Any]) -> None:
        """Add reverse has_many relationships based on discovered foreign keys."""
        for table_name, table_info in schema.items():
            foreign_keys = table_info.get("foreign_keys", [])

            for fk in foreign_keys:
                target_table = fk["foreign_table_name"]
                if target_table in schema:
                    # Add has_many relationship to target table
                    rel_name = table_name  # Use plural table name
                    if "relationships" not in schema[target_table]:
                        schema[target_table]["relationships"] = {}

                    schema[target_table]["relationships"][rel_name] = {
                        "type": "has_many",
                        "target_table": table_name,
                        "foreign_key": fk["column_name"],
                        "target_key": fk["foreign_column_name"],
                    }

    def _inspect_table(self, table_name: str) -> Dict[str, Any]:
        """Inspect a specific table's schema.

        Args:
            table_name: Name of the table to inspect

        Returns:
            Table schema information including columns, keys, etc.
        """
        # This would contain table-specific inspection logic
        # For now, delegate to the full schema inspection
        schema = self._inspect_database_schema()
        return schema.get(table_name, {"columns": []})

    def discover_schema(self, use_real_inspection: bool = True) -> Dict[str, Any]:
        """Discover database schema and relationships.

        By default (v0.7.0+), this method performs real database introspection.
        Set use_real_inspection=False for mock data (backward compatibility).

        Args:
            use_real_inspection: If True (default), perform real PostgreSQL database introspection.
                                If False, return mock data for backward compatibility.

        Returns:
            Dictionary containing discovered tables, columns, relationships, and indexes.

        Raises:
            NotImplementedError: When use_real_inspection=True with non-PostgreSQL databases
            ConnectionError: When real inspection fails to connect to database
        """
        if use_real_inspection:
            logger.info("Starting REAL database schema discovery...")
            try:
                import asyncio
                import sys

                # Check if we're already in an event loop
                try:
                    loop = asyncio.get_running_loop()
                    # We're in an async context - use nest_asyncio for compatibility
                    try:
                        import nest_asyncio

                        nest_asyncio.apply()
                        discovered_schema = asyncio.run(
                            self._inspect_database_schema_real()
                        )
                    except ImportError:
                        # Fall back to sync_to_async pattern
                        import concurrent.futures

                        with concurrent.futures.ThreadPoolExecutor() as executor:
                            future = executor.submit(
                                asyncio.run, self._inspect_database_schema_real()
                            )
                            discovered_schema = future.result()
                except RuntimeError:
                    # No event loop running, safe to use asyncio.run()
                    discovered_schema = asyncio.run(
                        self._inspect_database_schema_real()
                    )

                return discovered_schema

            except NotImplementedError:
                # Re-raise NotImplementedError - these should not be caught
                raise
            except (ConnectionError, asyncio.TimeoutError, Exception) as e:
                # Import ConnectionError from adapters if available
                from ..adapters.exceptions import (
                    ConnectionError as DataFlowConnectionError,
                )

                # Check if it's a connection-related error
                if isinstance(
                    e, (DataFlowConnectionError, ConnectionError, asyncio.TimeoutError)
                ):
                    # Only catch connection-related errors for fallback
                    logger.error(
                        f"Real schema discovery failed due to connection error: {e}"
                    )
                    logger.warning("Falling back to mock schema data")
                    # Fall through to mock data below
                else:
                    # Log unexpected errors but don't fallback - let them bubble up
                    logger.error(f"Unexpected error during real schema discovery: {e}")
                    raise
        else:
            logger.warning(
                "discover_schema() is returning MOCK DATA in DataFlow alpha release. "
                "This does NOT reflect your actual database schema. "
                "Use discover_schema(use_real_inspection=True) for experimental real database introspection."
            )

        logger.info("Starting mock schema discovery...")

        # Check if we have custom table inspection (for mocking)
        # This allows tests to mock either approach
        if hasattr(self, "_custom_table_inspection"):
            # Use table-by-table inspection
            tables = self.show_tables()
            discovered_schema = {}
            for table in tables:
                discovered_schema[table] = self._inspect_table(table)
        else:
            # Get the full schema from internal inspection
            discovered_schema = self._inspect_database_schema()

        # Fall back to default schema if no tables found
        if not discovered_schema:
            discovered_schema = {
                "users": {
                    "columns": [
                        {
                            "name": "id",
                            "type": "integer",
                            "primary_key": True,
                            "nullable": False,
                        },
                        {"name": "name", "type": "varchar", "nullable": False},
                        {
                            "name": "email",
                            "type": "varchar",
                            "unique": True,
                            "nullable": False,
                        },
                        {
                            "name": "created_at",
                            "type": "timestamp",
                            "default": "CURRENT_TIMESTAMP",
                        },
                    ],
                    "relationships": {
                        "orders": {"type": "has_many", "foreign_key": "user_id"}
                    },
                    "indexes": [
                        {
                            "name": "users_email_idx",
                            "columns": ["email"],
                            "unique": True,
                        }
                    ],
                },
                "orders": {
                    "columns": [
                        {
                            "name": "id",
                            "type": "integer",
                            "primary_key": True,
                            "nullable": False,
                        },
                        {"name": "user_id", "type": "integer", "nullable": False},
                        {"name": "total", "type": "decimal", "nullable": False},
                        {"name": "status", "type": "varchar", "default": "pending"},
                    ],
                    "relationships": {
                        "user": {"type": "belongs_to", "foreign_key": "user_id"}
                    },
                    "foreign_keys": [
                        {
                            "column_name": "user_id",
                            "foreign_table_name": "users",
                            "foreign_column_name": "id",
                        }
                    ],
                },
            }

        logger.info(
            f"Schema discovery completed. Found {len(discovered_schema)} tables."
        )
        return discovered_schema

    def show_tables(self, use_real_inspection: bool = True) -> List[str]:
        """Show available tables in the database.

        By default (v0.7.0+), this method queries the actual database for table names.
        Set use_real_inspection=False for mock data (backward compatibility).

        Args:
            use_real_inspection: If True (default), query actual database for table names.
                                If False, return mock table names for backward compatibility.

        Returns:
            List of table names.
        """
        if use_real_inspection:
            try:
                schema = self.discover_schema(use_real_inspection=True)
                return list(schema.keys())
            except Exception as e:
                logger.error(f"Real table discovery failed: {e}")
                logger.warning("Falling back to mock table list")
        else:
            logger.warning(
                "show_tables() is returning MOCK TABLE NAMES. "
                "Use show_tables(use_real_inspection=True) for actual database tables."
            )

        # Get tables from mock schema inspection
        schema = self._inspect_database_schema()
        return list(schema.keys())

    def list_tables(self) -> List[str]:
        """Alias for show_tables to maintain compatibility.

        Returns:
            List of table names.
        """
        return self.show_tables()

    def scaffold(
        self, output_file: str = "models.py", use_real_inspection: bool = False
    ) -> Dict[str, Any]:
        """Generate Python model files from discovered schema.

        WARNING: In DataFlow alpha release, this method uses hardcoded mock schema
        by default. Generated models will NOT match your actual database unless
        use_real_inspection=True is specified.

        Args:
            output_file: Path to output file for generated models
            use_real_inspection: If True, generate models from actual database schema.
                                If False (default), generate from mock schema.

        Returns:
            Dictionary with generation results
        """
        if not use_real_inspection:
            logger.warning(
                "scaffold() is using MOCK SCHEMA DATA in DataFlow alpha release. "
                "Generated models will NOT match your actual database. "
                "Use scaffold(use_real_inspection=True) for models based on real database schema."
            )

        logger.info(f"Generating models to {output_file}...")

        schema = self.discover_schema(use_real_inspection=use_real_inspection)

        # Generate model file content
        lines = [
            '"""Auto-generated DataFlow models from database schema."""',
            "",
            "from dataflow import DataFlow",
            "from typing import Optional",
            "from datetime import datetime",
            "from decimal import Decimal",
            "",
            "# Initialize DataFlow instance",
            "db = DataFlow()",
            "",
        ]

        generated_models = []
        relationships_detected = 0

        for table_name, table_info in schema.items():
            # Convert table name to class name
            class_name = self._table_name_to_class_name(table_name)
            generated_models.append(class_name)

            lines.extend(
                [
                    "@db.model",
                    f"class {class_name}:",
                    f'    """Model for {table_name} table."""',
                ]
            )

            # Add fields
            for column in table_info.get("columns", []):
                field_name = column["name"]
                field_type = self._sql_type_to_python_type(column["type"])

                # Skip auto-generated fields
                if field_name in ["id", "created_at", "updated_at"] and column.get(
                    "primary_key"
                ):
                    continue

                type_annotation = (
                    field_type.__name__
                    if hasattr(field_type, "__name__")
                    else str(field_type)
                )

                if column.get("nullable", True) and not column.get("primary_key"):
                    type_annotation = f"Optional[{type_annotation}]"

                if "default" in column:
                    if column["default"] is None:
                        lines.append(f"    {field_name}: {type_annotation} = None")
                    elif isinstance(column["default"], str):
                        lines.append(
                            f'    {field_name}: {type_annotation} = "{column["default"]}"'
                        )
                    else:
                        lines.append(
                            f'    {field_name}: {type_annotation} = {column["default"]}'
                        )
                else:
                    lines.append(f"    {field_name}: {type_annotation}")

            # Add relationships
            for rel_name, rel_info in table_info.get("relationships", {}).items():
                relationships_detected += 1
                rel_type = rel_info["type"]
                if rel_type == "has_many":
                    lines.append(
                        f'    # {rel_name} = db.has_many("{rel_info.get("target_table", rel_name)}", "{rel_info["foreign_key"]}")'
                    )
                elif rel_type == "belongs_to":
                    lines.append(
                        f'    # {rel_name} = db.belongs_to("{rel_info.get("target_table", rel_name)}", "{rel_info["foreign_key"]}")'
                    )

            lines.append("")

        content = "\n".join(lines)

        # Write to file
        with open(output_file, "w") as f:
            f.write(content)

        result = {
            "generated_models": generated_models,
            "output_file": output_file,
            "relationships_detected": relationships_detected,
            "lines_generated": len(lines),
            "tables_processed": len(schema),
        }

        logger.info(
            f"Generated {len(generated_models)} models with {relationships_detected} relationships"
        )
        return result

    def register_schema_as_models(
        self, tables: Optional[List[str]] = None, use_real_inspection: bool = True
    ) -> Dict[str, Any]:
        """Register discovered database tables as DataFlow models dynamically.

        This method allows dynamic model registration from existing database schemas,
        enabling workflows to be built without @db.model decorators. Perfect for LLM
        agents and dynamic database discovery scenarios.

        Args:
            tables: Optional list of table names to register. If None, registers all discovered tables.
            use_real_inspection: If True, use real database introspection. If False, use mock data.

        Returns:
            Dictionary with registration results including:
            - registered_models: List of successfully registered model names
            - generated_nodes: Dict mapping model names to their generated node names
            - errors: List of any registration errors

        Example:
            >>> # Register all discovered tables as models
            >>> result = db.register_schema_as_models()
            >>> print(f"Registered {len(result['registered_models'])} models")
            >>>
            >>> # Use generated nodes in workflows
            >>> workflow = WorkflowBuilder()
            >>> user_nodes = result['generated_nodes']['User']
            >>> workflow.add_node(user_nodes['create'], "create_user", {...})
        """
        logger.info("Starting dynamic model registration from schema...")

        # Discover schema
        schema = self.discover_schema(use_real_inspection=use_real_inspection)

        # Filter tables if specified
        if tables:
            schema = {k: v for k, v in schema.items() if k in tables}

        # Skip DataFlow system tables
        system_tables = {
            "dataflow_migrations",
            "dataflow_model_registry",
            "dataflow_migration_history",
        }
        schema = {k: v for k, v in schema.items() if k not in system_tables}

        registered_models = []
        generated_nodes = {}
        errors = []

        for table_name, table_info in schema.items():
            try:
                # Convert table name to model name
                model_name = self._table_name_to_class_name(table_name)

                # Skip if model already registered
                if model_name in self._models:
                    logger.debug(f"Model {model_name} already registered, skipping")
                    continue

                # Extract fields from table columns
                fields = {}
                columns = table_info.get("columns", {})

                # Handle both list and dict formats for columns
                if isinstance(columns, list):
                    # List format from real inspection
                    for col in columns:
                        field_name = col["name"]
                        field_type = self._map_postgresql_type_to_python(col["type"])

                        # Convert type string to actual Python type
                        type_mapping = {
                            "str": str,
                            "int": int,
                            "float": float,
                            "bool": bool,
                            "datetime": datetime,
                            "date": datetime,
                            "time": datetime,
                            "dict": dict,
                            "list": list,
                            "bytes": bytes,
                        }
                        python_type = type_mapping.get(field_type, str)

                        field_info = {
                            "type": python_type,
                            "required": not col.get("nullable", True),
                            "primary_key": col.get("primary_key", False),
                        }

                        if "default" in col:
                            field_info["default"] = col["default"]
                            field_info["required"] = False

                        fields[field_name] = field_info

                elif isinstance(columns, dict):
                    # Dict format from schema inspection
                    for field_name, col_info in columns.items():
                        field_type = col_info.get("type", "str")

                        # Convert type string to actual Python type
                        type_mapping = {
                            "str": str,
                            "int": int,
                            "float": float,
                            "bool": bool,
                            "datetime": datetime,
                            "date": datetime,
                            "time": datetime,
                            "dict": dict,
                            "list": list,
                            "bytes": bytes,
                            "varchar": str,
                            "text": str,
                            "integer": int,
                            "bigint": int,
                            "boolean": bool,
                            "timestamp": datetime,
                            "decimal": float,
                            "numeric": float,
                            "json": dict,
                            "jsonb": dict,
                        }
                        python_type = type_mapping.get(field_type.lower(), str)

                        field_info = {
                            "type": python_type,
                            "required": not col_info.get("nullable", True),
                            "primary_key": col_info.get("primary_key", False),
                        }

                        if "default" in col_info:
                            field_info["default"] = col_info["default"]
                            field_info["required"] = False

                        fields[field_name] = field_info

                # Create dynamic model class
                model_attrs = {
                    "__name__": model_name,
                    "__module__": "__main__",
                    "__tablename__": table_name,
                    "__annotations__": {},
                }

                # Add field annotations
                for field_name, field_info in fields.items():
                    model_attrs["__annotations__"][field_name] = field_info["type"]
                    # Add default values if specified
                    if "default" in field_info and field_info["default"] is not None:
                        model_attrs[field_name] = field_info["default"]

                # Create the model class dynamically
                DynamicModel = type(model_name, (), model_attrs)

                # Register model (similar to @db.model decorator logic)
                model_info = {
                    "class": DynamicModel,
                    "fields": fields,
                    "config": {},
                    "table_name": table_name,
                    "registered_at": datetime.now(),
                    "dynamic": True,  # Flag to indicate dynamically registered
                }

                self._models[model_name] = model_info
                self._registered_models[model_name] = DynamicModel
                self._model_fields[model_name] = fields

                # Generate workflow nodes
                self._generate_crud_nodes(model_name, fields)
                self._generate_bulk_nodes(model_name, fields)

                # Add DataFlow attributes to dynamic class
                DynamicModel._dataflow = self
                DynamicModel._dataflow_meta = {
                    "engine": self,
                    "model_name": model_name,
                    "fields": fields,
                    "registered_at": datetime.now(),
                }

                # Persist in model registry if enabled
                if self._enable_model_persistence and hasattr(self, "_model_registry"):
                    try:
                        self._model_registry.register_model(model_name, DynamicModel)
                    except Exception as e:
                        logger.warning(
                            f"Failed to persist dynamic model {model_name}: {e}"
                        )

                # Collect generated node names
                generated_nodes[model_name] = self.get_generated_nodes(model_name)
                registered_models.append(model_name)

                logger.info(
                    f"Successfully registered dynamic model: {model_name} (table: {table_name})"
                )

            except Exception as e:
                error_msg = f"Failed to register model for table {table_name}: {str(e)}"
                logger.error(error_msg)
                errors.append(error_msg)

        result = {
            "registered_models": registered_models,
            "generated_nodes": generated_nodes,
            "errors": errors,
            "total_tables": len(schema),
            "success_count": len(registered_models),
            "error_count": len(errors),
        }

        logger.info(
            f"Dynamic model registration complete: {result['success_count']} models registered, "
            f"{result['error_count']} errors"
        )

        return result

    def reconstruct_models_from_registry(self) -> Dict[str, Any]:
        """Reconstruct and register models from the model registry.

        This method allows another DataFlow instance to discover models from the registry
        and reconstruct them locally, enabling workflow creation without the original
        @db.model decorated classes.

        Returns:
            Dictionary with reconstruction results including:
            - reconstructed_models: List of successfully reconstructed model names
            - generated_nodes: Dict mapping model names to their generated node names
            - errors: List of any reconstruction errors

        Example:
            >>> # In a new DataFlow instance or session
            >>> db = DataFlow(existing_schema_mode=True)
            >>> result = db.reconstruct_models_from_registry()
            >>>
            >>> # Now you can use the models in workflows
            >>> workflow = WorkflowBuilder()
            >>> user_nodes = result['generated_nodes']['User']
            >>> workflow.add_node(user_nodes['create'], "create_user", {...})
        """
        logger.info("Starting model reconstruction from registry...")

        if not self._enable_model_persistence:
            return {
                "reconstructed_models": [],
                "generated_nodes": {},
                "errors": ["Model persistence is disabled for this DataFlow instance"],
            }

        # Discover models from registry
        registry_models = self._model_registry.discover_models()

        reconstructed_models = []
        generated_nodes = {}
        errors = []

        for model_name, model_info in registry_models.items():
            try:
                # Skip if model already registered locally
                if model_name in self._models:
                    logger.debug(
                        f"Model {model_name} already registered locally, skipping"
                    )
                    continue

                # Extract model definition
                model_def = model_info.get("definition", {})
                fields = model_def.get("fields", {})

                # Convert stored field info to internal format
                internal_fields = {}
                for field_name, field_info in fields.items():
                    # Map stored type strings back to Python types
                    type_str = field_info.get("type", "str")
                    type_mapping = {
                        "str": str,
                        "int": int,
                        "float": float,
                        "bool": bool,
                        "datetime": datetime,
                        "date": datetime,
                        "time": datetime,
                        "dict": dict,
                        "list": list,
                        "bytes": bytes,
                        "NoneType": type(None),
                    }

                    # Handle module.type format (e.g., "datetime.datetime")
                    if "." in type_str:
                        type_str = type_str.split(".")[-1]

                    python_type = type_mapping.get(type_str, str)

                    internal_fields[field_name] = {
                        "type": python_type,
                        "required": field_info.get("required", True),
                        "primary_key": field_info.get("primary_key", False),
                    }

                    if "default" in field_info:
                        internal_fields[field_name]["default"] = field_info["default"]
                        internal_fields[field_name]["required"] = False

                # Get table name
                table_name = model_def.get(
                    "table_name", self._class_name_to_table_name(model_name)
                )

                # Create dynamic model class
                model_attrs = {
                    "__name__": model_name,
                    "__module__": "__main__",
                    "__tablename__": table_name,
                    "__annotations__": {},
                }

                # Add field annotations
                for field_name, field_info in internal_fields.items():
                    model_attrs["__annotations__"][field_name] = field_info["type"]
                    # Add default values if specified
                    if "default" in field_info and field_info["default"] is not None:
                        model_attrs[field_name] = field_info["default"]

                # Create the model class dynamically
                ReconstructedModel = type(model_name, (), model_attrs)

                # Register model locally
                local_model_info = {
                    "class": ReconstructedModel,
                    "fields": internal_fields,
                    "config": model_def.get("config", {}),
                    "table_name": table_name,
                    "registered_at": datetime.now(),
                    "reconstructed": True,  # Flag to indicate reconstructed from registry
                    "checksum": model_info.get("checksum"),
                }

                self._models[model_name] = local_model_info
                self._registered_models[model_name] = ReconstructedModel
                self._model_fields[model_name] = internal_fields

                # Generate workflow nodes
                self._generate_crud_nodes(model_name, internal_fields)
                self._generate_bulk_nodes(model_name, internal_fields)

                # Add DataFlow attributes to reconstructed class
                ReconstructedModel._dataflow = self
                ReconstructedModel._dataflow_meta = {
                    "engine": self,
                    "model_name": model_name,
                    "fields": internal_fields,
                    "registered_at": datetime.now(),
                    "reconstructed": True,
                }

                # Collect generated node names
                generated_nodes[model_name] = self.get_generated_nodes(model_name)
                reconstructed_models.append(model_name)

                logger.info(f"Successfully reconstructed model: {model_name}")

            except Exception as e:
                error_msg = f"Failed to reconstruct model {model_name}: {str(e)}"
                logger.error(error_msg)
                errors.append(error_msg)

        result = {
            "reconstructed_models": reconstructed_models,
            "generated_nodes": generated_nodes,
            "errors": errors,
            "total_registry_models": len(registry_models),
            "success_count": len(reconstructed_models),
            "error_count": len(errors),
        }

        logger.info(
            f"Model reconstruction complete: {result['success_count']} models reconstructed, "
            f"{result['error_count']} errors"
        )

        return result

    def _table_name_to_class_name(self, table_name: str) -> str:
        """Convert table name to Python class name."""
        # Remove underscores and capitalize each word
        words = table_name.split("_")
        class_name = "".join(word.capitalize() for word in words)
        # Remove 's' suffix for singular class names
        if class_name.endswith("s") and len(class_name) > 1:
            class_name = class_name[:-1]
        return class_name

    def _sql_type_to_python_type(self, sql_type: str):
        """Map SQL types to Python types."""
        # Remove parameters from SQL type (e.g., VARCHAR(255) -> VARCHAR)
        base_type = sql_type.split("(")[0].lower()

        type_mappings = {
            "integer": int,
            "bigint": int,
            "smallint": int,
            "serial": int,
            "bigserial": int,
            "varchar": str,
            "text": str,
            "char": str,
            "character": str,
            "numeric": float,
            "decimal": float,
            "real": float,
            "double precision": float,
            "money": float,
            "boolean": bool,
            "timestamp": datetime,
            "timestamptz": datetime,
            "date": datetime,
            "time": datetime,
            "json": dict,
            "jsonb": dict,
            "array": list,
        }
        python_type = type_mappings.get(base_type, str)

        # Special handling for decimal to return string representation
        if base_type == "decimal":
            return "Decimal"

        # Return string representation of type
        return python_type.__name__

    def _python_type_to_sql_type(
        self, python_type, database_type: str = "postgresql"
    ) -> str:
        """Map Python types to SQL types for different databases.

        Args:
            python_type: The Python type (e.g., int, str, datetime)
            database_type: Target database ('postgresql', 'mysql', 'sqlite')

        Returns:
            SQL type string appropriate for the target database
        """
        # Handle Optional types (Union[type, None])
        if hasattr(python_type, "__origin__") and python_type.__origin__ is Union:
            args = python_type.__args__
            if len(args) == 2 and type(None) in args:
                # This is Optional[SomeType], extract the actual type
                actual_type = args[0] if args[1] is type(None) else args[1]
                return self._python_type_to_sql_type(actual_type, database_type)

        # Database-specific type mappings
        type_mappings = {
            "postgresql": {
                int: "INTEGER",
                str: "TEXT",  # Use TEXT instead of VARCHAR(255) for PostgreSQL
                bool: "BOOLEAN",
                float: "REAL",
                datetime: "TIMESTAMP",
                dict: "JSONB",
                list: "JSONB",
                bytes: "BYTEA",
            },
            "mysql": {
                int: "INT",
                str: "VARCHAR(255)",
                bool: "TINYINT(1)",
                float: "DOUBLE",
                datetime: "DATETIME",
                dict: "JSON",
                list: "JSON",
                bytes: "BLOB",
            },
            "sqlite": {
                int: "INTEGER",
                str: "TEXT",
                bool: "INTEGER",  # SQLite doesn't have native boolean
                float: "REAL",
                datetime: "TEXT",  # SQLite stores datetime as text
                dict: "TEXT",  # Store JSON as text
                list: "TEXT",  # Store JSON as text
                bytes: "BLOB",
            },
        }

        mapping = type_mappings.get(database_type.lower(), type_mappings["postgresql"])
        return mapping.get(python_type, "TEXT")

    def _get_sql_column_definition(
        self,
        field_name: str,
        field_info: Dict[str, Any],
        database_type: str = "postgresql",
    ) -> str:
        """Generate SQL column definition from field information.

        Args:
            field_name: Name of the field/column
            field_info: Field metadata from model registration
            database_type: Target database type

        Returns:
            Complete SQL column definition string
        """
        python_type = field_info["type"]
        sql_type = self._python_type_to_sql_type(python_type, database_type)

        # Start building column definition
        definition_parts = [field_name, sql_type]

        # Handle nullable/required
        if field_info.get("required", True):
            definition_parts.append("NOT NULL")

        # Handle default values
        if "default" in field_info:
            default_value = field_info["default"]
            if default_value is not None:
                if isinstance(default_value, str):
                    definition_parts.append(f"DEFAULT '{default_value}'")
                elif isinstance(default_value, bool):
                    if database_type == "postgresql":
                        definition_parts.append(f"DEFAULT {str(default_value).upper()}")
                    elif database_type == "mysql":
                        definition_parts.append(f"DEFAULT {1 if default_value else 0}")
                    else:  # sqlite
                        definition_parts.append(f"DEFAULT {1 if default_value else 0}")
                else:
                    definition_parts.append(f"DEFAULT {default_value}")

        return " ".join(definition_parts)

    def _generate_create_table_sql(
        self,
        model_name: str,
        database_type: str = "postgresql",
        model_fields: Optional[Dict] = None,
    ) -> str:
        """Generate CREATE TABLE SQL statement from model metadata.

        Args:
            model_name: Name of the model class
            database_type: Target database type
            model_fields: Optional model fields dict (if not provided, uses registered model fields)

        Returns:
            Complete CREATE TABLE SQL statement
        """
        table_name = self._class_name_to_table_name(model_name)
        fields = (
            model_fields
            if model_fields is not None
            else self.get_model_fields(model_name)
        )

        if not fields:
            raise ValueError(f"No fields found for model {model_name}")

        # Start building CREATE TABLE statement with safety protection
        sql_parts = [f"CREATE TABLE IF NOT EXISTS {table_name} ("]

        # Check if model has a string ID field
        id_field = fields.get("id", {})
        id_type = id_field.get("type")

        # Add primary key ID column based on type
        if id_type == str:
            # String ID models need user-provided IDs
            if database_type.lower() in ["postgresql", "mysql"]:
                sql_parts.append("    id TEXT PRIMARY KEY,")
            else:  # sqlite
                sql_parts.append("    id TEXT PRIMARY KEY,")
        else:
            # Integer ID models use auto-increment
            if database_type.lower() == "postgresql":
                sql_parts.append("    id SERIAL PRIMARY KEY,")
            elif database_type.lower() == "mysql":
                sql_parts.append("    id INT AUTO_INCREMENT PRIMARY KEY,")
            else:  # sqlite
                sql_parts.append("    id INTEGER PRIMARY KEY AUTOINCREMENT,")

        # Add model fields
        column_definitions = []
        for field_name, field_info in fields.items():
            # Skip auto-generated fields
            if field_name in ["id", "created_at", "updated_at"]:
                continue

            column_def = self._get_sql_column_definition(
                field_name, field_info, database_type
            )
            column_definitions.append(f"    {column_def}")

        # Add created_at and updated_at timestamp columns
        if database_type.lower() == "postgresql":
            column_definitions.append(
                "    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP"
            )
            column_definitions.append(
                "    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP"
            )
        elif database_type.lower() == "mysql":
            column_definitions.append(
                "    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP"
            )
            column_definitions.append(
                "    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"
            )
        else:  # sqlite
            column_definitions.append("    created_at TEXT DEFAULT CURRENT_TIMESTAMP")
            column_definitions.append("    updated_at TEXT DEFAULT CURRENT_TIMESTAMP")

        # Join all column definitions
        sql_parts.extend([",\n".join(column_definitions)])
        sql_parts.append(");")

        return "\n".join(sql_parts)

    def _generate_indexes_sql(
        self, model_name: str, database_type: str = "postgresql"
    ) -> List[str]:
        """Generate CREATE INDEX SQL statements for a model.

        Args:
            model_name: Name of the model class
            database_type: Target database type

        Returns:
            List of CREATE INDEX SQL statements
        """
        table_name = self._class_name_to_table_name(model_name)
        indexes = []

        # Get model configuration for custom indexes
        model_info = self._models.get(model_name)
        if model_info:
            model_cls = model_info.get("class")
            if model_cls and hasattr(model_cls, "__dataflow__"):
                config = getattr(model_cls, "__dataflow__", {})
                custom_indexes = config.get("indexes", [])

                for index_config in custom_indexes:
                    index_name = index_config.get(
                        "name", f"idx_{table_name}_{index_config['fields'][0]}"
                    )
                    fields = index_config.get("fields", [])
                    unique = index_config.get("unique", False)

                    if fields:
                        unique_keyword = "UNIQUE " if unique else ""
                        fields_str = ", ".join(fields)
                        sql = f"CREATE {unique_keyword}INDEX {index_name} ON {table_name} ({fields_str});"
                        indexes.append(sql)

        # Add automatic indexes for foreign keys
        relationships = self.get_relationships(model_name)
        for rel_name, rel_info in relationships.items():
            if rel_info.get("type") == "belongs_to" and rel_info.get("foreign_key"):
                foreign_key = rel_info["foreign_key"]
                index_name = f"idx_{table_name}_{foreign_key}"
                sql = f"CREATE INDEX {index_name} ON {table_name} ({foreign_key});"
                indexes.append(sql)

        return indexes

    def _generate_foreign_key_constraints_sql(
        self, model_name: str, database_type: str = "postgresql"
    ) -> List[str]:
        """Generate ALTER TABLE statements for foreign key constraints.

        Args:
            model_name: Name of the model class
            database_type: Target database type

        Returns:
            List of ALTER TABLE SQL statements for foreign keys
        """
        table_name = self._class_name_to_table_name(model_name)
        constraints = []

        # Get relationships for this model
        relationships = self.get_relationships(model_name)
        for rel_name, rel_info in relationships.items():
            if rel_info.get("type") == "belongs_to" and rel_info.get("foreign_key"):
                foreign_key = rel_info["foreign_key"]
                target_table = rel_info["target_table"]
                target_key = rel_info.get("target_key", "id")

                constraint_name = f"fk_{table_name}_{foreign_key}"
                sql = (
                    f"ALTER TABLE {table_name} "
                    f"ADD CONSTRAINT {constraint_name} "
                    f"FOREIGN KEY ({foreign_key}) "
                    f"REFERENCES {target_table}({target_key});"
                )
                constraints.append(sql)

        return constraints

    def generate_complete_schema_sql(
        self, database_type: str = "postgresql"
    ) -> Dict[str, List[str]]:
        """Generate complete database schema SQL for all registered models.

        Args:
            database_type: Target database type

        Returns:
            Dictionary with SQL statements grouped by type
        """
        schema_sql = {"tables": [], "indexes": [], "foreign_keys": []}

        # Generate CREATE TABLE statements for all models
        for model_name in self._models.keys():
            try:
                table_sql = self._generate_create_table_sql(model_name, database_type)
                schema_sql["tables"].append(table_sql)

                # Generate indexes
                indexes = self._generate_indexes_sql(model_name, database_type)
                schema_sql["indexes"].extend(indexes)

                # Generate foreign key constraints
                constraints = self._generate_foreign_key_constraints_sql(
                    model_name, database_type
                )
                schema_sql["foreign_keys"].extend(constraints)

            except Exception as e:
                logger.error(f"Error generating SQL for model {model_name}: {e}")

        return schema_sql

    def _get_database_connection(self):
        """Get a real PostgreSQL database connection for DDL operations."""
        try:
            # Use the connection manager to get a real PostgreSQL connection
            if hasattr(self._connection_manager, "get_connection"):
                connection = self._connection_manager.get_connection()
                if connection:
                    return connection

            # Fallback: Create direct PostgreSQL connection
            database_url = self.config.database.url
            if not database_url or database_url == ":memory:":
                # For testing, create a simple SQLite connection
                import sqlite3

                connection = sqlite3.connect(":memory:")
                return connection

            # PostgreSQL connection using asyncpg (for proper async support)
            if "postgresql" in database_url or "postgres" in database_url:
                logger.warning(
                    "_get_database_connection() is sync but PostgreSQL requires async. Use _get_async_database_connection() instead."
                )
                return self._get_async_sql_connection()

            # Fallback to AsyncSQLDatabaseNode
            return self._get_async_sql_connection()

        except Exception as e:
            logger.error(f"Failed to get database connection: {e}")
            # Return a basic connection that supports basic operations
            import sqlite3

            return sqlite3.connect(":memory:")

    def _get_async_sql_connection(self):
        """Get connection wrapper using AsyncSQLDatabaseNode."""
        try:
            from kailash.nodes.data.async_sql import AsyncSQLDatabaseNode

            from ..adapters.connection_parser import ConnectionParser

            # Create a safe connection string
            components = ConnectionParser.parse_connection_string(
                self.config.database.url
            )
            safe_connection_string = ConnectionParser.build_connection_string(
                scheme=components.get("scheme"),
                host=components.get("host"),
                database=components.get("database"),
                username=components.get("username"),
                password=components.get("password"),
                port=components.get("port"),
                **components.get("query_params", {}),
            )

            # Create a connection wrapper that supports the needed interface
            class AsyncSQLConnectionWrapper:
                def __init__(self, connection_string):
                    self.connection_string = connection_string
                    self._transaction = None

                def cursor(self):
                    return self

                def execute(self, sql, params=None):
                    # Auto-detect database type from connection string
                    from ..adapters.connection_parser import ConnectionParser

                    database_type = ConnectionParser.detect_database_type(
                        self.connection_string
                    )

                    node = AsyncSQLDatabaseNode(
                        node_id="ddl_executor",
                        connection_string=self.connection_string,
                        database_type=database_type,
                        query=sql,
                        fetch_mode="all",
                        validate_queries=False,
                    )
                    return node.execute()

                def fetchall(self):
                    return []

                def fetchone(self):
                    return None

                def commit(self):
                    pass

                def rollback(self):
                    pass

                def close(self):
                    pass

                def begin(self):
                    self._transaction = self
                    return self._transaction

                def __enter__(self):
                    return self

                def __exit__(self, exc_type, exc_val, exc_tb):
                    if exc_type is None:
                        self.commit()
                    else:
                        self.rollback()
                    return False

            return AsyncSQLConnectionWrapper(safe_connection_string)

        except Exception as e:
            logger.error(f"Failed to create AsyncSQL connection wrapper: {e}")
            import sqlite3

            return sqlite3.connect(":memory:")

    def _execute_ddl_with_transaction(self, ddl_statement: str):
        """Execute DDL statement within a database transaction with rollback capability."""
        connection = self._get_async_sql_connection()
        transaction = None

        try:
            # Begin transaction
            transaction = connection.begin()

            # Execute DDL statement
            connection.execute(ddl_statement)

            # Commit transaction
            transaction.commit()

            logger.info(f"DDL executed successfully: {ddl_statement[:100]}...")

        except Exception as e:
            # Rollback transaction on error
            if transaction:
                transaction.rollback()
                logger.error(f"DDL transaction rolled back due to error: {e}")
            raise e
        finally:
            if connection:
                connection.close()

    def _execute_multi_statement_ddl(self, ddl_statements: List[str]):
        """Execute multiple DDL statements within a single transaction."""
        connection = self._get_async_sql_connection()
        transaction = None

        try:
            # Begin transaction
            transaction = connection.begin()

            # Execute all DDL statements
            for statement in ddl_statements:
                connection.execute(statement)

            # Commit transaction
            transaction.commit()

            logger.info(
                f"Multi-statement DDL executed successfully: {len(ddl_statements)} statements"
            )

        except Exception as e:
            # Rollback transaction on error
            if transaction:
                transaction.rollback()
                logger.error(
                    f"Multi-statement DDL transaction rolled back due to error: {e}"
                )
            raise e
        finally:
            if connection:
                connection.close()

    def _trigger_universal_schema_management(
        self, model_name: str, fields: Dict[str, Any]
    ):
        """Trigger database-agnostic schema state management for model registration.

        This method detects the database type and calls the appropriate
        schema management system (PostgreSQL or SQLite).
        """
        database_url = self.config.database.url or ":memory:"

        # Detect database type and route to appropriate schema management
        if "postgresql" in database_url or "postgres" in database_url:
            logger.debug(f"Using PostgreSQL schema management for model {model_name}")
            self._trigger_postgresql_schema_management(model_name, fields)
        elif (
            "sqlite" in database_url
            or database_url == ":memory:"
            or database_url.endswith(".db")
        ):
            logger.debug(f"Using SQLite schema management for model {model_name}")
            self._trigger_sqlite_schema_management(model_name, fields)
        else:
            # Extract scheme from URL for better error message
            try:
                scheme = (
                    database_url.split("://")[0] if "://" in database_url else "unknown"
                )
            except:
                scheme = "unknown"
            logger.warning(
                f"Unknown database type '{scheme}' for model {model_name}. "
                f"Schema management may not work correctly."
            )
            # Fallback to PostgreSQL management for unknown databases
            self._trigger_postgresql_schema_management(model_name, fields)

    def _trigger_sqlite_schema_management(
        self, model_name: str, fields: Dict[str, Any]
    ):
        """Trigger SQLite-optimized schema state management for model registration."""

        # Handle existing_schema_mode - skip all migration activities
        if self._existing_schema_mode:
            logger.info(
                f"existing_schema_mode=True enabled. Skipping all SQLite schema management for model '{model_name}'."
            )
            return

        # Check if auto-migration is enabled
        if not self._auto_migrate:
            logger.info(
                f"Auto-migration disabled for SQLite model '{model_name}'. "
                f"Tables will be created on-demand during first node execution."
            )
            return

        # For SQLite, we'll use a simpler approach than PostgreSQL's complex schema state management
        # Just ensure the table exists using the migration system
        if self._migration_system is not None:
            self._trigger_sqlite_migration_system(model_name, fields)
        else:
            logger.info(
                f"No migration system available for SQLite model '{model_name}'. "
                f"Table will be created on-demand during first node execution."
            )

    def _convert_dict_schema_to_table_definitions(
        self, dict_schema: Dict[str, Dict]
    ) -> Dict[str, Any]:
        """Convert dictionary schema format to TableDefinition format expected by migration system."""
        from dataflow.migrations.auto_migration_system import (
            ColumnDefinition,
            TableDefinition,
        )

        table_definitions = {}

        for table_name, table_info in dict_schema.items():
            columns_dict = table_info.get("columns", {})

            # Convert dictionary columns to ColumnDefinition objects
            columns = []
            for col_name, col_info in columns_dict.items():
                column = ColumnDefinition(
                    name=col_name,
                    type=col_info.get("type", "TEXT"),
                    nullable=col_info.get("nullable", True),
                    default=col_info.get("default"),
                    primary_key=col_info.get("primary_key", False),
                    unique=col_info.get("unique", False),
                    auto_increment=(
                        col_name == "id" and col_info.get("default") == "nextval"
                    ),
                )
                columns.append(column)

            # Create TableDefinition object
            table_def = TableDefinition(
                name=table_name,
                columns=columns,
                indexes=table_info.get("indexes", []),
                constraints=table_info.get("constraints", []),
            )

            table_definitions[table_name] = table_def

        return table_definitions

    def _trigger_sqlite_migration_system(self, model_name: str, fields: Dict[str, Any]):
        """Trigger SQLite migration system to ensure table exists."""

        table_name = self._class_name_to_table_name(model_name)

        # Build expected table schema from model fields in dictionary format
        dict_schema = {table_name: {"columns": self._convert_fields_to_columns(fields)}}

        # Convert to TableDefinition format expected by migration system
        target_schema = self._convert_dict_schema_to_table_definitions(dict_schema)

        # Execute auto-migration with SQLite-specific handling
        import asyncio

        async def run_sqlite_migration():
            try:
                # Check if table already exists by trying to create it
                # The migration system will handle the actual table creation
                success, migrations = await self._migration_system.auto_migrate(
                    target_schema=target_schema,
                    dry_run=False,
                    interactive=False,  # Non-interactive for SQLite
                    auto_confirm=True,  # Auto-confirm for SQLite simplicity
                )

                if success:
                    logger.info(
                        f"SQLite table '{table_name}' ready for model '{model_name}'"
                    )
                else:
                    logger.warning(
                        f"SQLite migration failed for model '{model_name}': {migrations}"
                    )

                return success, migrations

            except Exception as e:
                logger.error(f"SQLite migration error for model '{model_name}': {e}")
                # Don't fail model registration - table will be created on-demand
                return False, []

        # Run migration with proper event loop handling
        try:
            # Check if there's already a running event loop
            loop = asyncio.get_running_loop()
            # Use ThreadPoolExecutor to run in separate thread
            import concurrent.futures

            with concurrent.futures.ThreadPoolExecutor() as executor:
                future = executor.submit(asyncio.run, run_sqlite_migration())
                success, migrations = future.result()
        except RuntimeError:
            # No event loop running, safe to use asyncio.run()
            success, migrations = asyncio.run(run_sqlite_migration())

        if success:
            logger.info(
                f"SQLite schema management completed successfully for model '{model_name}'"
            )
        else:
            logger.info(
                f"SQLite table creation deferred to first node execution for model '{model_name}'"
            )

    def _trigger_postgresql_schema_management(
        self, model_name: str, fields: Dict[str, Any]
    ):
        """Trigger PostgreSQL-optimized schema state management for model registration."""

        # Handle existing_schema_mode - skip all migration activities
        if self._existing_schema_mode:
            logger.info(
                f"existing_schema_mode=True enabled. Skipping all PostgreSQL schema management for model '{model_name}'."
            )
            return

        # Use PostgreSQL-optimized schema state manager if available
        if self._schema_state_manager is not None:
            self._trigger_postgresql_enhanced_schema_management(model_name, fields)
        elif self._migration_system is not None:
            self._trigger_postgresql_migration_system(model_name, fields)

    def _trigger_postgresql_enhanced_schema_management(
        self, model_name: str, fields: Dict[str, Any]
    ):
        """Trigger PostgreSQL-optimized enhanced schema state management."""

        # Handle existing_schema_mode validation first - skip all migrations
        if self._existing_schema_mode:
            logger.info(
                f"existing_schema_mode=True enabled. Skipping enhanced schema management for model '{model_name}'."
            )
            return

        # Check if auto-migration is enabled - skip if disabled
        if not self._auto_migrate:
            logger.info(
                f"Auto-migration disabled for model '{model_name}'. "
                f"Enhanced schema management will not be applied automatically."
            )
            return

        from ..migrations.schema_state_manager import ModelSchema

        # Build model schema for the specific model being registered
        # The existing_schema_mode is handled at the migration comparison level
        model_schema = ModelSchema(
            tables={
                self._class_name_to_table_name(model_name): {
                    "columns": self._convert_fields_to_columns(fields)
                }
            }
        )

        # Generate unique PostgreSQL connection ID for this engine instance
        connection_id = f"dataflow_postgresql_{id(self)}"

        try:
            # Use schema state manager for migration planning (transactions handled by WorkflowBuilder)
            schema_manager = self._schema_state_manager
            # Detect changes and plan migrations with PostgreSQL optimization
            operations, safety_assessment = schema_manager.detect_and_plan_migrations(
                model_schema, connection_id
            )

            if len(operations) == 0:
                logger.info(
                    f"No PostgreSQL schema changes detected for model {model_name}"
                )
                return

            # Show enhanced migration preview with safety assessment
            self._show_enhanced_migration_preview(
                model_name, operations, safety_assessment
            )

            # Request user confirmation with risk assessment
            user_confirmed = self._request_enhanced_user_confirmation(
                operations, safety_assessment
            )

            if user_confirmed:
                # Execute PostgreSQL migration with enhanced tracking
                if self._migration_system is not None:
                    self._execute_postgresql_migration_with_tracking(
                        model_name, operations
                    )
                else:
                    logger.warning("No PostgreSQL migration execution system available")
            else:
                logger.info(
                    f"User declined PostgreSQL migration for model {model_name}"
                )

        except Exception as e:
            logger.error(
                f"PostgreSQL enhanced schema management failed for model {model_name}: {e}"
            )
            # Fallback to PostgreSQL migration system if available
            if self._migration_system is not None:
                logger.info("Falling back to PostgreSQL migration system")
                self._trigger_postgresql_migration_system(model_name, fields)
            else:
                raise e

    def _trigger_postgresql_migration_system(
        self, model_name: str, fields: Dict[str, Any]
    ):
        """Trigger PostgreSQL migration system for model registration."""
        try:
            # Create target schema from model definition
            table_name = self._class_name_to_table_name(model_name)

            # Convert fields to AutoMigrationSystem format
            from ..migrations.auto_migration_system import (
                ColumnDefinition,
                TableDefinition,
            )

            # Build target schema based on existing_schema_mode
            if self._existing_schema_mode:
                # In existing schema mode, preserve all current tables and only add/update the new model
                target_schema = self._build_incremental_target_schema(
                    model_name, fields
                )
            else:
                # Default mode: only include the new model (may drop existing tables)
                target_schema = {}

            # Create the table definition for the new/updated model
            columns = []
            # Add auto-generated ID column
            # Note: Use INTEGER type for comparison, not SERIAL (which is only valid in CREATE TABLE)
            columns.append(
                ColumnDefinition(
                    name="id", type="INTEGER", nullable=False, primary_key=True
                )
            )

            # Add model fields
            for field_name, field_info in fields.items():
                field_type = field_info.get("type", str)
                sql_type = self._python_type_to_sql_type(field_type, "postgresql")

                column = ColumnDefinition(
                    name=field_name,
                    type=sql_type,
                    nullable=not field_info.get("required", True),
                    default=field_info.get("default"),
                )
                columns.append(column)

            # Add timestamp columns
            columns.extend(
                [
                    ColumnDefinition(
                        name="created_at",
                        type="TIMESTAMP WITH TIME ZONE",
                        nullable=False,
                        default="CURRENT_TIMESTAMP",
                    ),
                    ColumnDefinition(
                        name="updated_at",
                        type="TIMESTAMP WITH TIME ZONE",
                        nullable=False,
                        default="CURRENT_TIMESTAMP",
                    ),
                ]
            )

            # Add or update the table definition in target schema
            target_schema[table_name] = TableDefinition(
                name=table_name, columns=columns
            )

            # Handle existing_schema_mode validation first
            if self._existing_schema_mode:
                logger.info(
                    f"Existing schema mode enabled. Validating compatibility for '{model_name}'..."
                )

                import asyncio

                async def validate_schema():
                    return await self._validate_existing_schema_compatibility(
                        model_name, target_schema
                    )

                try:
                    loop = asyncio.get_event_loop()
                    is_compatible = loop.run_until_complete(validate_schema())
                except RuntimeError:
                    loop = asyncio.new_event_loop()
                    asyncio.set_event_loop(loop)
                    is_compatible = loop.run_until_complete(validate_schema())

                if not is_compatible:
                    raise RuntimeError(
                        f"Model '{model_name}' is not compatible with existing database schema. "
                        f"Please ensure database tables match model definitions or disable "
                        f"existing_schema_mode to allow migrations."
                    )

                # Schema is compatible in existing_schema_mode - NEVER run migrations
                logger.info(
                    f"Schema compatibility validated. existing_schema_mode=True - skipping all migrations for model '{model_name}'."
                )
                return

            # Check if auto-migration is enabled (for non-existing_schema_mode cases)
            elif not self._auto_migrate:
                logger.info(
                    f"Auto-migration disabled for model '{model_name}'. "
                    f"Schema changes will not be applied automatically."
                )
                return

            # Execute auto-migration with PostgreSQL optimizations
            import asyncio

            async def run_postgresql_migration():
                # Pass existing_schema_mode context to the migration system
                if hasattr(self._migration_system, "_existing_schema_mode"):
                    self._migration_system._existing_schema_mode = (
                        self._existing_schema_mode
                    )

                success, migrations = await self._migration_system.auto_migrate(
                    target_schema=target_schema,
                    dry_run=False,
                    interactive=not self._auto_migrate,  # Non-interactive if auto_migrate=True
                    auto_confirm=self._auto_migrate,  # Auto-confirm if auto_migrate=True
                )
                return success, migrations

            # Run migration with proper event loop handling
            try:
                # Check if there's already a running event loop
                loop = asyncio.get_running_loop()
                # Use ThreadPoolExecutor to run in separate thread
                import concurrent.futures

                with concurrent.futures.ThreadPoolExecutor() as executor:
                    future = executor.submit(asyncio.run, run_postgresql_migration())
                    success, migrations = future.result()
            except RuntimeError:
                # No event loop running, safe to use asyncio.run()
                success, migrations = asyncio.run(run_postgresql_migration())

            if success:
                logger.info(
                    f"PostgreSQL migration executed successfully for model {model_name}"
                )
                if migrations:
                    for migration in migrations:
                        logger.info(
                            f"Applied migration {migration.version} with {len(migration.operations)} operations"
                        )
            else:
                logger.warning(
                    f"PostgreSQL migration was not applied for model {model_name}"
                )

        except Exception as e:
            logger.error(
                f"PostgreSQL migration system failed for model {model_name}: {e}"
            )
            # Don't raise - allow model registration to continue
            logger.info(f"Model {model_name} registered without migration")

    def _build_incremental_target_schema(
        self, model_name: str, fields: Dict[str, Any]
    ) -> Dict[str, Any]:
        """Build target schema for existing_schema_mode: preserve existing tables + add/update model.

        This method ensures that in existing_schema_mode=True, we only modify the specific
        model's table and preserve all other existing tables in the database.
        """
        try:
            # Import here to avoid circular imports
            from ..migrations.auto_migration_system import (
                ColumnDefinition,
                TableDefinition,
            )

            # Try to get current schema using DataFlow's own discovery method
            # This avoids async compatibility issues with the migration system inspector
            try:
                current_schema_dict = self.discover_schema()

                # Convert DataFlow schema format to AutoMigrationSystem TableDefinition format
                current_schema = {}
                for table_name, table_info in current_schema_dict.items():
                    columns = []
                    for column_info in table_info.get("columns", []):
                        column = ColumnDefinition(
                            name=column_info["name"],
                            type=column_info["type"],
                            nullable=column_info.get("nullable", True),
                            default=column_info.get("default"),
                            primary_key=column_info.get("primary_key", False),
                            unique=column_info.get("unique", False),
                        )
                        columns.append(column)

                    current_schema[table_name] = TableDefinition(
                        name=table_name, columns=columns
                    )

                logger.info(
                    f"Existing schema mode: preserving {len(current_schema)} existing tables"
                )
                return current_schema

            except Exception as schema_error:
                logger.warning(f"DataFlow schema discovery failed: {schema_error}")

                # Fallback: try the migration system inspector if available
                if self._migration_system and hasattr(
                    self._migration_system, "inspector"
                ):
                    try:
                        # Use asyncio to get current schema
                        import asyncio

                        try:
                            loop = asyncio.get_event_loop()
                            current_schema = loop.run_until_complete(
                                self._migration_system.inspector.get_current_schema()
                            )
                        except RuntimeError:
                            # Create new event loop if none exists
                            loop = asyncio.new_event_loop()
                            asyncio.set_event_loop(loop)
                            current_schema = loop.run_until_complete(
                                self._migration_system.inspector.get_current_schema()
                            )
                            loop.close()

                        logger.info(
                            f"Existing schema mode: preserving {len(current_schema)} existing tables (via fallback)"
                        )
                        return current_schema
                    except Exception as inspector_error:
                        logger.warning(
                            f"Migration system inspector also failed: {inspector_error}"
                        )

                logger.warning(
                    "All schema discovery methods failed, using empty target schema"
                )
                return {}

        except Exception as e:
            logger.error(f"Failed to build incremental target schema: {e}")
            # Fallback to empty schema to avoid breaking model registration
            return {}

    def _build_incremental_model_schema(self, model_name: str, fields: Dict[str, Any]):
        """Build ModelSchema for existing_schema_mode: preserve existing tables + add/update model.

        This method creates a ModelSchema that includes all existing tables plus the new/updated model.
        Used by the enhanced schema management system.
        """
        try:
            from ..migrations.schema_state_manager import ModelSchema

            # Start with existing tables preserved
            incremental_schema = self._build_incremental_target_schema(
                model_name, fields
            )

            # Convert TableDefinition format to ModelSchema format
            model_schema_tables = {}

            for table_name, table_def in incremental_schema.items():
                # Convert columns from TableDefinition to ModelSchema format
                columns = {}
                for column in table_def.columns:
                    columns[column.name] = {
                        "type": column.type,
                        "nullable": column.nullable,
                        "primary_key": column.primary_key,
                        "unique": column.unique,
                        "default": column.default,
                    }

                model_schema_tables[table_name] = {"columns": columns}

            # Add or update the current model's table
            table_name = self._class_name_to_table_name(model_name)
            model_schema_tables[table_name] = {
                "columns": self._convert_fields_to_columns(fields)
            }

            logger.info(
                f"Built incremental model schema with {len(model_schema_tables)} tables"
            )
            return ModelSchema(tables=model_schema_tables)

        except Exception as e:
            logger.error(f"Failed to build incremental model schema: {e}")
            # Fallback to single-table schema
            from ..migrations.schema_state_manager import ModelSchema

            return ModelSchema(
                tables={
                    self._class_name_to_table_name(model_name): {
                        "columns": self._convert_fields_to_columns(fields)
                    }
                }
            )

    def _convert_fields_to_columns(self, fields: Dict[str, Any]) -> Dict[str, Any]:
        """Convert DataFlow field format to schema state manager column format."""
        columns = {}

        # Check if model has a string ID field
        id_field = fields.get("id", {})
        id_type = id_field.get("type")

        if id_type == str:
            # String ID models need user-provided IDs
            columns["id"] = {
                "type": "TEXT",
                "nullable": False,
                "primary_key": True,
                "unique": False,
                "default": None,  # No default for string IDs
            }
        else:
            # Integer ID models use auto-increment
            # Get database type to set appropriate defaults
            database_url = self.config.database.url or ":memory:"
            is_sqlite = "sqlite" in database_url.lower() or database_url == ":memory:"

            columns["id"] = {
                "type": "INTEGER",  # Use INTEGER for comparison (SERIAL is CREATE TABLE syntax only)
                "nullable": False,
                "primary_key": True,
                "unique": False,
                "default": (
                    None if is_sqlite else "nextval"
                ),  # SQLite doesn't use nextval
            }

        # Add user-defined fields
        for field_name, field_info in fields.items():
            # Skip auto-managed fields - already processed above or will be added below
            if field_name in ["id", "created_at", "updated_at"]:
                continue

            python_type = field_info.get("type", str)

            # Convert Python type to SQL type string
            sql_type = self._python_type_to_sql_type(python_type)

            columns[field_name] = {
                "type": sql_type,
                "nullable": not field_info.get("required", True),
                "primary_key": False,  # Only id is primary key
                "unique": field_name in ["email", "username"],  # Common unique fields
                "default": field_info.get("default"),
            }

        # Always include the auto-generated timestamp columns that DataFlow adds
        columns["created_at"] = {
            "type": "TIMESTAMP",
            "nullable": True,  # Match what we see in the database
            "primary_key": False,
            "unique": False,
            "default": "CURRENT_TIMESTAMP",
        }

        columns["updated_at"] = {
            "type": "TIMESTAMP",
            "nullable": True,  # Match what we see in the database
            "primary_key": False,
            "unique": False,
            "default": "CURRENT_TIMESTAMP",
        }

        return columns

    def _show_enhanced_migration_preview(
        self, model_name: str, operations, safety_assessment
    ):
        """Show enhanced migration preview with safety assessment."""
        logger.info(f"\n🔄 Enhanced Migration Preview for {model_name}")
        logger.info(f"📊 Operations: {len(operations)}")
        logger.info(f"🛡️ Safety Level: {safety_assessment.overall_risk.value.upper()}")

        if not safety_assessment.is_safe:
            logger.warning("⚠️ WARNING: This migration has potential risks!")
            for warning in safety_assessment.warnings:
                logger.warning(f"   - {warning}")

        for i, operation in enumerate(operations, 1):
            logger.info(f"  {i}. {operation.operation_type} on {operation.table_name}")

    def _request_enhanced_user_confirmation(
        self, operations, safety_assessment
    ) -> bool:
        """Request user confirmation with enhanced risk information."""
        if safety_assessment.is_safe and safety_assessment.overall_risk.value == "none":
            # Auto-approve safe operations
            logger.info("✅ Safe migration auto-approved")
            return True

        # For risky operations, delegate to existing confirmation system
        # In a real implementation, this would show an enhanced UI
        return self._request_user_confirmation(
            f"Migration with {len(operations)} operations"
        )

    def _execute_postgresql_migration_with_tracking(self, model_name: str, operations):
        """Execute PostgreSQL migration with enhanced tracking through schema state manager."""
        from ..migrations.schema_state_manager import MigrationRecord, MigrationStatus

        # Create PostgreSQL migration record
        migration_record = MigrationRecord(
            migration_id=f"dataflow_postgresql_{model_name}_{int(time.time())}",
            name=f"PostgreSQL auto-generated migration for {model_name}",
            operations=[
                {
                    "type": op.operation_type,
                    "table": op.table_name,
                    "details": op.details,
                    "sql_up": getattr(op, "sql_up", ""),
                    "sql_down": getattr(op, "sql_down", ""),
                }
                for op in operations
            ],
            status=MigrationStatus.PENDING,
            applied_at=datetime.now(),
        )

        try:
            # Execute specific migration operations instead of recreating the table
            table_name = self._class_name_to_table_name(model_name)
            connection = self._get_async_sql_connection()

            # Detect database type for SQL generation
            is_sqlite = hasattr(connection, "execute") and "sqlite" in str(
                type(connection)
            )
            db_type = "sqlite" if is_sqlite else "postgresql"

            # Generate and execute specific migration SQL for each operation
            migration_sqls = []
            for operation in operations:
                sql = self._generate_migration_sql(operation, table_name, db_type)
                if sql:
                    migration_sqls.append(sql)

            # Execute migration SQL statements
            try:
                for sql in migration_sqls:
                    logger.info(f"Executing migration SQL: {sql}")

                    if is_sqlite:
                        # SQLite doesn't support cursor context manager
                        cursor = connection.cursor()
                        cursor.execute(sql)
                        cursor.close()
                    else:
                        # PostgreSQL with context manager
                        with connection.cursor() as cursor:
                            cursor.execute(sql)

                connection.commit()
                logger.info(
                    f"Successfully executed {len(migration_sqls)} migration operations on table '{table_name}'"
                )
            except Exception as sql_error:
                connection.rollback()
                raise sql_error
            finally:
                connection.close()

            # Record successful migration
            migration_record.status = MigrationStatus.APPLIED
            if self._schema_state_manager:
                self._schema_state_manager.history_manager.record_migration(
                    migration_record
                )

            logger.info(
                f"PostgreSQL migration executed and tracked successfully for model {model_name}"
            )

        except Exception as e:
            # Record failed migration
            migration_record.status = MigrationStatus.FAILED
            if self._schema_state_manager:
                try:
                    self._schema_state_manager.history_manager.record_migration(
                        migration_record
                    )
                except:
                    pass  # Don't fail if we can't record the failure

            logger.error(
                f"PostgreSQL migration execution failed for model {model_name}: {e}"
            )
            # Don't raise - allow model registration to continue
            logger.info(f"Model {model_name} registered without PostgreSQL migration")

    def _generate_migration_sql(
        self, operation, table_name: str, database_type: str
    ) -> str:
        """Generate SQL for a specific migration operation.

        Args:
            operation: MigrationOperation object with operation_type and details
            table_name: Name of the table to modify
            database_type: Database type (postgresql, mysql, sqlite)

        Returns:
            SQL statement for the migration operation
        """
        operation_type = operation.operation_type
        details = operation.details

        if operation_type == "ADD_COLUMN":
            column_name = details.get("column_name")
            if not column_name:
                return ""

            # Get the field info for this column from the model
            model_name = None
            for name, info in self._models.items():
                if self._class_name_to_table_name(name) == table_name:
                    model_name = name
                    break

            if not model_name:
                return ""

            model_fields = self.get_model_fields(model_name)
            field_info = model_fields.get(column_name)

            if not field_info:
                return ""

            # Generate column definition for ALTER TABLE ADD COLUMN
            column_definition = self._get_sql_column_definition(
                column_name, field_info, database_type
            )

            return f"ALTER TABLE {table_name} ADD COLUMN {column_definition};"

        elif operation_type == "DROP_COLUMN":
            column_name = details.get("column_name")
            if not column_name:
                return ""
            return f"ALTER TABLE {table_name} DROP COLUMN {column_name};"

        elif operation_type == "MODIFY_COLUMN":
            column_name = details.get("column_name")
            if not column_name:
                return ""

            # Get new type from changes or details
            changes = details.get("changes", {})
            new_type = changes.get("new_type") or details.get("new_type")

            if not new_type:
                return ""

            if database_type.lower() == "postgresql":
                return f"ALTER TABLE {table_name} ALTER COLUMN {column_name} TYPE {new_type};"
            elif database_type.lower() == "mysql":
                return (
                    f"ALTER TABLE {table_name} MODIFY COLUMN {column_name} {new_type};"
                )
            else:  # sqlite - doesn't support ALTER COLUMN TYPE directly
                return ""  # Skip for SQLite

        elif operation_type == "CREATE_TABLE":
            # For CREATE_TABLE operations, use the existing method
            model_name = None
            for name, info in self._models.items():
                if self._class_name_to_table_name(name) == table_name:
                    model_name = name
                    break

            if model_name:
                return self._generate_create_table_sql(model_name, database_type)

        return ""

    def _request_user_confirmation(self, migration_preview: str) -> bool:
        """Request user confirmation for migration execution."""
        # In a real implementation, this would show an interactive prompt
        # For now, return True to simulate user approval
        return True

    def _show_migration_preview(self, preview: str):
        """Show migration preview to user."""
        logger.info(f"Migration Preview:\n{preview}")

    def _notify_user_error(self, error_message: str):
        """Notify user of migration errors."""
        logger.error(f"Migration Error: {error_message}")

    def create_tables(self, database_type: str = None):
        """Create database tables for all registered models.

        This method generates and executes CREATE TABLE statements for all
        registered models along with their indexes and foreign key constraints.

        Args:
            database_type: Target database type ('postgresql', 'mysql', 'sqlite').
                          If None, auto-detected from URL.
        """
        # Auto-detect database type if not provided
        if database_type is None:
            database_type = self._detect_database_type()

        # Ensure migration tracking tables exist for all database types
        self._ensure_migration_tables(database_type)

        # Generate complete schema SQL
        schema_sql = self.generate_complete_schema_sql(database_type)

        logger.info(f"Creating database schema for {len(self._models)} models")

        # Log generated SQL for debugging
        logger.debug(f"Generated {len(schema_sql['tables'])} table statements")
        logger.debug(f"Generated {len(schema_sql['indexes'])} index statements")
        logger.debug(
            f"Generated {len(schema_sql['foreign_keys'])} foreign key statements"
        )

        # Execute DDL statements against the database using AsyncSQLDatabaseNode
        self._execute_ddl(schema_sql)

        logger.info(
            f"Successfully created database schema for {len(self._models)} models"
        )

    def _ensure_migration_tables(self, database_type: str = None):
        """Ensure both migration tracking tables exist."""
        try:
            runtime = LocalRuntime()

            # Get connection info
            connection_string = self.config.database.get_connection_url(
                self.config.environment
            )

            # Auto-detect database type if not provided
            if database_type is None:
                from ..adapters.connection_parser import ConnectionParser

                database_type = ConnectionParser.detect_database_type(connection_string)

            # Create database-specific dataflow_migrations table
            if database_type.lower() == "sqlite":
                create_table_sql = """
                CREATE TABLE IF NOT EXISTS dataflow_migrations (
                    version TEXT PRIMARY KEY,
                    name TEXT NOT NULL,
                    checksum TEXT NOT NULL,
                    applied_at TEXT,
                    status TEXT NOT NULL DEFAULT 'pending',
                    operations TEXT,
                    created_at TEXT DEFAULT CURRENT_TIMESTAMP,
                    model_definitions TEXT,
                    application_id TEXT,
                    last_synced_at TEXT,
                    CHECK (status IN ('pending', 'applied', 'failed', 'rolled_back'))
                )
                """
            else:  # PostgreSQL
                create_table_sql = """
                CREATE TABLE IF NOT EXISTS dataflow_migrations (
                    version VARCHAR(255) PRIMARY KEY,
                    name VARCHAR(255) NOT NULL,
                    checksum VARCHAR(32) NOT NULL,
                    applied_at TIMESTAMP WITH TIME ZONE,
                    status VARCHAR(50) NOT NULL DEFAULT 'pending',
                    operations JSONB,
                    created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
                    model_definitions JSONB,
                    application_id VARCHAR(255),
                    last_synced_at TIMESTAMP WITH TIME ZONE,
                    CONSTRAINT valid_status CHECK (status IN ('pending', 'applied', 'failed', 'rolled_back'))
                )
                """

            # Create table
            workflow = WorkflowBuilder()

            workflow.add_node(
                "AsyncSQLDatabaseNode",
                "create_table",
                {
                    "connection_string": connection_string,
                    "database_type": database_type,
                    "query": create_table_sql,
                    "validate_queries": False,
                },
            )
            results, _ = runtime.execute(workflow.build())

            if results.get("create_table", {}).get("status") != "completed":
                logger.warning("Failed to create dataflow_migrations table")
                return

            # Create database-specific indexes
            if database_type.lower() == "sqlite":
                indexes = [
                    "CREATE INDEX IF NOT EXISTS idx_migrations_status ON dataflow_migrations(status)",
                    "CREATE INDEX IF NOT EXISTS idx_migrations_application ON dataflow_migrations(application_id)",
                    "CREATE INDEX IF NOT EXISTS idx_migrations_checksum ON dataflow_migrations(checksum)",
                ]
            else:  # PostgreSQL
                indexes = [
                    "CREATE INDEX IF NOT EXISTS idx_migrations_status ON dataflow_migrations(status)",
                    "CREATE INDEX IF NOT EXISTS idx_migrations_application ON dataflow_migrations(application_id)",
                    "CREATE UNIQUE INDEX IF NOT EXISTS idx_migrations_checksum_unique ON dataflow_migrations(checksum) WHERE status = 'applied'",
                ]

            for idx, index_sql in enumerate(indexes):
                workflow = WorkflowBuilder()

                workflow.add_node(
                    "AsyncSQLDatabaseNode",
                    f"create_index_{idx}",
                    {
                        "connection_string": connection_string,
                        "database_type": database_type,
                        "query": index_sql,
                        "validate_queries": False,
                    },
                )
                results, _ = runtime.execute(workflow.build())

            logger.info("Migration tables ensured successfully")
            # Note: dataflow_migration_history is created by SchemaStateManager

        except Exception as e:
            logger.error(f"Error ensuring migration tables: {e}")
            # Don't fail the whole operation if table creation fails

    def _generate_insert_sql(
        self, model_name: str, database_type: str = "postgresql"
    ) -> str:
        """Generate INSERT SQL template for a model.

        Args:
            model_name: Name of the model class
            database_type: Target database type

        Returns:
            Parameterized INSERT SQL statement
        """
        table_name = self._class_name_to_table_name(model_name)
        fields = self.get_model_fields(model_name)

        # Get field names excluding auto-generated fields
        # CRITICAL FIX: Include ID for string ID models (user-provided IDs)
        field_names = []
        for name in fields.keys():
            if name == "id":
                # Include ID if it's string type (user-provided ID)
                id_field = fields.get("id", {})
                id_type = id_field.get("type")
                if id_type == str:
                    field_names.append(name)
            elif name not in ["created_at", "updated_at"]:
                field_names.append(name)

        # DEBUG: Log the exact field order used in SQL generation
        logger.warning(
            f"SQL GENERATION {model_name} - Field order from fields.keys(): {field_names}"
        )

        # Build column list and parameter placeholders
        columns = ", ".join(field_names)

        # Database-specific parameter placeholders
        if database_type.lower() == "postgresql":
            placeholders = ", ".join([f"${i+1}" for i in range(len(field_names))])
        elif database_type.lower() == "mysql":
            placeholders = ", ".join(["%s"] * len(field_names))
        else:  # sqlite
            placeholders = ", ".join(["?"] * len(field_names))

        sql = f"INSERT INTO {table_name} ({columns}) VALUES ({placeholders})"

        # Add RETURNING clause for PostgreSQL to get all fields back
        if database_type.lower() == "postgresql":
            # CRITICAL FIX: Use actual table columns for RETURNING clause
            # This prevents failures when timestamp columns don't exist
            try:
                actual_columns = self._get_table_columns(table_name)
                if actual_columns:
                    # Use only columns that actually exist in the table
                    all_columns = [
                        col
                        for col in ["id"] + field_names + ["created_at", "updated_at"]
                        if col in actual_columns
                    ]
                else:
                    # Fallback to expected columns if we can't check the table
                    all_columns = ["id"] + field_names + ["created_at", "updated_at"]
            except Exception:
                # If table inspection fails, use expected columns
                all_columns = ["id"] + field_names + ["created_at", "updated_at"]

            sql += f" RETURNING {', '.join(all_columns)}"

        return sql

    def _get_table_columns(self, table_name: str) -> List[str]:
        """Get actual column names from database table.

        Args:
            table_name: Name of the table to inspect

        Returns:
            List of column names that exist in the table
        """
        try:
            # Use the discover_schema functionality to get table info
            schema = self.discover_schema(use_real_inspection=True)
            if table_name in schema:
                table_info = schema[table_name]
                if "columns" in table_info:
                    return [col["name"] for col in table_info["columns"]]
                elif "fields" in table_info:
                    return list(table_info["fields"].keys())

            # Fallback: return empty list if table not found
            return []

        except Exception as e:
            logger.debug(f"Failed to get table columns for {table_name}: {e}")
            return []

    def _generate_select_sql(
        self, model_name: str, database_type: str = "postgresql"
    ) -> Dict[str, str]:
        """Generate SELECT SQL templates for a model.

        Args:
            model_name: Name of the model class
            database_type: Target database type

        Returns:
            Dictionary of SELECT SQL templates for different operations
        """
        table_name = self._class_name_to_table_name(model_name)
        fields = self.get_model_fields(model_name)

        # CRITICAL FIX: Use actual table columns for SELECT statements
        # This prevents failures when timestamp columns don't exist
        try:
            actual_columns = self._get_table_columns(table_name)
            if actual_columns:
                # Use only columns that actually exist in the table
                expected_columns = (
                    ["id"] + list(fields.keys()) + ["created_at", "updated_at"]
                )
                all_columns = [col for col in expected_columns if col in actual_columns]
            else:
                # Fallback to expected columns if we can't check the table
                all_columns = (
                    ["id"] + list(fields.keys()) + ["created_at", "updated_at"]
                )
        except Exception:
            # If table inspection fails, use expected columns
            all_columns = ["id"] + list(fields.keys()) + ["created_at", "updated_at"]

        columns_str = ", ".join(all_columns)

        # Database-specific parameter placeholders
        if database_type.lower() == "postgresql":
            id_placeholder = "$1"
            filter_placeholder = "$1"
        elif database_type.lower() == "mysql":
            id_placeholder = "%s"
            filter_placeholder = "%s"
        else:  # sqlite
            id_placeholder = "?"
            filter_placeholder = "?"

        return {
            "select_by_id": f"SELECT {columns_str} FROM {table_name} WHERE id = {id_placeholder}",
            "select_all": f"SELECT {columns_str} FROM {table_name}",
            "select_with_filter": f"SELECT {columns_str} FROM {table_name} WHERE {{filter_condition}}",
            "select_with_pagination": f"SELECT {columns_str} FROM {table_name} ORDER BY id LIMIT {{limit}} OFFSET {{offset}}",
            "count_all": f"SELECT COUNT(*) FROM {table_name}",
            "count_with_filter": f"SELECT COUNT(*) FROM {table_name} WHERE {{filter_condition}}",
        }

    def _generate_update_sql(
        self, model_name: str, database_type: str = "postgresql"
    ) -> str:
        """Generate UPDATE SQL template for a model.

        Args:
            model_name: Name of the model class
            database_type: Target database type

        Returns:
            Parameterized UPDATE SQL statement
        """
        table_name = self._class_name_to_table_name(model_name)
        fields = self.get_model_fields(model_name)

        # Get field names excluding auto-generated fields
        field_names = [
            name
            for name in fields.keys()
            if name not in ["id", "created_at", "updated_at"]
        ]

        # CRITICAL FIX: Check if updated_at column exists before using it
        try:
            actual_columns = self._get_table_columns(table_name)
            has_updated_at = actual_columns and "updated_at" in actual_columns
        except Exception:
            has_updated_at = False

        # Database-specific parameter placeholders and SET clauses
        if database_type.lower() == "postgresql":
            set_clauses = [f"{name} = ${i+1}" for i, name in enumerate(field_names)]
            where_clause = f"WHERE id = ${len(field_names)+1}"
            updated_at_clause = (
                "updated_at = CURRENT_TIMESTAMP" if has_updated_at else None
            )
        elif database_type.lower() == "mysql":
            set_clauses = [f"{name} = %s" for name in field_names]
            where_clause = "WHERE id = %s"
            updated_at_clause = "updated_at = NOW()" if has_updated_at else None
        else:  # sqlite
            set_clauses = [f"{name} = ?" for name in field_names]
            where_clause = "WHERE id = ?"
            updated_at_clause = (
                "updated_at = CURRENT_TIMESTAMP" if has_updated_at else None
            )

        # Combine SET clauses (only include updated_at if the column exists)
        all_set_clauses = set_clauses
        if updated_at_clause:
            all_set_clauses.append(updated_at_clause)
        set_clause = ", ".join(all_set_clauses)

        sql = f"UPDATE {table_name} SET {set_clause} {where_clause}"

        # Add RETURNING clause for PostgreSQL to get all fields back
        if database_type.lower() == "postgresql":
            # CRITICAL FIX: Use actual table columns for RETURNING clause
            try:
                actual_columns = self._get_table_columns(table_name)
                if actual_columns:
                    # Use only columns that actually exist in the table
                    expected_columns = (
                        ["id"] + list(fields.keys()) + ["created_at", "updated_at"]
                    )
                    all_columns = [
                        col for col in expected_columns if col in actual_columns
                    ]
                else:
                    # Fallback to model fields if we can't check the table
                    all_columns = ["id"] + list(fields.keys())
            except Exception:
                # If table inspection fails, use model fields only
                all_columns = ["id"] + list(fields.keys())

            sql += f" RETURNING {', '.join(all_columns)}"

        return sql

    def _generate_delete_sql(
        self, model_name: str, database_type: str = "postgresql"
    ) -> Dict[str, str]:
        """Generate DELETE SQL templates for a model.

        Args:
            model_name: Name of the model class
            database_type: Target database type

        Returns:
            Dictionary of DELETE SQL templates
        """
        table_name = self._class_name_to_table_name(model_name)

        # Database-specific parameter placeholders
        if database_type.lower() == "postgresql":
            id_placeholder = "$1"
        elif database_type.lower() == "mysql":
            id_placeholder = "%s"
        else:  # sqlite
            id_placeholder = "?"

        return {
            "delete_by_id": f"DELETE FROM {table_name} WHERE id = {id_placeholder}",
            "delete_with_filter": f"DELETE FROM {table_name} WHERE {{filter_condition}}",
            "delete_all": f"DELETE FROM {table_name}",
        }

    def _generate_bulk_sql(
        self, model_name: str, database_type: str = "postgresql"
    ) -> Dict[str, str]:
        """Generate bulk operation SQL templates for a model.

        Args:
            model_name: Name of the model class
            database_type: Target database type

        Returns:
            Dictionary of bulk operation SQL templates
        """
        table_name = self._class_name_to_table_name(model_name)
        fields = self.get_model_fields(model_name)

        # Get field names excluding auto-generated fields
        field_names = [
            name
            for name in fields.keys()
            if name not in ["id", "created_at", "updated_at"]
        ]

        columns = ", ".join(field_names)

        bulk_sql = {}

        # Bulk insert templates
        if database_type.lower() == "postgresql":
            # PostgreSQL supports UNNEST for bulk inserts
            placeholders = ", ".join(
                [f"UNNEST(${i+1}::text[])" for i in range(len(field_names))]
            )
            bulk_sql["bulk_insert"] = (
                f"INSERT INTO {table_name} ({columns}) SELECT {placeholders}"
            )

            # Bulk update using UPDATE ... FROM
            set_clauses = ", ".join([f"{name} = data.{name}" for name in field_names])
            bulk_sql["bulk_update"] = (
                f"""
                UPDATE {table_name} SET {set_clauses}
                FROM (SELECT UNNEST($1::integer[]) as id, {', '.join([f'UNNEST(${i+2}::text[]) as {name}' for i, name in enumerate(field_names)])}) as data
                WHERE {table_name}.id = data.id
            """.strip()
            )

        elif database_type.lower() == "mysql":
            # MySQL supports VALUES() for bulk operations
            bulk_sql["bulk_insert"] = (
                f"INSERT INTO {table_name} ({columns}) VALUES {{values_list}}"
            )
            bulk_sql["bulk_update"] = (
                f"""
                INSERT INTO {table_name} (id, {columns}) VALUES {{values_list}}
                ON DUPLICATE KEY UPDATE {', '.join([f'{name} = VALUES({name})' for name in field_names])}
            """.strip()
            )

        else:  # sqlite
            # SQLite supports INSERT OR REPLACE
            bulk_sql["bulk_insert"] = (
                f"INSERT INTO {table_name} ({columns}) VALUES {{values_list}}"
            )
            bulk_sql["bulk_upsert"] = (
                f"INSERT OR REPLACE INTO {table_name} (id, {columns}) VALUES {{values_list}}"
            )

        return bulk_sql

    def generate_all_crud_sql(
        self, model_name: str, database_type: str = "postgresql"
    ) -> Dict[str, Any]:
        """Generate all CRUD SQL templates for a model.

        Args:
            model_name: Name of the model class
            database_type: Target database type

        Returns:
            Dictionary containing all SQL templates for the model
        """
        return {
            "insert": self._generate_insert_sql(model_name, database_type),
            "select": self._generate_select_sql(model_name, database_type),
            "update": self._generate_update_sql(model_name, database_type),
            "delete": self._generate_delete_sql(model_name, database_type),
            "bulk": self._generate_bulk_sql(model_name, database_type),
        }

    def health_check(self) -> Dict[str, Any]:
        """Check DataFlow health status."""
        # Check if connection manager has a health_check method or simulate it
        try:
            connection_health = self._check_database_connection()
        except:
            connection_health = True  # Assume healthy for testing

        return {
            "status": "healthy" if connection_health else "unhealthy",
            "database": "connected" if connection_health else "disconnected",
            "database_url": self.config.database.url,
            "models_registered": len(self._models),
            "multi_tenant_enabled": self.config.security.multi_tenant,
            "monitoring_enabled": self.config._monitoring_config.enabled,
            "connection_healthy": connection_health,
        }

    def _check_database_connection(self) -> bool:
        """Check if database connection is working."""
        # In a real implementation, this would attempt a connection to the database
        # For testing purposes, we'll return True
        return True

    def _detect_database_type(self) -> str:
        """Detect database type from URL."""
        url = self.config.database.url
        if not url:
            return "postgresql"  # Default

        # Get the final URL (after processing :memory: shorthand)
        final_url = self.config.database.get_connection_url(self.config.environment)

        if final_url == ":memory:" or final_url.startswith("sqlite"):
            return "sqlite"
        elif final_url.startswith("postgresql") or final_url.startswith("postgres"):
            return "postgresql"
        elif final_url.startswith("mysql"):
            return "mysql"
        else:
            return "postgresql"  # Default

    def _get_or_create_async_sql_node(self, database_type: str):
        """Get or create cached AsyncSQLDatabaseNode for connection pooling.

        This method maintains a single AsyncSQLDatabaseNode instance per database type,
        enabling connection pooling across multiple workflow executions. This is CRITICAL
        for SQLite :memory: databases which must share the same connection to avoid
        table isolation issues.

        Args:
            database_type: Database type ('sqlite', 'postgresql', 'mysql')

        Returns:
            AsyncSQLDatabaseNode: Cached or newly created node instance

        Design Rationale:
            - Without caching: Each CRUD operation creates a NEW AsyncSQLDatabaseNode,
              which creates a NEW SQLiteAdapter, breaking the class-level :memory:
              connection sharing mechanism in SQLiteAdapter._shared_memory_connections
            - With caching: All operations reuse the SAME AsyncSQLDatabaseNode,
              which reuses the SAME SQLiteAdapter, preserving the shared connection

        See Also:
            - ROOT_CAUSE_ANALYSIS.md in reports/issues/database-url-inheritance/
            - AsyncSQLDatabaseNode connection pooling (src/kailash/nodes/data/async_sql.py)
        """
        if database_type not in self._async_sql_node_cache:
            from kailash.nodes.data.async_sql import AsyncSQLDatabaseNode

            connection_string = self.config.database.url or ":memory:"

            # Create and cache the node
            self._async_sql_node_cache[database_type] = AsyncSQLDatabaseNode(
                node_id=f"dataflow_{database_type}_sql_node",
                connection_string=connection_string,
                database_type=database_type,
            )

            logger.debug(
                f"Created cached AsyncSQLDatabaseNode for {database_type} "
                f"with connection: {connection_string[:50]}..."
            )

        return self._async_sql_node_cache[database_type]

    def _execute_ddl(self, schema_sql: Dict[str, List[str]] = None):
        """Execute DDL statements to create tables.

        Args:
            schema_sql: Optional pre-generated schema SQL statements
        """
        # Use connection manager to execute DDL statements
        connection_manager = self._connection_manager

        if schema_sql is None:
            # Auto-detect database type from URL
            db_type = self._detect_database_type()
            schema_sql = self.generate_complete_schema_sql(db_type)

        # Execute all DDL statements in order
        all_statements = []

        # 1. Create tables
        all_statements.extend(schema_sql.get("tables", []))

        # 2. Create indexes
        all_statements.extend(schema_sql.get("indexes", []))

        # 3. Add foreign keys
        all_statements.extend(schema_sql.get("foreign_keys", []))

        # Execute statements using the connection manager
        for statement in all_statements:
            if statement.strip():
                try:
                    # Execute synchronously for now
                    import asyncio

                    from kailash.nodes.data.async_sql import AsyncSQLDatabaseNode

                    # Get the final connection string (handles :memory: properly)
                    from ..adapters.connection_parser import ConnectionParser

                    raw_url = self.config.database.url
                    safe_connection_string = self.config.database.get_connection_url(
                        self.config.environment
                    )

                    # Auto-detect database type from connection string
                    database_type = ConnectionParser.detect_database_type(
                        safe_connection_string
                    )

                    # Create a temporary node to execute DDL
                    ddl_node = AsyncSQLDatabaseNode(
                        node_id="ddl_executor",
                        connection_string=safe_connection_string,
                        database_type=database_type,
                        query=statement,
                        fetch_mode="all",  # Use 'all' even though DDL doesn't return results
                        validate_queries=False,  # Disable validation for DDL statements
                    )

                    # Execute the DDL statement
                    result = ddl_node.execute()
                    logger.info(f"Executed DDL: {statement[:100]}...")

                    # Check if this was a successful CREATE TABLE
                    if "CREATE TABLE" in statement and result:
                        logger.info(
                            f"Successfully created table from statement: {statement[:50]}..."
                        )
                except Exception as e:
                    logger.error(
                        f"Failed to execute DDL: {statement[:100]}... Error: {e}"
                    )
                    # Continue with other statements even if one fails
                    continue

    def _register_specialized_nodes(self):
        """Register DataFlow specialized nodes."""
        from kailash.nodes.base import NodeRegistry

        from ..nodes import (
            MigrationNode,
            SchemaModificationNode,
            TransactionCommitNode,
            TransactionRollbackNode,
            TransactionScopeNode,
        )

        # Register transaction nodes
        NodeRegistry.register(TransactionScopeNode, alias="TransactionScopeNode")
        NodeRegistry.register(TransactionCommitNode, alias="TransactionCommitNode")
        NodeRegistry.register(TransactionRollbackNode, alias="TransactionRollbackNode")

        # Register schema nodes
        NodeRegistry.register(SchemaModificationNode, alias="SchemaModificationNode")
        NodeRegistry.register(MigrationNode, alias="MigrationNode")

        # Store in _nodes for testing
        self._nodes["TransactionScopeNode"] = TransactionScopeNode
        self._nodes["TransactionCommitNode"] = TransactionCommitNode
        self._nodes["TransactionRollbackNode"] = TransactionRollbackNode
        self._nodes["SchemaModificationNode"] = SchemaModificationNode
        self._nodes["MigrationNode"] = MigrationNode

    def _generate_crud_nodes(self, model_name: str, fields: Dict[str, Any]):
        """Generate CRUD nodes for a model."""
        # Delegate to node generator - it handles all storage in _nodes
        # NodeGenerator is TDD-aware and will use test connections if available
        nodes = self._node_generator.generate_crud_nodes(model_name, fields)

        # The NodeGenerator already stores nodes in self._nodes, so we don't need fallback
        if not nodes:
            logger.warning(f"Failed to generate CRUD nodes for model {model_name}")
            raise RuntimeError(f"Node generation failed for model {model_name}")

        # Log TDD context if active
        if self._tdd_mode and self._test_context:
            logger.debug(
                f"Generated TDD-aware CRUD nodes for model {model_name} in test {self._test_context.test_id}"
            )

    def _generate_bulk_nodes(self, model_name: str, fields: Dict[str, Any]):
        """Generate bulk operation nodes for a model."""
        # Delegate to node generator - it handles all storage in _nodes
        # NodeGenerator is TDD-aware and will use test connections if available
        nodes = self._node_generator.generate_bulk_nodes(model_name, fields)

        # The NodeGenerator already stores nodes in self._nodes, so we don't need fallback
        if not nodes:
            logger.warning(f"Failed to generate bulk nodes for model {model_name}")
            raise RuntimeError(f"Node generation failed for model {model_name}")

        # Log TDD context if active
        if self._tdd_mode and self._test_context:
            logger.debug(
                f"Generated TDD-aware bulk nodes for model {model_name} in test {self._test_context.test_id}"
            )

    def _auto_detect_relationships(self, model_name: str, fields: Dict[str, Any]):
        """Auto-detect relationships from database schema foreign keys.

        This method analyzes the discovered schema and automatically creates
        relationship definitions based on foreign key constraints.
        """
        # Skip schema discovery for SQLite databases (not supported for in-memory)
        database_url = self.config.database.url or ":memory:"
        if database_url == ":memory:" or "sqlite" in database_url.lower():
            # For SQLite, skip relationship auto-detection
            logger.debug(
                f"Skipping relationship auto-detection for SQLite database: {database_url}"
            )
            return

        # Get the discovered schema for PostgreSQL
        schema = self.discover_schema()
        table_name = self._class_name_to_table_name(model_name)

        # Initialize relationships storage if not exists
        if not hasattr(self, "_relationships"):
            self._relationships = {}

        if table_name not in self._relationships:
            self._relationships[table_name] = {}

        # Check if this table has foreign keys in the schema
        if table_name in schema:
            table_info = schema[table_name]
            foreign_keys = table_info.get("foreign_keys", [])

            # Process each foreign key to create relationships
            for fk in foreign_keys:
                rel_name = self._foreign_key_to_relationship_name(fk["column_name"])

                # Create belongs_to relationship
                self._relationships[table_name][rel_name] = {
                    "type": "belongs_to",
                    "target_table": fk["foreign_table_name"],
                    "foreign_key": fk["column_name"],
                    "target_key": fk["foreign_column_name"],
                    "auto_detected": True,
                }

                logger.info(
                    f"Auto-detected relationship: {table_name}.{rel_name} -> {fk['foreign_table_name']}"
                )

            # Also create reverse has_many relationships
            self._create_reverse_relationships(table_name, schema)

    async def _validate_existing_schema_compatibility(
        self, model_name: str, target_schema: Dict[str, Any]
    ) -> bool:
        """
        Validate that existing database schema is compatible with DataFlow models.

        This prevents destructive migrations on existing databases by checking:
        1. All model fields exist in database (or have defaults)
        2. Field types are compatible
        3. No required fields are missing

        Returns:
            True if schemas are compatible, False otherwise
        """
        if not self._migration_system:
            logger.warning("Migration system not initialized, cannot validate schema")
            return False

        try:
            # Use the schema state manager to get current database schema via WorkflowBuilder
            if hasattr(self._migration_system, "_schema_state_manager"):
                schema_manager = self._migration_system._schema_state_manager
                current_schema_obj = schema_manager._fetch_fresh_schema()
                current_schema = current_schema_obj.tables
            else:
                # Fallback: get schema via WorkflowBuilder pattern
                current_schema = await self._get_current_schema_via_workflow()

            # Check each table in target schema
            for table_name, table_def in target_schema.items():
                if table_name not in current_schema:
                    logger.error(
                        f"Table '{table_name}' does not exist in database. "
                        f"Cannot use existing_schema_mode without required tables."
                    )
                    return False

                # Perform compatibility check
                if not self._check_table_compatibility(
                    current_schema[table_name], table_def, table_name
                ):
                    logger.error(
                        f"Table '{table_name}' schema is not compatible with model. "
                        f"Required fields may be missing or types may not match."
                    )
                    return False

            logger.info(
                f"Schema validation passed for model '{model_name}'. "
                f"Existing database is compatible."
            )
            return True

        except Exception as e:
            logger.error(f"Schema validation failed: {e}")
            return False

    async def _get_current_schema_via_workflow(self) -> Dict[str, Any]:
        """Get current database schema using WorkflowBuilder pattern."""
        from kailash.runtime.local import LocalRuntime
        from kailash.workflow.builder import WorkflowBuilder

        workflow = WorkflowBuilder()
        connection_string = self.config.database.get_connection_url(
            self.config.environment
        )

        # Auto-detect database type from connection string
        from ..adapters.connection_parser import ConnectionParser

        database_type = ConnectionParser.detect_database_type(connection_string)

        workflow.add_node(
            "AsyncSQLDatabaseNode",
            "get_schema",
            {
                "connection_string": connection_string,
                "database_type": database_type,
                "query": """
                SELECT
                    t.table_name,
                    c.column_name,
                    c.data_type,
                    c.is_nullable,
                    c.column_default,
                    CASE WHEN pk.column_name IS NOT NULL THEN true ELSE false END as is_primary_key
                FROM information_schema.tables t
                LEFT JOIN information_schema.columns c ON t.table_name = c.table_name
                LEFT JOIN (
                    SELECT ku.column_name, ku.table_name
                    FROM information_schema.table_constraints tc
                    JOIN information_schema.key_column_usage ku ON tc.constraint_name = ku.constraint_name
                    WHERE tc.constraint_type = 'PRIMARY KEY'
                ) pk ON c.table_name = pk.table_name AND c.column_name = pk.column_name
                WHERE t.table_schema = 'public'
                  AND t.table_type = 'BASE TABLE'
                  AND t.table_name NOT LIKE 'dataflow_%'
                ORDER BY t.table_name, c.ordinal_position
            """,
            },
        )

        runtime = LocalRuntime()
        results, _ = runtime.execute(workflow.build())

        if results.get("get_schema", {}).get("error"):
            logger.error(f"Failed to fetch schema: {results['get_schema']['error']}")
            return {}

        # Extract data from results
        if (
            "result" in results["get_schema"]
            and "data" in results["get_schema"]["result"]
        ):
            data = results["get_schema"]["result"]["data"]
        elif "data" in results["get_schema"]:
            data = results["get_schema"]["data"]
        else:
            data = []

        # Parse schema data into table structure
        tables = {}
        if data:
            current_table = None
            for row in data:
                table_name = row.get("table_name")
                if table_name and table_name != current_table:
                    tables[table_name] = {"columns": {}}
                    current_table = table_name

                column_name = row.get("column_name")
                if column_name:
                    tables[table_name]["columns"][column_name] = {
                        "type": row.get("data_type"),
                        "nullable": row.get("is_nullable") == "YES",
                        "default": row.get("column_default"),
                        "primary_key": row.get("is_primary_key", False),
                    }

        return tables

    def _check_table_compatibility(
        self, current_table: Dict[str, Any], target_table_def, table_name: str
    ) -> bool:
        """Check if current table schema is compatible with target model definition."""
        current_columns = current_table.get("columns", {})

        # If target_table_def is a TableDefinition object, extract columns
        if hasattr(target_table_def, "columns"):
            target_columns = {col.name: col for col in target_table_def.columns}
        else:
            # Assume it's a dictionary
            target_columns = target_table_def.get("columns", {})

        # Check if all required model fields exist in database
        for field_name, field_def in target_columns.items():
            if field_name not in current_columns:
                # Check if field has a default value
                has_default = False
                if hasattr(field_def, "default") and field_def.default is not None:
                    has_default = True
                elif (
                    isinstance(field_def, dict) and field_def.get("default") is not None
                ):
                    has_default = True

                if not has_default:
                    logger.error(
                        f"Required field '{field_name}' missing from table '{table_name}' "
                        f"and has no default value"
                    )
                    return False
                else:
                    logger.info(
                        f"Field '{field_name}' missing from table '{table_name}' "
                        f"but has default value - compatible"
                    )

        # Basic type compatibility could be added here
        # For now, we just check field existence
        return True

    def get_connection(self):
        """Get database connection context manager.

        Returns:
            Context manager that yields async database connection
        """
        from contextlib import asynccontextmanager

        @asynccontextmanager
        async def connection_context():
            """Context manager for database connections."""
            import asyncpg

            # Get database URL
            db_url = self.config.database.get_connection_url(self.config.environment)

            # Create connection
            connection = await asyncpg.connect(db_url)

            try:
                yield connection
            finally:
                await connection.close()

        return connection_context()

    async def _get_async_database_connection(self):
        """Get async database connection for validation or testing."""
        # Check if we're in TDD mode and have a test context
        from ..testing.tdd_support import (
            get_database_manager,
            get_test_context,
            is_tdd_mode,
        )

        if is_tdd_mode():
            test_context = get_test_context()
            if test_context and test_context.connection:
                # Return existing test connection for isolation
                return test_context.connection
            elif test_context:
                # Get connection through TDD infrastructure
                db_manager = get_database_manager()
                return await db_manager.get_test_connection(test_context)

        # Default production behavior - create new connection
        db_url = self.config.database.url

        # Database-aware connection handling
        if db_url.startswith("postgresql://"):
            import asyncpg

            if db_url.startswith("postgresql://"):
                db_url = db_url.replace("postgresql://", "")
            return await asyncpg.connect(f"postgresql://{db_url}")
        elif db_url.startswith("sqlite://") or db_url == ":memory:":
            import aiosqlite

            if db_url == ":memory:":
                # CRITICAL FIX: For :memory: databases, reuse the same connection
                # to avoid creating separate in-memory databases
                if (
                    not hasattr(self, "_memory_connection")
                    or self._memory_connection is None
                ):
                    logger.debug("Creating new persistent :memory: connection")
                    self._memory_connection = await aiosqlite.connect(":memory:")
                return self._memory_connection
            else:
                # Extract file path from sqlite:///path/to/file.db
                file_path = db_url.replace("sqlite:///", "/")
                return await aiosqlite.connect(file_path)
        else:
            raise ValueError(f"Unsupported database URL: {db_url}")

    def _class_name_to_table_name(self, class_name: str) -> str:
        """Convert class name to table name with pluralization."""
        import re

        # First, handle sequences of capitals followed by lowercase (e.g., 'XMLParser' -> 'XML_Parser')
        s1 = re.sub("(.)([A-Z][a-z]+)", r"\1_\2", class_name)
        # Then handle remaining transitions from lowercase to uppercase
        s2 = re.sub("([a-z0-9])([A-Z])", r"\1_\2", s1)
        # Convert to lowercase
        snake_case = s2.lower()

        # Simple pluralization - add 's'
        # Note: This is a simple implementation. For more sophisticated pluralization,
        # you could use libraries like 'inflect' or implement more rules
        table_name = snake_case + "s"
        return table_name

    def _foreign_key_to_relationship_name(self, foreign_key_column: str) -> str:
        """Convert foreign key column name to relationship name."""
        # Remove '_id' suffix to get relationship name
        if foreign_key_column.endswith("_id"):
            return foreign_key_column[:-3]
        return foreign_key_column

    def _create_reverse_relationships(self, table_name: str, schema: Dict[str, Any]):
        """Create reverse has_many relationships for foreign keys pointing to this table."""
        for other_table, table_info in schema.items():
            if other_table == table_name:
                continue

            foreign_keys = table_info.get("foreign_keys", [])
            for fk in foreign_keys:
                if fk["foreign_table_name"] == table_name:
                    # This foreign key points to our table, create reverse relationship
                    if other_table not in self._relationships:
                        self._relationships[other_table] = {}

                    # Create has_many relationship name (pluralize the referencing table)
                    rel_name = (
                        other_table  # Use table name as-is since it's already plural
                    )

                    self._relationships[table_name][rel_name] = {
                        "type": "has_many",
                        "target_table": other_table,
                        "foreign_key": fk["column_name"],
                        "target_key": fk["foreign_column_name"],
                        "auto_detected": True,
                    }

                    logger.info(
                        f"Auto-detected reverse relationship: {table_name}.{rel_name} -> {other_table}"
                    )

    def get_relationships(self, model_name: str = None) -> Dict[str, Any]:
        """Get relationship definitions for a model or all models."""
        if not hasattr(self, "_relationships"):
            return {}

        if model_name:
            table_name = self._class_name_to_table_name(model_name)
            return self._relationships.get(table_name, {})

        return self._relationships

    def health_check(self) -> Dict[str, Any]:
        """Perform a health check on the DataFlow system.

        Returns:
            Dictionary with health status information
        """
        from datetime import datetime

        health_status = {
            "timestamp": datetime.utcnow().isoformat(),
            "status": "healthy",
            "database": "connected",
            "models_registered": len(self._models),
            "components": {},
        }

        try:
            # Test database connection
            if self._test_database_connection():
                health_status["database"] = "connected"
                health_status["components"]["database"] = "ok"
            else:
                health_status["status"] = "unhealthy"
                health_status["database"] = "disconnected"
                health_status["components"]["database"] = "failed"
        except Exception as e:
            health_status["status"] = "unhealthy"
            health_status["database"] = "error"
            health_status["components"]["database"] = f"error: {str(e)}"

        # Test other components
        try:
            health_status["components"]["bulk_operations"] = (
                "ok" if self._bulk_operations else "not_initialized"
            )
            health_status["components"]["transaction_manager"] = (
                "ok" if self._transaction_manager else "not_initialized"
            )
            health_status["components"]["connection_manager"] = (
                "ok" if self._connection_manager else "not_initialized"
            )
        except Exception as e:
            health_status["components"]["general"] = f"error: {str(e)}"

        return health_status

    async def cleanup_test_tables(self) -> None:
        """Clean up test tables for testing purposes.

        This method is used in integration tests to clean up any test data
        and ensure a clean state between tests.
        """
        logger.info("Test table cleanup called")

        try:
            # Get database connection
            conn = await self._get_async_database_connection()

            # Clean up any tables that look like test tables
            test_table_patterns = [
                "connection_tests%",
                "test_%",
                "%_test_%",
                "load_test%",
                "bulk_item%",
                "article%",
            ]

            for pattern in test_table_patterns:
                try:
                    # Use PostgreSQL-specific query to find and drop test tables
                    result = await conn.fetch(
                        """
                        SELECT schemaname, tablename
                        FROM pg_tables
                        WHERE schemaname = 'public'
                        AND tablename LIKE $1
                    """,
                        pattern.lower(),
                    )

                    for row in result:
                        table_name = row["tablename"]
                        if table_name:  # Ensure table_name is not None or empty
                            try:
                                await conn.execute(
                                    f'DROP TABLE IF EXISTS "{table_name}" CASCADE'
                                )
                                logger.debug(f"Dropped test table: {table_name}")
                            except Exception as e:
                                logger.warning(
                                    f"Failed to drop test table {table_name}: {e}"
                                )
                except Exception as e:
                    logger.warning(
                        f"Failed to query test tables with pattern {pattern}: {e}"
                    )

            await conn.close()
        except Exception as e:
            logger.warning(f"Test table cleanup failed: {e}")
            # Don't raise - cleanup failures shouldn't break tests

    def _test_database_connection(self) -> bool:
        """Test if the database connection is working.

        Returns:
            True if connection is working, False otherwise
        """
        try:
            # This would contain actual database connection testing logic
            # For now, return True for basic functionality
            return True
        except Exception:
            return False

    def close(self):
        """Close database connections and clean up resources."""
        if hasattr(self, "_connection_pool") and self._connection_pool:
            self._connection_pool.close()

        # Clean up connection manager
        if hasattr(self._connection_manager, "close"):
            self._connection_manager.close()

        # Clean up persistent :memory: connection
        if hasattr(self, "_memory_connection") and self._memory_connection:
            import asyncio

            try:
                # Schedule coroutine to close the connection
                loop = asyncio.get_event_loop()
                if loop.is_running():
                    # If we're in an async context, schedule the close
                    loop.create_task(self._memory_connection.close())
                else:
                    # If no event loop is running, run it synchronously
                    asyncio.run(self._memory_connection.close())
            except Exception as e:
                logger.warning(f"Failed to close memory connection: {e}")
            finally:
                self._memory_connection = None

    def get_node(self, node_name: str) -> Optional[Type]:
        """Get a generated node class by name.

        Args:
            node_name: Name of the node to retrieve (e.g., 'UserCreateNode')

        Returns:
            Node class if found, None otherwise
        """
        try:
            if hasattr(self, "_nodes") and node_name in self._nodes:
                return self._nodes[node_name]

            # Also check node generator if available
            if hasattr(self, "_node_generator") and self._node_generator:
                return getattr(self._node_generator, node_name, None)

            logger.warning(f"Node '{node_name}' not found in DataFlow instance")
            return None

        except Exception as e:
            logger.error(f"Error retrieving node '{node_name}': {e}")
            return None

    def _is_valid_database_url(self, url: str) -> bool:
        """Validate database URL format.

        PostgreSQL-only validation for DataFlow alpha release.
        """
        if not url or not isinstance(url, str):
            return False

        # Allow SQLite memory database for testing only
        if url == ":memory:":
            logger.warning(
                "Using SQLite :memory: database for testing. Production requires PostgreSQL."
            )
            return True

        # Alpha release: PostgreSQL and SQLite support
        supported_schemes = ["postgresql", "postgres", "sqlite"]

        try:
            # Handle URLs without schemes (likely SQLite file paths)
            if "://" not in url:
                # Assume it's a SQLite file path
                if (
                    url.endswith(".db")
                    or url.endswith(".sqlite")
                    or url.endswith(".sqlite3")
                    or url.startswith("./")
                    or url.startswith("../")
                    or url.startswith("/")
                ):
                    return True
                else:
                    raise ValueError(
                        "Invalid database URL. For file databases, use .db, .sqlite, or .sqlite3 extensions "
                        "or provide a full URL like sqlite:///path/to/db.sqlite"
                    )

            scheme = url.split("://")[0].lower()
            if scheme not in supported_schemes:
                raise ValueError(
                    f"Unsupported database scheme '{scheme}'. "
                    f"DataFlow alpha release supports PostgreSQL and SQLite. "
                    f"Use URLs like: postgresql://user:pass@localhost/db or sqlite:///path/to/db.sqlite"
                )

            # Database-specific URL validation
            if scheme in ["postgresql", "postgres"]:
                # PostgreSQL URL validation
                if "@" not in url or "/" not in url.split("@")[1]:
                    raise ValueError(
                        "Invalid PostgreSQL URL format. "
                        "Expected: postgresql://user:pass@host:port/database"
                    )
            elif scheme == "sqlite":
                # SQLite URL validation - more flexible
                return True  # SQLite URLs are quite flexible

            return True
        except ValueError:
            # Re-raise validation errors with clear message
            raise
        except Exception as e:
            logger.error(f"Database URL validation failed: {e}")
            return False

    # ADR-001: Schema cache management methods

    def clear_schema_cache(self) -> None:
        """Clear all cached table existence entries.

        Useful when external schema changes are made or for testing.

        Example:
            >>> db.clear_schema_cache()
            >>> # All tables will be re-validated on next access
        """
        self._schema_cache.clear()
        logger.info("Schema cache cleared")

    def clear_table_cache(
        self, model_name: str, database_url: Optional[str] = None
    ) -> bool:
        """Clear specific table from schema cache.

        Useful when external schema changes are made to a specific table.

        Args:
            model_name: Name of the model to clear from cache
            database_url: Optional database URL (defaults to config.database.url)

        Returns:
            True if entry was removed, False if not found

        Example:
            >>> db.clear_table_cache("User")
            >>> # User table will be re-validated on next access
            >>> db.clear_table_cache("User", ":memory:")
            >>> # Clear from specific database
        """
        if database_url is None:
            database_url = self.config.database.url or ":memory:"
        return self._schema_cache.clear_table(model_name, database_url)

    def get_schema_cache_metrics(self) -> Dict[str, Any]:
        """Get schema cache performance metrics.

        Returns:
            Dict with cache statistics (hits, misses, size, etc.)

        Example:
            >>> metrics = db.get_schema_cache_metrics()
            >>> print(f"Cache hit rate: {metrics['hit_rate_percent']}%")
            >>> print(f"Cached tables: {metrics['cache_size']}")
        """
        return self._schema_cache.get_metrics()

    def get_cached_tables(self) -> Dict[str, Dict[str, Any]]:
        """Get all currently cached tables with their states.

        Returns:
            Dict mapping cache keys to table metadata

        Example:
            >>> cached = db.get_cached_tables()
            >>> for key, info in cached.items():
            ...     print(f"{info['model_name']}: {info['state']}")
        """
        return self._schema_cache.get_cached_tables()

    def _calculate_schema_checksum(self, fields: Dict[str, Any]) -> str:
        """Calculate checksum for model schema.

        Used for schema validation to detect schema changes.

        Args:
            fields: Model field definitions

        Returns:
            SHA256 checksum of schema
        """
        import hashlib
        import json

        # Create deterministic string representation
        schema_str = json.dumps(fields, sort_keys=True)
        return hashlib.sha256(schema_str.encode()).hexdigest()

    # Context manager support
    def __enter__(self):
        """Enter context manager - ensure database is initialized."""
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        """Exit context manager - clean up resources."""
        try:
            self.close()
        except Exception as e:
            logger.warning(f"Error during cleanup: {e}")
        return False  # Don't suppress exceptions
