"""
ZeroArt Fonts: Predefined ASCII art fonts
""" 

from typing import Dict, List
from .base import ZeroArtFont

class BlockFont(ZeroArtFont):
    """Block-style ASCII art font"""
    def __init__(self) -> None:
        super().__init__("block")
        self.letters: Dict[str, List[str]] = {
            ' ': ["   ", "   ", "   ", "   ", "   "]
        }
        self._populate_letters()

    def _populate_letters(self) -> None:
        """Populate the font with predefined letters"""
        block_letters = {
            'A': [
                "  █████  ",
                " ██   ██ ",
                " ███████ ",
                " ██   ██ ",
                " ██   ██ "
            ],
            'B': [
                " ██████  ",
                " ██   ██ ",
                " ██████  ",
                " ██   ██ ",
                " ██████  "
            ],
            'C': [
                " ██████  ",
                " ██      ",
                " ██      ",
                " ██      ",
                " ██████  "
            ],
            'D': [
                " ██████  ",
                " ██   ██ ",
                " ██   ██ ",
                " ██   ██ ",
                " ██████  "
            ],
            'E': [
                " ███████ ",
                " ██      ",
                " ██████  ",
                " ██      ",
                " ███████ "
            ],
            'F': [
                " ███████ ",
                " ██      ",
                " ██████  ",
                " ██      ",
                " ██      "
            ],
            'G': [
                " ██████  ",
                " ██      ",
                " ██  ███ ",
                " ██   ██ ",
                " ██████  "
            ],
            'H': [
                " ██   ██ ",
                " ██   ██ ",
                " ███████ ",
                " ██   ██ ",
                " ██   ██ "
            ],
            'I': [
                " ███████ ",
                "   ██   ",
                "   ██   ",
                "   ██   ",
                " ███████ "
            ],
            'J': [
                "      ██ ",
                "      ██ ",
                "      ██ ",
                " ██   ██ ",
                "  █████  "
            ],
            'K': [
                " ██   ██ ",
                " ██  ██  ",
                " ████   ",
                " ██  ██  ",
                " ██   ██ "
            ],
            'L': [
                " ██      ",
                " ██      ",
                " ██      ",
                " ██      ",
                " ███████ "
            ],
            'M': [
                " ██   ██ ",
                " ███ ███ ",
                " ██ █ ██ ",
                " ██   ██ ",
                " ██   ██ "
            ],
            'N': [
                " ██   ██ ",
                " ███  ██ ",
                " ██ █ ██ ",
                " ██  ███ ",
                " ██   ██ "
            ],
            'O': [
                "  █████  ",
                " ██   ██ ",
                " ██   ██ ",
                " ██   ██ ",
                "  █████  "
            ],
            'P': [
                " ██████  ",
                " ██   ██ ",
                " ██████  ",
                " ██      ",
                " ██      "
            ],
            'Q': [
                "  █████  ",
                " ██   ██ ",
                " ██   ██ ",
                "  ██ ██  ",
                "  ██████ "
            ],
            'R': [
                " ██████  ",
                " ██   ██ ",
                " ██████  ",
                " ██  ██  ",
                " ██   ██ "
            ],
            'S': [
                " ██████  ",
                " ██      ",
                " ██████  ",
                "      ██ ",
                " ██████  "
            ],
            'T': [
                " ███████ ",
                "   ██   ",
                "   ██   ",
                "   ██   ",
                "   ██   "
            ],
            'U': [
                " ██   ██ ",
                " ██   ██ ",
                " ██   ██ ",
                " ██   ██ ",
                "  █████  "
            ],
            'V': [
                " ██   ██ ",
                " ██   ██ ",
                " ██   ██ ",
                "  ██ ██  ",
                "   ███   "
            ],
            'W': [
                " ██   ██ ",
                " ██   ██ ",
                " ██ █ ██ ",
                " ███ ███ ",
                "  ██ ██  "
            ],
            'X': [
                " ██   ██ ",
                "  ██ ██  ",
                "   ███   ",
                "  ██ ██  ",
                " ██   ██ "
            ],
            'Y': [
                " ██   ██ ",
                "  ██ ██  ",
                "   ███   ",
                "   ███   ",
                "   ███   "
            ],
            'Z': [
                " ███████ ",
                "     ██  ",
                "    ██   ",
                "   ██    ",
                " ███████ "
            ]
        }
        
        self.letters.update(block_letters)

class SlantFont(ZeroArtFont):
    """Slanted ASCII art font"""
    def __init__(self) -> None:
        super().__init__("slant")
        self.letters: Dict[str, List[str]] = {
            ' ': ["   ", "   ", "   ", "   ", "   "]
        }
        self._populate_letters()

    def _populate_letters(self) -> None:
        """Populate the font with predefined letters"""
        slant_letters = {
            'A': [
                "    /\\    ",
                "   /  \\   ",
                "  / /\\ \\  ",
                " / ____ \\ ",
                "/_/    \\_\\"
            ],
            'B': [
                " ____  ",
                "|  _ \\ ",
                "| |_) |",
                "|  _ < ",
                "| |_) |",
                "|____/ "
            ],
            'C': [
                "  _____  ",
                " / ____|  ",
                "| |       ",
                "| |       ",
                " \\_____|  "
            ],
            'D': [
                " _____   ",
                "|  __ \\  ",
                "| |  | | ",
                "| |  | | ",
                "| |__| | ",
                "|_____/  "
            ],
            'E': [
                " ______  ",
                "|  ____|  ",
                "| |__     ",
                "|  __|    ",
                "| |____   ",
                "|______|  "
            ],
            'F': [
                " ______  ",
                "|  ____|  ",
                "| |__     ",
                "|  __|    ",
                "| |       ",
                "|_|       "
            ],
            'G': [
                "  _____  ",
                " / ____|  ",
                "| |  __   ",
                "| | |_ |  ",
                "| |__| |  ",
                " \\_____|  "
            ],
            'H': [
                " _    _  ",
                "| |  | | ",
                "| |__| | ",
                "|  __  | ",
                "|_|  |_| "
            ],
            'I': [
                " _____ ",
                "|_   _|",
                "  | |  ",
                "  | |  ",
                " _| |_ ",
                "|_____|"
            ],
            'J': [
                "    ___ ",
                "   |_  |",
                "     | |",
                " _   | |",
                "| |__| |",
                " \\____/ "
            ],
            'K': [
                " _  __ ",
                "| |/ / ",
                "| ' /  ",
                "|  <   ",
                "| . \\  ",
                "|_|\\_\\ "
            ],
            'L': [
                " _      ",
                "| |     ",
                "| |     ",
                "| |     ",
                "| |____ ",
                "|______|"
            ],
            'M': [
                " __  __ ",
                "|  \\/  |",
                "| |\\/| |",
                "| |  | |",
                "|_|  |_|",
                "        "
            ],
            'N': [
                " _   _ ",
                "| \\ | |",
                "|  \\| |",
                "| . ` |",
                "| |\\  |",
                "|_| \\_|"
            ],
            'O': [
                "  _____  ",
                " / ____|  ",
                "| |  __   ",
                "| | |_ |  ",
                "| |__| |  ",
                " \\_____|  "
            ],
            'P': [
                " _____   ",
                "|  __ \\  ",
                "| |__) | ",
                "|  ___/  ",
                "| |      ",
                "|_|      "
            ],
            'Q': [
                "  _____  ",
                " / ____|  ",
                "| |  __   ",
                "| | |_ |  ",
                "| |__| |  ",
                " \\____/\\ "
            ],
            'R': [
                " _____   ",
                "|  __ \\  ",
                "| |__) | ",
                "|  _  /  ",
                "| | \\ \\  ",
                "|_|  \\_\\ "
            ],
            'S': [
                "  _____  ",
                " / ____|  ",
                "| (___    ",
                " \\___ \\   ",
                " ____) |  ",
                "|_____/   "
            ],
            'T': [
                " ______  ",
                "|_   _|  ",
                "  | |    ",
                "  | |    ",
                "  | |    ",
                "  |_|    "
            ],
            'U': [
                " _    _  ",
                "| |  | | ",
                "| |  | | ",
                "| |  | | ",
                "| |__| | ",
                " \\____/  "
            ],
            'V': [
                " __     __ ",
                " \\ \\   / / ",
                "  \\ \\_/ /  ",
                "   \\   /   ",
                "    \\_/    "
            ],
            'W': [
                " __    __  ",
                " \\ \\  / /  ",
                "  \\ \\/ /   ",
                "   \\  /    ",
                "    \\/     "
            ],
            'X': [
                " __  __ ",
                " \\ \\/ / ",
                "  \\  /  ",
                "  /  \\  ",
                " /_/\\_\\ "
            ],
            'Y': [
                " _     _ ",
                "| |   | |",
                "| |   | |",
                "| |   | |",
                "|_|   |_|"
            ],
            'Z': [
                " ______  ",
                "|__  / | ",
                "  / /| | ",
                " / /_| | ",
                "/____|_| "
            ]
        }
        
        self.letters.update(slant_letters)

# Add more custom fonts here
class NeonFont(ZeroArtFont):
    """Neon-style ASCII art font"""
    def __init__(self) -> None:
        super().__init__("neon")
        self.letters: Dict[str, List[str]] = {
            ' ': ["   ", "   ", "   ", "   ", "   "]
        }
        self._populate_letters()

    def _populate_letters(self) -> None:
        """Populate neon-style letters"""
        neon_letters = {
            'A': [
                "  ░█████░  ",
                " ██░░░░██ ",
                " ████████ ",
                " ██░░░░██ ",
                " ██░░░░██ "
            ],
            'B': [
                " ████████ ",
                " ██░░░░██ ",
                " ████████ ",
                " ██░░░░██ ",
                " ████████ "
            ],
            'C': [
                "  ░██████ ",
                " ██░░░░   ",
                " ██       ",
                " ██░░░░   ",
                "  ░██████ "
            ],
            'D': [
                " ████████ ",
                " ██░░░░██ ",
                " ██░░░░██ ",
                " ██░░░░██ ",
                " ████████ "
            ],
            'E': [
                " ████████ ",
                " ██░░░░   ",
                " ████████ ",
                " ██░░░░   ",
                " ████████ "
            ],
            'F': [
                " ████████ ",
                " ██░░░░   ",
                " ████████ ",
                " ██░░░░   ",
                " ██░░░░   "
            ],
            'G': [
                "  ░██████ ",
                " ██░░░░   ",
                " ██  ████ ",
                " ██░░░░██ ",
                "  ░██████ "
            ],
            'H': [
                " ██░░░░██ ",
                " ██░░░░██ ",
                " ████████ ",
                " ██░░░░██ ",
                " ██░░░░██ "
            ],
            'I': [
                " ████████ ",
                "   ██░░   ",
                "   ██░░   ",
                "   ██░░   ",
                " ████████ "
            ],
            'J': [
                "      ██░ ",
                "      ██░ ",
                "      ██░ ",
                " ██░░░░██ ",
                "  ██████  "
            ],
            'K': [
                " ██░░░░██ ",
                " ██░░██░  ",
                " ████░░   ",
                " ██░░██░  ",
                " ██░░░░██ "
            ],
            'L': [
                " ██░░░░   ",
                " ██░░░░   ",
                " ██░░░░   ",
                " ██░░░░   ",
                " ████████ "
            ],
            'M': [
                " ██░░░░██ ",
                " ███░░███ ",
                " ██░█░███ ",
                " ██░░░░██ ",
                " ██░░░░██ "
            ],
            'N': [
                " ██░░░░██ ",
                " ███░░░██ ",
                " ██░█░░██ ",
                " ██░░█░██ ",
                " ██░░░███ "
            ],
            'O': [
                "  ██████  ",
                " ██░░░░██ ",
                " ██░░░░██ ",
                " ██░░░░██ ",
                "  ██████  "
            ],
            'P': [
                " ████████ ",
                " ██░░░░██ ",
                " ████████ ",
                " ██░░░░   ",
                " ██░░░░   "
            ],
            'Q': [
                "  ██████  ",
                " ██░░░░██ ",
                " ██░░░░██ ",
                " ██░░██░  ",
                "  ██████░ "
            ],
            'R': [
                " ████████ ",
                " ██░░░░██ ",
                " ████████ ",
                " ██░░██░  ",
                " ██░░░░██ "
            ],
            'S': [
                "  ██████░ ",
                " ██░░░░   ",
                "  ██████  ",
                "     ░░██ ",
                " ████████ "
            ],
            'T': [
                " ████████ ",
                "   ██░░   ",
                "   ██░░   ",
                "   ██░░   ",
                "   ██░░   "
            ],
            'U': [
                " ██░░░░██ ",
                " ██░░░░██ ",
                " ██░░░░██ ",
                " ██░░░░██ ",
                "  ██████  "
            ],
            'V': [
                " ██░░░░██ ",
                " ██░░░░██ ",
                " ██░░░░██ ",
                "  ██░░██  ",
                "   ████   "
            ],
            'W': [
                " ██░░░░██ ",
                " ██░░░░██ ",
                " ██░█░░██ ",
                " ███░███░ ",
                "  ██░░██  "
            ],
            'X': [
                " ██░░░░██ ",
                "  ██░░██  ",
                "   ████   ",
                "  ██░░██  ",
                " ██░░░░██ "
            ],
            'Y': [
                " ██░░░░██ ",
                "  ██░░██  ",
                "   ████   ",
                "   ██░░   ",
                "   ██░░   "
            ],
            'Z': [
                " ████████ ",
                "     ██░  ",
                "    ██░   ",
                "   ██░    ",
                " ████████ "
            ]
        }
        
        self.letters.update(neon_letters)

class CyberFont(ZeroArtFont):
    """Cyberpunk-style ASCII art font"""
    def __init__(self) -> None:
        super().__init__("cyber")
        self.letters: Dict[str, List[str]] = {
            ' ': ["   ", "   ", "   ", "   ", "   "]
        }
        self._populate_letters()

    def _populate_letters(self) -> None:
        """Populate cyberpunk-style letters"""
        cyber_letters = {
            'A': [
                "  ▓█▓▓▓▓  ",
                " ▓▓   ▓▓ ",
                " ▓▓▓▓▓▓▓ ",
                " ▓▓   ▓▓ ",
                " ▓▓   ▓▓ "
            ],
            'B': [
                " ▓▓▓▓▓▓▓ ",
                " ▓▓   ▓▓ ",
                " ▓▓▓▓▓▓▓ ",
                " ▓▓   ▓▓ ",
                " ▓▓▓▓▓▓▓ "
            ],
            'C': [
                " ▓▓▓▓▓▓▓ ",
                " ▓▓      ",
                " ▓▓      ",
                " ▓▓      ",
                " ▓▓▓▓▓▓▓ "
            ],
            'D': [
                " ▓▓▓▓▓▓  ",
                " ▓▓   ▓▓ ",
                " ▓▓   ▓▓ ",
                " ▓▓   ▓▓ ",
                " ▓▓▓▓▓▓  "
            ],
            'E': [
                " ▓▓▓▓▓▓▓ ",
                " ▓▓      ",
                " ▓▓▓▓▓▓  ",
                " ▓▓      ",
                " ▓▓▓▓▓▓▓ "
            ],
            'F': [
                " ▓▓▓▓▓▓▓ ",
                " ▓▓      ",
                " ▓▓▓▓▓▓  ",
                " ▓▓      ",
                " ▓▓      "
            ],
            'G': [
                " ▓▓▓▓▓▓▓ ",
                " ▓▓      ",
                " ▓▓  ▓▓▓ ",
                " ▓▓   ▓▓ ",
                " ▓▓▓▓▓▓▓ "
            ],
            'H': [
                " ▓▓   ▓▓ ",
                " ▓▓   ▓▓ ",
                " ▓▓▓▓▓▓▓ ",
                " ▓▓   ▓▓ ",
                " ▓▓   ▓▓ "
            ],
            'I': [
                " ▓▓▓▓▓▓▓ ",
                "   ▓▓    ",
                "   ▓▓    ",
                "   ▓▓    ",
                " ▓▓▓▓▓▓▓ "
            ],
            'J': [
                "     ▓▓  ",
                "     ▓▓  ",
                "     ▓▓  ",
                " ▓▓  ▓▓  ",
                " ▓▓▓▓▓   "
            ],
            'K': [
                " ▓▓   ▓▓ ",
                " ▓▓  ▓▓  ",
                " ▓▓▓▓▓   ",
                " ▓▓  ▓▓  ",
                " ▓▓   ▓▓ "
            ],
            'L': [
                " ▓▓      ",
                " ▓▓      ",
                " ▓▓      ",
                " ▓▓      ",
                " ▓▓▓▓▓▓▓ "
            ],
            'M': [
                " ▓▓   ▓▓ ",
                " ▓▓▓ ▓▓▓ ",
                " ▓▓▓▓▓▓▓ ",
                " ▓▓ ▓ ▓▓ ",
                " ▓▓   ▓▓ "
            ],
            'N': [
                " ▓▓   ▓▓ ",
                " ▓▓▓  ▓▓ ",
                " ▓▓ ▓ ▓▓ ",
                " ▓▓  ▓▓▓ ",
                " ▓▓   ▓▓ "
            ],
            'O': [
                " ▓▓▓▓▓▓▓ ",
                " ▓▓   ▓▓ ",
                " ▓▓   ▓▓ ",
                " ▓▓   ▓▓ ",
                " ▓▓▓▓▓▓▓ "
            ],
            'P': [
                " ▓▓▓▓▓▓▓ ",
                " ▓▓   ▓▓ ",
                " ▓▓▓▓▓▓▓ ",
                " ▓▓      ",
                " ▓▓      "
            ],
            'Q': [
                " ▓▓▓▓▓▓  ",
                " ▓▓   ▓▓ ",
                " ▓▓   ▓▓ ",
                " ▓▓  ▓▓▓ ",
                " ▓▓▓▓▓ ▓ "
            ],
            'R': [
                " ▓▓▓▓▓▓  ",
                " ▓▓   ▓▓ ",
                " ▓▓▓▓▓▓  ",
                " ▓▓  ▓▓  ",
                " ▓▓   ▓▓ "
            ],
            'S': [
                " ▓▓▓▓▓▓▓ ",
                " ▓▓      ",
                " ▓▓▓▓▓▓▓ ",
                "      ▓▓ ",
                " ▓▓▓▓▓▓▓ "
            ],
            'T': [
                " ▓▓▓▓▓▓▓ ",
                "   ▓▓    ",
                "   ▓▓    ",
                "   ▓▓    ",
                "   ▓▓    "
            ],
            'U': [
                " ▓▓   ▓▓ ",
                " ▓▓   ▓▓ ",
                " ▓▓   ▓▓ ",
                " ▓▓   ▓▓ ",
                " ▓▓▓▓▓▓▓ "
            ],
            'V': [
                " ▓▓   ▓▓ ",
                " ▓▓   ▓▓ ",
                " ▓▓   ▓▓ ",
                "  ▓▓ ▓▓  ",
                "   ▓▓▓   "
            ],
            'W': [
                " ▓▓   ▓▓ ",
                " ▓▓   ▓▓ ",
                " ▓▓ ▓ ▓▓ ",
                " ▓▓▓ ▓▓▓ ",
                "  ▓▓ ▓▓  "
            ],
            'X': [
                " ▓▓   ▓▓ ",
                "  ▓▓ ▓▓  ",
                "   ▓▓▓   ",
                "  ▓▓ ▓▓  ",
                " ▓▓   ▓▓ "
            ],
            'Y': [
                " ▓▓   ▓▓ ",
                "  ▓▓ ▓▓ ",
                "   ▓▓   ",
                "   ▓▓   ",
                "   ▓▓   "
            ],
            'Z': [
                " ▓▓▓▓▓▓▓ ",
                "     ▓▓  ",
                "    ▓▓   ",
                "   ▓▓    ",
                " ▓▓▓▓▓▓▓ "
            ]
        }
        
        self.letters.update(cyber_letters)

class DottedFont(ZeroArtFont):
    """Dotted-style ASCII art font"""
    def __init__(self) -> None:
        super().__init__("dotted")
        self.letters: Dict[str, List[str]] = {
            ' ': ["   ", "   ", "   ", "   ", "   "]
        }
        self._populate_letters()

    def _populate_letters(self) -> None:
        """Populate dotted-style letters"""
        dotted_letters = {
            'A': [
                " ........ ",
                " ::    :: ",
                " :::::::: ",
                " ::    :: ",
                " ::    :: "
            ],
            'B': [
                " ::::::: ",
                " ::    :: ",
                " ::::::: ",
                " ::    :: ",
                " ::::::: "
            ],
            'C': [
                " ::::::: ",
                " ::      ",
                " ::      ",
                " ::      ",
                " ::::::: "
            ],
            'D': [
                " ::::::: ",
                " ::    :: ",
                " ::    :: ",
                " ::    :: ",
                " ::::::: "
            ],
            'E': [
                " :::::::: ",
                " ::       ",
                " ::::::   ",
                " ::       ",
                " :::::::: "
            ],
            'F': [
                " :::::::: ",
                " ::       ",
                " ::::::   ",
                " ::       ",
                " ::       "
            ],
            'G': [
                " ::::::: ",
                " ::      ",
                " :: :::: ",
                " ::   :: ",
                " ::::::: "
            ],
            'H': [
                " ::    :: ",
                " ::    :: ",
                " :::::::: ",
                " ::    :: ",
                " ::    :: "
            ],
            'I': [
                " :::::::: ",
                "    ::    ",
                "    ::    ",
                "    ::    ",
                " :::::::: "
            ],
            'J': [
                "       :: ",
                "       :: ",
                "       :: ",
                " ::    :: ",
                "  ::::::  "
            ],
            'K': [
                " ::    :: ",
                " ::   ::  ",
                " :::::    ",
                " ::  ::   ",
                " ::    :: "
            ],
            'L': [
                " ::       ",
                " ::       ",
                " ::       ",
                " ::       ",
                " :::::::: "
            ],
            'M': [
                " ::    :: ",
                " :::  ::: ",
                " :: :: :: ",
                " ::    :: ",
                " ::    :: "
            ],
            'N': [
                " ::    :: ",
                " :::   :: ",
                " :: :: :: ",
                " ::  :::: ",
                " ::    :: "
            ],
            'O': [
                "  ::::::  ",
                " ::    :: ",
                " ::    :: ",
                " ::    :: ",
                "  ::::::  "
            ],
            'P': [
                " ::::::: ",
                " ::    :: ",
                " ::::::: ",
                " ::       ",
                " ::       "
            ],
            'Q': [
                "  ::::::  ",
                " ::    :: ",
                " ::    :: ",
                " ::  :::: ",
                "  ::::  : "
            ],
            'R': [
                " ::::::: ",
                " ::    :: ",
                " ::::::: ",
                " ::  ::   ",
                " ::    :: "
            ],
            'S': [
                " ::::::: ",
                " ::      ",
                " ::::::: ",
                "      :: ",
                " ::::::: "
            ],
            'T': [
                " :::::::: ",
                "    ::    ",
                "    ::    ",
                "    ::    ",
                "    ::    "
            ],
            'U': [
                " ::    :: ",
                " ::    :: ",
                " ::    :: ",
                " ::    :: ",
                "  ::::::  "
            ],
            'V': [
                " ::    :: ",
                " ::    :: ",
                " ::    :: ",
                "  ::  ::  ",
                "    ::    "
            ],
            'W': [
                " ::    :: ",
                " ::    :: ",
                " :: :: :: ",
                " :::  ::: ",
                " ::    :: "
            ],
            'X': [
                " ::    :: ",
                "  ::  ::  ",
                "    ::    ",
                "  ::  ::  ",
                " ::    :: "
            ],
            'Y': [
                " ::    :: ",
                "  ::  ::  ",
                "    ::    ",
                "    ::    ",
                "    ::    "
            ],
            'Z': [
                " :::::::: ",
                "      ::  ",
                "    ::    ",
                "  ::      ",
                " :::::::: "
            ]
        }
        
        self.letters.update(dotted_letters)

class ShadowFont(ZeroArtFont):
    """Shadow-style ASCII art font"""
    def __init__(self) -> None:
        super().__init__("shadow")
        self.letters: Dict[str, List[str]] = {
            ' ': ["   ", "   ", "   ", "   ", "   "]
        }
        self._populate_letters()

    def _populate_letters(self) -> None:
        """Populate shadow-style letters"""
        shadow_letters = {
            'A': [
                "   ██    ",
                "  ████   ",
                " ██  ██  ",
                "████████ ",
                "██    ██▓",
                "       ▓▓"
            ],
            'B': [
                "██████   ",
                "██   ██  ",
                "███████  ",
                "██    ██ ",
                "███████  ",
                " ▓▓▓▓▓▓▓ "
            ],
            'C': [
                " ██████  ",
                "██    ██ ",
                "██       ",
                "██    ██ ",
                " ██████  ",
                "  ▓▓▓▓▓▓ "
            ],
            'D': [
                "██████   ",
                "██   ██  ",
                "██    ██ ",
                "██    ██ ",
                "██████   ",
                "▓▓▓▓▓▓   "
            ],
            'E': [
                "████████ ",
                "██       ",
                "██████   ",
                "██       ",
                "████████ ",
                "▓▓▓▓▓▓▓▓ "
            ],
            'F': [
                "████████ ",
                "██       ",
                "██████   ",
                "██       ",
                "██       ",
                "▓▓       "
            ],
            'G': [
                " ██████  ",
                "██       ",
                "██   ███ ",
                "██    ██ ",
                " ██████  ",
                "  ▓▓▓▓▓▓ "
            ],
            'H': [
                "██    ██ ",
                "██    ██ ",
                "████████ ",
                "██    ██ ",
                "██    ██ ",
                "▓▓    ▓▓ "
            ],
            'I': [
                "████████ ",
                "   ██    ",
                "   ██    ",
                "   ██    ",
                "████████ ",
                "▓▓▓▓▓▓▓▓ "
            ],
            'J': [
                "     ██  ",
                "     ██  ",
                "     ██  ",
                "██   ██  ",
                " █████   ",
                "  ▓▓▓▓   "
            ],
            'K': [
                "██   ██  ",
                "██  ██   ",
                "█████    ",
                "██  ██   ",
                "██   ██  ",
                "▓▓   ▓▓  "
            ],
            'L': [
                "██       ",
                "██       ",
                "██       ",
                "██       ",
                "████████ ",
                "▓▓▓▓▓▓▓▓ "
            ],
            'M': [
                "██    ██ ",
                "███  ███ ",
                "████████ ",
                "██ ██ ██ ",
                "██    ██ ",
                "▓▓    ▓▓ "
            ],
            'N': [
                "██    ██ ",
                "███   ██ ",
                "████  ██ ",
                "██ ██ ██ ",
                "██   ███ ",
                "▓▓    ▓▓ "
            ],
            'O': [
                " ██████  ",
                "██    ██ ",
                "██    ██ ",
                "██    ██ ",
                " ██████  ",
                "  ▓▓▓▓▓▓ "
            ],
            'P': [
                "███████  ",
                "██    ██ ",
                "███████  ",
                "██       ",
                "██       ",
                "▓▓       "
            ],
            'Q': [
                " ██████  ",
                "██    ██ ",
                "██    ██ ",
                "██ ▄▄ ██ ",
                " ██████  ",
                "  ▓▓▓▓▓▓▓"
            ],
            'R': [
                "███████  ",
                "██    ██ ",
                "███████  ",
                "██   ██  ",
                "██    ██ ",
                "▓▓    ▓▓ "
            ],
            'S': [
                " ██████  ",
                "██       ",
                " ██████  ",
                "      ██ ",
                "███████  ",
                "▓▓▓▓▓▓▓  "
            ],
            'T': [
                "████████ ",
                "   ██    ",
                "   ██    ",
                "   ██    ",
                "   ██    ",
                "   ▓▓    "
            ],
            'U': [
                "██    ██ ",
                "██    ██ ",
                "██    ██ ",
                "██    ██ ",
                " ██████  ",
                "  ▓▓▓▓▓▓ "
            ],
            'V': [
                "██    ██ ",
                "██    ██ ",
                "██    ██ ",
                " ██  ██  ",
                "  ████   ",
                "   ▓▓    "
            ],
            'W': [
                "██    ██ ",
                "██    ██ ",
                "██ ██ ██ ",
                "████████ ",
                " ██  ██  ",
                "  ▓  ▓▓  "
            ],
            'X': [
                "██    ██ ",
                " ██  ██  ",
                "  ████   ",
                " ██  ██  ",
                "██    ██ ",
                "▓▓    ▓▓ "
            ],
            'Y': [
                "██    ██ ",
                " ██  ██  ",
                "  ████   ",
                "   ██    ",
                "   ██    ",
                "   ▓▓    "
            ],
            'Z': [
                "████████ ",
                "     ██  ",
                "   ██    ",
                " ██      ",
                "████████ ",
                "▓▓▓▓▓▓▓▓ "
            ]
        }
        
        self.letters.update(shadow_letters)

class ThreeDFont(ZeroArtFont):
    """3D-style ASCII art font"""
    def __init__(self) -> None:
        super().__init__("3d")
        self.letters: Dict[str, List[str]] = {
            ' ': ["   ", "   ", "   ", "   ", "   "]
        }
        self._populate_letters()

    def _populate_letters(self) -> None:
        """Populate 3D-style letters"""
        three_d_letters = {
            'A': [
                "   /\\   ",
                "  /  \\  ",
                " / /\\ \\ ",
                "/_/----\\_\\",
                "  /    \\  "
            ],
            'B': [
                " ____  ",
                "|  _ \\ ",
                "| |_) |",
                "|  _ < ",
                "| |_) |",
                "|____/ "
            ],
            'C': [
                "  ____  ",
                " / ___| ",
                "| |     ",
                "| |___  ",
                " \\____| "
            ],
            'D': [
                " ____  ",
                "|  _ \\ ",
                "| | | |",
                "| | | |",
                "| |/ / ",
                "|___/  "
            ],
            'E': [
                " ____  ",
                "|  ___| ",
                "| |__   ",
                "|  __|  ",
                "| |___  ",
                "|_____| "
            ],
            'F': [
                " ____  ",
                "|  ___| ",
                "| |__   ",
                "|  __|  ",
                "| |     ",
                "|_|     "
            ],
            'G': [
                "  ____  ",
                " / ___| ",
                "| |  _  ",
                "| |_| | ",
                " \\___| |",
                "      |_|"
            ],
            'H': [
                " _    _ ",
                "| |  | |",
                "| |__| |",
                "|  __  |",
                "| |  | |",
                "|_|  |_|"
            ],
            'I': [
                " ___ ",
                "|_  |",
                "  | |",
                "  | |",
                "  | |",
                "  |_|"
            ],
            'J': [
                "   ___ ",
                "  |_  |",
                "    | |",
                "    | |",
                "/\\ | |",
                "\\/|_|"
            ],
            'K': [
                " _  __ ",
                "| |/ / ",
                "| ' /  ",
                "|  <   ",
                "| . \\  ",
                "|_|\\_\\ "
            ],
            'L': [
                " _     ",
                "| |    ",
                "| |    ",
                "| |    ",
                "| |___ ",
                "|_____|"
            ],
            'M': [
                " __  __ ",
                "|  \\/  |",
                "| |\\/| |",
                "| |  | |",
                "| |  | |",
                "|_|  |_|"
            ],
            'N': [
                " _   _ ",
                "| \\ | |",
                "|  \\| |",
                "| . ` |",
                "| |\\  |",
                "|_| \\_|"
            ],
            'O': [
                "  ____  ",
                " / __ \\ ",
                "| |  | |",
                "| |  | |",
                "| |__| |",
                " \\____/ "
            ],
            'P': [
                " ____  ",
                "|  _ \\ ",
                "| |_) |",
                "|  __/ ",
                "| |    ",
                "|_|    "
            ],
            'Q': [
                "  ____  ",
                " / __ \\ ",
                "| |  | |",
                "| |  | |",
                "| |__| |",
                " \\__\\_\\"
            ],
            'R': [
                " ____  ",
                "|  _ \\ ",
                "| |_) |",
                "|  _ < ",
                "| | \\ \\",
                "|_|  \\_\\"
            ],
            'S': [
                " ____  ",
                "/ ___| ",
                "\\___ \\ ",
                " ___) |",
                "|____/ "
            ],
            'T': [
                " _____ ",
                "|_   _|",
                "  | |  ",
                "  | |  ",
                "  | |  ",
                "  |_|  "
            ],
            'U': [
                " _    _ ",
                "| |  | |",
                "| |  | |",
                "| |  | |",
                "| |__| |",
                " \\____/ "
            ],
            'V': [
                "__     __",
                "\\ \\   / /",
                " \\ \\_/ / ",
                "  \\   /  ",
                "   \\_/   "
            ],
            'W': [
                "__        __",
                "\\ \\      / /",
                " \\ \\    / / ",
                "  \\ \\/\\/ /  ",
                "   \\_/\\_/   "
            ],
            'X': [
                "__  __",
                "\\ \\/ /",
                " \\  / ",
                " /  \\ ",
                "/_/\\_\\"
            ],
            'Y': [
                "__     __",
                "\\ \\   / /",
                " \\ \\_/ / ",
                "  \\   /  ",
                "   |_|   "
            ],
            'Z': [
                " _____",
                "|__  /",
                "  / / ",
                " / /_ ",
                "/____|"
            ]
        }
        
        self.letters.update(three_d_letters)

class ElectronicFont(ZeroArtFont):
    """Electronic-style ASCII art font"""
    def __init__(self) -> None:
        super().__init__("electronic")
        self.letters: Dict[str, List[str]] = {
            ' ': ["   ", "   ", "   ", "   ", "   "]
        }
        self._populate_letters()

    def _populate_letters(self) -> None:
        """Populate electronic-style letters"""
        electronic_letters = {
            'A': [
                "  /\\  ",
                " /--\\ ",
                "/----\\",
                "/      \\"
            ],
            'B': [
                "|--- ",
                "|=== ",
                "|--- ",
                "|___ "
            ],
            'C': [
                " /===",
                "|    ",
                "|    ",
                " \\___"
            ],
            'D': [
                "|==\\  ",
                "|  | | ",
                "|  | | ",
                "|==/  "
            ],
            'E': [
                "|===",
                "|---",
                "|   ",
                "|___"
            ],
            'F': [
                "|===",
                "|---",
                "|   ",
                "|   "
            ],
            'G': [
                " /===",
                "|    ",
                "| ---",
                " \\-|_"
            ],
            'H': [
                "|    | ",
                "|----| ",
                "|    | ",
                "|    | "
            ],
            'I': [
                "===",
                " | ",
                " | ",
                "==="
            ],
            'J': [
                "  ===",
                "   | ",
                "   | ",
                "_--/ "
            ],
            'K': [
                "|  / ",
                "|--< ",
                "|  \\ ",
                "|   \\"
            ],
            'L': [
                "|    ",
                "|    ",
                "|    ",
                "|___ "
            ],
            'M': [
                "/\\  /\\",
                "| \\/ |",
                "|    | ",
                "|    | "
            ],
            'N': [
                "/\\   | ",
                "| \\  | ",
                "|  \\ | ",
                "|   \\| "
            ],
            'O': [
                " /==\\ ",
                "|    | ",
                "|    | ",
                " \\==/ "
            ],
            'P': [
                "|===",
                "|---",
                "|   ",
                "|   "
            ],
            'Q': [
                " /==\\ ",
                "|    | ",
                "|  \\| ",
                " \\==/\\"
            ],
            'R': [
                "|===",
                "|---",
                "|  \\",
                "|   \\"
            ],
            'S': [
                "/===",
                "\\---",
                "---/",
                "___/"
            ],
            'T': [
                "=====",
                "  |  ",
                "  |  ",
                "  |  "
            ],
            'U': [
                "|    | ",
                "|    | ",
                "|    | ",
                " \\__/ "
            ],
            'V': [
                "\\    /",
                " \\  / ",
                "  \\/  ",
                "   V   "
            ],
            'W': [
                "\\      /",
                " \\    / ",
                "  \\/\\/  ",
                "   V  V   "
            ],
            'X': [
                "\\  /",
                " \\/ ",
                " /\\ ",
                "/  \\"
            ],
            'Y': [
                "\\  /",
                " \\/ ",
                "  |  ",
                "  |  "
            ],
            'Z': [
                "===",
                "  /",
                " / ",
                "==="
            ]
        }
        
        self.letters.update(electronic_letters)

class IsometricFont(ZeroArtFont):
    """Isometric-style ASCII art font"""
    def __init__(self) -> None:
        super().__init__("isometric")
        self.letters: Dict[str, List[str]] = {
            ' ': ["     ", "     ", "     ", "     ", "     "]
        }
        self._populate_letters()

    def _populate_letters(self) -> None:
        """Populate isometric-style letters"""
        isometric_letters = {
            'A': [
                "   /\\   ",
                "  /  \\  ",
                " /----\\ ",
                "/      \\"
            ],
            'B': [
                "|---\\",
                "|    |",
                "|---/ ",
                "|____/"
            ],
            'C': [
                "  /---",
                " /    ",
                "|     ",
                " \\___ "
            ],
            'D': [
                "|---\\",
                "|    |",
                "|    |",
                "|___/"
            ],
            'E': [
                "|----",
                "|--- ",
                "|    ",
                "|____"
            ],
            'F': [
                "|----",
                "|--- ",
                "|    ",
                "|    "
            ],
            'G': [
                "  /---",
                " /    ",
                "|  --",
                " \\_|_"
            ],
            'H': [
                "|    | ",
                "|----| ",
                "|    | ",
                "|    | "
            ],
            'I': [
                "---",
                " | ",
                " | ",
                "---"
            ],
            'J': [
                "----",
                "   | ",
                "   | ",
                "_--/ "
            ],
            'K': [
                "|   /",
                "|--< ",
                "|   \\"
            ],
            'L': [
                "|    ",
                "|    ",
                "|    ",
                "|___ "
            ],
            'M': [
                "/\\  /\\",
                "| \\/ |",
                "|    | ",
                "|    | "
            ],
            'N': [
                "|\\  | ",
                "| \\ | ",
                "|  \\| ",
                "|   \\ "
            ],
            'O': [
                " /---\\ ",
                "|     | ",
                "|     | ",
                " \\---/ "
            ],
            'P': [
                "|---",
                "|--- ",
                "|    ",
                "|    "
            ],
            'Q': [
                " /---\\ ",
                "|     | ",
                "|   / | ",
                " \\--/\\"
            ],
            'R': [
                "|---",
                "|--- ",
                "|   \\",
                "|    \\"
            ],
            'S': [
                "/---",
                "\\---",
                "---/",
                "--/"
            ],
            'T': [
                "-----",
                "  |  ",
                "  |  ",
                "  |  "
            ],
            'U': [
                "|    | ",
                "|    | ",
                "|    | ",
                " \\__/ "
            ],
            'V': [
                "\\    /",
                " \\  / ",
                "  \\/  "
            ],
            'W': [
                "\\    /",
                " \\  / ",
                "  \\/\\/  ",
                "   V  V   "
            ],
            'X': [
                "\\  /",
                " \\/ ",
                " /\\ ",
                "/  \\"
            ],
            'Y': [
                "\\  /",
                " \\/ ",
                "  |  ",
                "  |  "
            ],
            'Z': [
                "----",
                "   /",
                "  / ",
                "----"
            ]
        }
        
        self.letters.update(isometric_letters)