# Mecsimcalc b0.1.9 documentation

This library is designed to provide a set of functions for handling and converting various types of data, such as base64 encoded data, Pandas DataFrames, and Pillow images.

- [GitHub Repository](https://github.com/MecSimCalc/MecSimCalc-utils)
- [PyPi Page](https://pypi.org/project/mecsimcalc/)

## General

### input_to_file

[**[Source]**](https://github.com/MecSimCalc/MecSimCalc-utils/blob/b0.1.9/mecsimcalc/file_utils/general_utils.py#L7C1-L66C1)

```python
input_to_file(input_file, metadata = False)
```

#### Description:

Converts a base64 encoded string into a file object and metadata

#### Arguments:

| Argument         | Type                | Description                                                |
| ---------------- | ------------------- | ---------------------------------------------------------- |
| **`input_file`** | **str**             | Base64 encoded string, prefixed with metadata              |
| **`metadata`**   | **bool** (optional) | Flag to return metadata with the file. (Defaults to False) |

#### Raises:

| Exception        | Description                                                                        |
| ---------------- | ---------------------------------------------------------------------------------- |
| **`ValueError`** | If the input string doesn't contain ';base64,' to separate metadata and file data. |

#### Returns:

| Return Type             | Description                                                              | Condition         |
| ----------------------- | ------------------------------------------------------------------------ | ----------------- |
| **`io.BytesIO`**        | The decoded file data (The thing you get when you open a file in Python) | metadata is False |
| **`(io.BytesIO, str)`** | The decoded file data and its metadata                                   | metadata is True  |

#### Example:

```python
>>> input_file = inputs['file']
>>> file, metadata = input_to_file(input_file, metadata = True)
>>> print(metadata)
data:image/jpeg;base64,
>>> type(file)
<class '_io.BytesIO'>
```

### metadata_to_filetype

[**[Source]**](https://github.com/MecSimCalc/MecSimCalc-utils/blob/b0.1.9/mecsimcalc/file_utils/general_utils.py#L68C1-L100C21)


```python
metadata_to_filetype(metadata):
```

#### Description:

Extracts the file type from the metadata

#### Arguments:

| Argument       | Type    | Description                                                                                   |
| -------------- | ------- | --------------------------------------------------------------------------------------------- |
| **`metadata`** | **str** | The metadata string in the form "Data:(MIME type);base64,"(returned from **`input_to_file`**) |

#### Returns:

| Return Type | Description                 |
| ----------- | --------------------------- |
| **`str`**   | The file type (e.g. "jpeg") |

#### Example:

```python
>>> input_file = inputs['file']
>>> file, metadata = input_to_file(input_file, metadata = True)
>>> print(metadata)
data:image/jpeg;base64,
>>> download_file_type = metadata_to_filetype(metadata)
>>> print(download_file_type)
jpeg
```

## Text

### string_to_file

[**[Source]**](https://github.com/MecSimCalc/MecSimCalc-utils/blob/b0.1.9/mecsimcalc/file_utils/text_utils.py#L4C1-L63C85)

```python
string_to_file(
    text
    filename= "myfile",
    download_text = "Download File",
)
```

#### Description:

Generates a downloadable text file containing the given text

#### Arguments:

| Argument            | Type               | Description                                                              |
| ------------------- | ------------------ | ------------------------------------------------------------------------ |
| **`text`**          | **str**            | Text to be downloaded                                                    |
| **`filename`**      | **str** (optional) | Name of the download file. (Defaults to "myfile")                        |
| **`download_text`** | **str** (optional) | Text to be displayed as the download link. (Defaults to "Download File") |

#### Raises:

| Exception       | Description                        |
| --------------- | ---------------------------------- |
| **`TypeError`** | If the input text is not a string. |

#### Returns:

| Return Type | Description        |
| ----------- | ------------------ |
| **`str`**   | HTML download link |

#### Example:

#### Python

```python
>>> download_link = string_to_file("Hello World!")
>>> return {"download": download_link}
```

#### Jinja2

```python
# outputs.downloadLink is the html download link generated by the function
{{ outputs.download }}
```

## Spreadsheets

### file_to_dataframe

[**[Source]**](https://github.com/MecSimCalc/MecSimCalc-utils/blob/b0.1.9/mecsimcalc/file_utils/spreadsheet_utils.py#L9C1-L50C14)

```python
file_to_dataframe(file_data):
```

#### Description:

Converts a base64 encoded file data into a pandas DataFrame

#### Arguments:

| Argument        | Type           | Description                                       |
| --------------- | -------------- | ------------------------------------------------- |
| **`file_data`** | **io.BytesIO** | Decoded file data (e.g. from **`input_to_file`**) |

#### Raises:

| Exception                   | Description                                                                                                 |
| --------------------------- | ----------------------------------------------------------------------------------------------------------- |
| **`pd.errors.ParserError`** | If the file data cannot be converted to a DataFrame (i.e. file is not an Excel or CSV file or is corrupted) |

#### Returns:

| Return Type        | Description                      |
| ------------------ | -------------------------------- |
| **`pd.DataFrame`** | DataFrame created from file data |

#### Example:

```python
>>> input_file = inputs['file']
>>> decoded_file = input_to_file(input_file)
>>> df = file_to_dataframe(decoded_file)
>>> print(df)
   A  B  C
0  a  b  c
1  d  e  f
```

### input_to_dataframe

[**[Source]**](https://github.com/MecSimCalc/MecSimCalc-utils/blob/b0.1.9/mecsimcalc/file_utils/spreadsheet_utils.py#L53C1-L92C44)


```python
input_to_dataframe(file):
```

#### Description:

Converts a base64 encoded file data into a pandas DataFrame

#### Arguments:

| Argument            | Type     | Description                                                          |
| ------------------- | -------- | -------------------------------------------------------------------- |
| **`input_file`**    | **str**  | Base64 encoded file data                                             |
| **`get_file_type`** | **bool** | If True, the function also returns the file type (Defaults to False) |

#### Returns:

| Return Type               | Description                                      | Condition              |
| ------------------------- | ------------------------------------------------ | ---------------------- |
| **`pd.DataFrame`**        | DataFrame created from file data                 | get_file_type is False |
| **`(pd.DataFrame, str)`** | Tuple containing the DataFrame and the file type | get_file_type is True  |

#### Example:

```python
>>> input_file = inputs['file']
>>> df, file_type = input_to_dataframe(input_file, get_file_type = True)
>>> print(df)
   A  B  C
0  a  b  c
1  d  e  f
>>> print(file_type)
csv
```

### print_dataframe

[**[Source]**](https://github.com/MecSimCalc/MecSimCalc-utils/blob/b0.1.9/mecsimcalc/file_utils/spreadsheet_utils.py#L95C1-L186C39)


```python
print_dataframe(
    df,
    download = False,
    download_text = "Download Table",
    download_file_name = "mytable",
    download_file_type = "csv",
):
```

#### Description:

Creates an HTML table and a download link for a given DataFrame

#### Arguments:

| Argument                 | Type                | Description                                                              |
| ------------------------ | ------------------- | ------------------------------------------------------------------------ |
| **`df`**                 | **pd.DataFrame**    | DataFrame to be converted                                                |
| **`download`**           | **bool** (optional) | If True, function returns a download link (Defaults to False)            |
| **`download_text`**      | **str** (optional)  | Text to be displayed as the download link (Defaults to "Download Table") |
| **`download_file_name`** | **str** (optional)  | Name of file when downloaded (Defaults to "mytable")                     |
| **`download_file_type`** | **str** (optional)  | File type of downloaded file (Defaults to "csv")                         |

#### Returns:

| Return Type           | Description                      | Condition         |
| --------------------- | -------------------------------- | ----------------- |
| **`str`**             | HTML table                       | download is False |
| **`Tuple[str, str]`** | (HTML table, HTML download link) | download is True  |

#### Example:

#### Python Code:

```python
>>> input_file = inputs['file']
>>> df = input_to_dataframe(input_file)
>>> table, download = print_dataframe(df, download = True, download_file_name = "FunkyTable", download_text = "Download My Funky Table HERE!", download_file_type = "xlsx")

>>> return {
        "table":table,
        "download":download,
    }
```

#### Output using Jinja2 Template:

```python
# outputs.table is the HTML table
Displaying Table
{{ outputs.table }}

# outputs.download is the download link
Downloading Table
{{ outputs.download }}
```
## Tables

### table_to_dataframe

[**[Source]**](https://github.com/MecSimCalc/MecSimCalc-utils/blob/b0.1.9/mecsimcalc/file_utils/table_utils.py#L5C1-L45C1)


```python
table_to_dataframe(column_headers, rows) -> pd.DataFrame:
```

#### Description:

Create a DataFrame from given rows and column headers

#### Arguments:

| Argument             | Type                | Description                                                                     |
| -------------------- | ------------------- | ------------------------------------------------------------------------------- |
| **`column_headers`** | **List[str]**       | List of column headers                                                          |
| **`rows`**           | **List[List[str]]** | List of rows to be converted into a DataFrame. Each column is a list of strings |

#### Returns:

| Return Type        | Description                             |
| ------------------ | --------------------------------------- |
| **`pd.DataFrame`** | DataFrame created from headers and rows |

#### Example:

```python
>>> column_headers = ["A", "B", "C"]
>>> rows = [["a", "b", "c"], ["d", "e", "f"]]
>>> df = table_to_dataframe(column_headers, rows)
>>> print(df)
   A  B  C
0  a  b  c
1  d  e  f

```

### print_table

[**[Source]**](https://github.com/MecSimCalc/MecSimCalc-utils/blob/b0.1.9/mecsimcalc/file_utils/table_utils.py#L47C1-L80C58)


```python
print_table(column_headers, rows):
```

#### Description:

Creates an HTML table from given rows and column headers

#### Arguments:

| Argument             | Type                | Description                                                                  |
| -------------------- | ------------------- | ---------------------------------------------------------------------------- |
| **`column_headers`** | **List[str]**       | List of column headers                                                       |
| **`rows`**           | **List[List[str]]** | List of rows to be converted into a table. Each column is a list of strings  |
| **`index`**          | **bool** (optional) | Whether to use the first column as the DataFrame's index. (Defaults to True) |

#### Returns:

| Return Type | Description                              |
| ----------- | ---------------------------------------- |
| **`str`**   | HTML table created from rows and headers |

#### Example:

#### Python Code:

```python
>>> column_headers = ["A", "B", "C"]
>>> rows = [["a", "b", "c"], ["d", "e", "f"]]
>>> table = print_table(column_headers, rows)
>>> return {
        "table":table,
    }
```

#### Output using Jinja2 Template:

```python
# outputs.table is the HTML table
Displaying Table
{{ outputs.table }}
```

## Images

### file_to_PIL

[**[Source]**](https://github.com/MecSimCalc/MecSimCalc-utils/blob/b0.1.9/mecsimcalc/file_utils/image_utils.py#L23C1-L55C88)

```python
file_to_PIL(file):
```

#### Description:

Transforms a file into a Pillow Image object

#### Arguments:

| Argument   | Type    | Description                                     |
| ---------- | ------- | ----------------------------------------------- |
| **`file`** | **str** | Decoded file data (returned from input_to_file) |

#### Raises:

| Exception Type   | Description                             |
| ---------------- | --------------------------------------- |
| **`ValueError`** | If the file does not contain image data |

#### Returns:

| Return Type | Description         |
| ----------- | ------------------- |
| **`Image`** | Pillow Image object |

#### Example:

#### Python Code:

```python
>>> input_file = inputs['file']
>>> decoded_file = input_to_file(input_file)
>>> image = file_to_PIL(decoded_file)
>>> return {
        "image":image,
    }
```

#### Output using Jinja2 Template:

```python
# outputs.image is the Pillow Image object
Displaying Image
{{ outputs.image }}
```

### input_to_PIL

[**[Source]**](https://github.com/MecSimCalc/MecSimCalc-utils/blob/b0.1.9/mecsimcalc/file_utils/image_utils.py#L58C1-L109C1)

```python
input_to_PIL(input_file, get_file_type=False):
```

#### Description:

Converts a base64 encoded file data into a pillow image

#### Arguments:

| Argument            | Type     | Description                                                          |
| ------------------- | -------- | -------------------------------------------------------------------- |
| **`input_file`**    | **str**  | Base64 encoded file data                                             |
| **`get_file_type`** | **bool** | If True, the function also returns the file type (Defaults to False) |

#### Returns:

| Return Type                       | Description              | Condition              |
| --------------------------------- | ------------------------ | ---------------------- |
| **`PIL.Image.Image`**             | Pillow Image object      | get_file_type is False |
| **`Tuple[PIL.Image.Image, str]`** | (pillow image, metadata) | get_file_type is True  |

#### Example:

```python
>>> input_file = inputs['file']
>>> image, file_type = input_to_PIL(input_file, get_file_type=True)
>>> print(file_type)
jpeg
>>> type(image)
<class 'PIL.JpegImagePlugin.JpegImageFile'>
```

### print_image

[**[Source]**](https://github.com/MecSimCalc/MecSimCalc-utils/blob/b0.1.9/mecsimcalc/file_utils/image_utils.py#L110C1-L209C36)


```python
print_image(
    image,
    width = 200,
    height = 200,
    original_size = False,
    download = False,
    download_text = "Download Image",
    download_file_name= "myimg",
    download_file_type = "png",
):
```

#### Description:

Transforms a Pillow image into an HTML image, with an optional download link

#### Arguments:

| Argument                 | Type                | Description                                                                        |
| ------------------------ | ------------------- | ---------------------------------------------------------------------------------- |
| **`image`**              | **PIL.Image.Image** | Pillow image                                                                       |
| **`width`**              | **int** (optional)  | Output width of the image in pixels (Defaults to 200)                              |
| **`height`**             | **int** (optional)  | Output height of the image in pixels (Defaults to 200)                             |
| **`original_size`**      | **bool** (optional) | If True, the HTML image will be displayed in its original size (Defaults to False) |
| **`download`**           | **bool** (optional) | If True, function returns a download link (Defaults to False)                      |
| **`download_text`**      | **str** (optional)  | The text to be displayed on the download link (Defaults to "Download Image")       |
| **`download_file_name`** | **str** (optional)  | The name of the image file when downloaded (Defaults to "myimg")                   |
| **`download_file_type`** | **str** (optional)  | The file type of the image when downloaded (Defaults to "png")                     |

#### Returns:

| Return Type           | Description                 | Condition         |
| --------------------- | --------------------------- | ----------------- |
| **`str`**             | HTML image                  | download is False |
| **`Tuple[str, str]`** | (HTML image, download link) | download is True  |

#### Example:

#### Python Code:

```python
>>> input_file = inputs['file']
>>> image, metadata = input_to_PIL(input_file)
>>> html_image, download = print_image(image, original_size = True, download = True, download_text = "Download Image Here", download_file_name = "myimage", download_file_type = "jpeg")
>>> return {
        "image":html_image,
        "download":download,
    }
```

#### Output using Jinja2 Template:

```python
# outputs.image is the HTML image
Displaying Image
{{ outputs.image }}

# outputs.download is the download link
Downloading Image
{{ outputs.download }}
```

## Plots

### print_plot

[**[Source]**](https://github.com/MecSimCalc/MecSimCalc-utils/blob/b0.1.9/mecsimcalc/file_utils/plotting_utils.py#L13C1-L97C35)

```python
print_plot(
    plot_obj,
    width = 500,
    dpi= 100,
    download= False,
    download_text = "Download Plot",
    download_file_name = "myplot",
)
```

#### Description:

Converts a matplotlib.pyplot.axis or matplotlib.figure into an HTML image tag and optionally provides a download link for the image

#### Arguments:

| Argument                 | Type                | Description                                                                 |
| ------------------------ | ------------------- | --------------------------------------------------------------------------- |
| **`plot_obj`**           | **axes or figure**  | Matplotlib figure                                                           |
| **`width`**              | **int** (optional)  | Output width of the image in pixels (Defaults to 500)                       |
| **`dpi`**                | **int** (optional)  | Output dpi of the image in pixels (Defaults to 100)                         |
| **`download`**           | **bool** (optional) | If True, function returns a download link (Defaults to False)               |
| **`download_text`**      | **str** (optional)  | The text to be displayed on the download link (Defaults to "Download Plot") |
| **`download_file_name`** | **str** (optional)  | The name of the image file when downloaded (Defaults to "myplot")           |

#### Returns:

| Return Type           | Description                      | Condition         |
| --------------------- | -------------------------------- | ----------------- |
| **`str`**             | HTML image                       | download is False |
| **`Tuple[str, str]`** | (HTML image, HTML download link) | download is True  |

#### Example:

#### Python Code:

```python
>>> import matplotlib.pyplot as plt
>>> import numpy as np
>>> x = np.linspace(0, 2 * np.pi, 400)
>>> y = np.sin(x)
>>> fig, ax = plt.subplots()
>>> ax.plot(x, y)
>>> ax.set_title('A single plot')
>>> image, download = print_plot(fig, width = 500, dpi = 100, download = True, download_text = "Download Sin Function Plot", download_file_name = "sin(x)")
>>> return {
        "image":image,
        "download":download,
    }
```

#### Output using Jinja2 Template:

```python
# outputs.image is the HTML image
Displaying Image
{{ outputs.image }}

# outputs.download is the download link
Downloading Image
{{ outputs.download }}
```

### print_animation

[**[Source]**](https://github.com/MecSimCalc/MecSimCalc-utils/blob/b0.1.9/mecsimcalc/file_utils/plotting_utils.py#L99C1-L150C1)

```python
print_animation(ani: FuncAnimation, fps: int = 30, save_dir: str = "/tmp/temp_animation.gif") -> str:
```

#### Description:

Converts a matplotlib animation into an HTML image tag.

#### Arguments:

| Argument       | Type            | Description                                    |
| -------------- | --------------- | ---------------------------------------------- |
| **`ani`**      | **FuncAnimation** | The matplotlib animation to be converted.      |
| **`fps`**      | **int** (optional) | Frames per second for the animation. Defaults to `30`. |
| **`save_dir`** | **str** (optional) | The directory to save the animation. Defaults to `"/tmp/temp_animation.gif"`. (Note: The file will be deleted after the execution of the app is finished.) You can only write to the tmp directory in mecsimcalc. |

#### Returns:

| Return Type | Description              |
| ----------- | ------------------------ |
| **`str`**   | The HTML image tag as a string. |

#### Example:

```python
>>> fig, ax = plt.subplots()
>>> x = np.linspace(0, 10, 1000)
>>> y = np.sin(x)
>>> line, = ax.plot(x, y)
>>> def update(frame):
>>>     line.set_ydata(np.sin(x + frame / 100))
>>> ani = FuncAnimation(fig, update, frames=100)
>>> animation = msc.print_animation(ani)
>>> return {
        "animation": animation
    }
```

### animate_plot

[**[Source]**](https://github.com/MecSimCalc/MecSimCalc-utils/blob/b0.1.9/mecsimcalc/file_utils/plotting_utils.py#L152C1-L243C105)

```python
animate_plot(
    x: np.ndarray,
    y: np.ndarray,
    duration: int = 5,
    fps: int = None,
    title: str = "y = f(x)",
    show_axes: bool = True,
    save_dir: str = "/tmp/temp_animation.gif",
) -> str:
```

#### Description:

Creates an animated plot from given x and y data and returns it as an HTML image tag.

#### Arguments:

| Argument       | Type            | Description                                           |
| -------------- | --------------- | ----------------------------------------------------- |
| **`x`**        | **np.ndarray**  | The x-coordinates of the data points.                 |
| **`y`**        | **np.ndarray**  | The y-coordinates of the data points.                 |
| **`duration`** | **int** (optional) | The duration of the animation in seconds. Defaults to `5`. |
| **`fps`**      | **int** (optional) | Frames per second for the animation. Defaults to `None`. (fps = len(x) / duration if fps=None) |
| **`title`**    | **str** (optional) | Title of the plot. Defaults to `"y = f(x)"`.          |
| **`show_axes`** | **bool** (optional) | Whether to show the x and y axes. Defaults to `True`.  |
| **`save_dir`** | **str** (optional) | The directory to save the animation. Defaults to `"/tmp/temp_animation.gif"`. (Note: The file will be deleted after the execution of the app is finished.) You can only write to the tmp directory in mecsimcalc.|

#### Returns:

| Return Type | Description                            |
| ----------- | -------------------------------------- |
| **`str`**   | The HTML image tag containing the animated plot. |

#### Example:

```python
>>> import numpy as np
>>> x = np.linspace(0, 10, 1000)
>>> y = np.sin(x)
>>> animation_html = animate_plot(x, y, duration=5, title="Sine Wave", show_axes=True)
>>> return {
        "animation": animation_html
    }
```

## Quiz Toolkit

### append_to_google_sheet

[**[Source]**](https://github.com/MecSimCalc/MecSimCalc-utils/blob/b0.1.9/mecsimcalc/file_utils/quiz_utils.py#L11C1-L134C1)

```python
append_to_google_sheet(
    service_account_info = {...},
    spreadsheet_id = "123abc...",
    values = [["name", 12837, ...]],
    range_name = 'Sheet1!A1',
    include_timestamp = True
)
```

#### Description:

This function appends given values to a specified Google Sheet and optionally includes a current timestamp with each entry. It transforms data into a Google Sheets document, facilitating dynamic data entry directly from your application.


#### Arguments:

| Argument               | Type     | Description                                                                                                                                  |
|------------------------|----------|----------------------------------------------------------------------------------------------------------------------------------------------|
| **`service_account_info`** | **dict** | The service account credentials used for Google Sheets API authentication.  |
| **`spreadsheet_id`**       | **str**  | The unique identifier of the target Google Spreadsheet.                     |
| **`values`**               | **list of lists** | The data to append. Each list element represents a row of data.                                                                              |
| **`range_name`**           | **str** (optional)  | The A1 notation of the range to start appending data (Defaults to 'Sheet1!A1').                                                              |
| **`include_timestamp`**    | **bool** (optional) | If True, appends the current timestamp to each row of data (Defaults to True).                                                               |

#### Returns:

| Return Type | Description                              |
|-------------|------------------------------------------|
| **`dict`**  | The response from the Google Sheets API, containing details of the append operation. |

#### Example:

#### Code step:

```python
import mecsimcalc as msc

def main(inputs):
    service_account_info = {
        # Your service account info here
    }
    spreadsheet_id = 'your_spreadsheet_id_here'
    values = [
    [ inputs['input_1'], inputs['input_2'], inputs['input_3'] ],
    ]    
    result = msc.append_to_google_sheet(service_account_info, spreadsheet_id, values)

```


### send_gmail

[**[Source]**](https://github.com/MecSimCalc/MecSimCalc-utils/blob/b0.1.9/mecsimcalc/file_utils/quiz_utils.py#L136C1-L207C21)

```python
send_gmail(
	sender_email='sender@example.com', 
	receiver_email='receiver@example.com', 
	subject="Quiz", 
	app_password = "xxxx xxxx xxxx xxxx", 
	values = [
	["name", "grade"]
	]
)
```
#### Description:

This function sends an email with specified values formatted in the message body, utilizing a service account for authentication.

#### Arguments:

| Argument            | Type     | Description                                                                |
|---------------------|----------|----------------------------------------------------------------------------|
| **`sender_email`**      | **str**  | The email address of the sender.                                           |
| **`receiver_email`**    | **str**  | The email address of the receiver.                                         |
| **`subject`**           | **str**  | The subject line of the email.                                             |
| **`app_password`**      | **str**  | The app-specific password for the sender's email account.                  |
| **`values`**            | **list** | A list of lists. Each list contains data to be included in the email body. |

#### Returns:

| Return Type | Description                     |
|-------------|---------------------------------|
| **`bool`**  | Returns True if the email was sent successfully, otherwise False. |

#### Example Usage:

```python
# Example code to use the send_gmail function
import mecsimcalc as msc

def main(inputs):
    # Define parameters
    sender_email = 'sender@example.com'
    receiver_email = 'receiver@example.com'
    subject = 'Test Email'
    app_password = 'your_app_password_here'
    
    name = inputs['name']
    grade = inputs['grade']
    
    values = [
    [name, grade]
    ]

    # Send the email
    msc.send_gmail(sender_email, receiver_email, subject, app_password, values)

```
