class Retrieval:
    """
    Handles semantic retrieval of documents from the ChromaDB vector store
    using embeddings generated by a SentenceTransformer model.

    This class integrates with:
      - VectorStore (for document and embedding storage)
      - EmbeddingManager (for query embedding generation)

    Attributes:
        vector_store (VectorStore): The initialized vector store instance.
        embedding_manager (EmbeddingManager): The embedding generator for queries.
    """

    def __init__(self, vector_store, embedding_manager):
        """
        Initializes the Retrieval class with required components.

        Args:
            vector_store (VectorStore): Instance of the VectorStore class.
            embedding_manager (EmbeddingManager): Instance of the EmbeddingManager class.
        """
        self.vector_store = vector_store
        self.embedding_manager = embedding_manager

    def retrieve(self, query: str, top_k: int = 3, score_threshold: float = 0.3) -> list:
        """
        Retrieves the most relevant documents from the vector store for a given query.

        Args:
            query (str): The input query string for semantic search.
            top_k (int): Number of top matching documents to retrieve.
            score_threshold (float): Minimum similarity score required to include a document.

        Returns:
            list[dict]: A list of retrieved documents with metadata and similarity scores.

        Raises:
            ValueError: If query is empty or vector store is uninitialized.
            RuntimeError: If retrieval from ChromaDB fails.
        """
        try:
            # Validate input query
            if not query.strip():
                raise ValueError("Query cannot be empty.")

            if self.vector_store is None or self.vector_store.collection is None:
                raise ValueError("Vector store not initialized properly.")

            # Generate query embeddings
            query_embedding = self.embedding_manager.generate_embeddings([query])[0]

            # Search in vector store
            results = self.vector_store.collection.query(
                query_embeddings=[query_embedding.tolist()],
                n_results=top_k
            )

            retrieved_docs = []

            # Process results safely
            if results.get('documents') and results['documents'][0]:
                documents = results['documents'][0]
                metadatas = results['metadatas'][0]
                ids = results['ids'][0]
                distances = results['distances'][0]

                for doc_id, document, metadata, distance in zip(ids, documents, metadatas, distances):
                    similarity_score = 1 - distance  # Convert distance to similarity

                    if similarity_score >= score_threshold:
                        retrieved_docs.append({
                            'id': doc_id,
                            'content': document,
                            'metadata': metadata,
                            'similarity_score': round(similarity_score, 4),
                            'distance': round(distance, 4)
                        })

            return retrieved_docs

        except ValueError as e:
            raise e
        except Exception as e:
            raise RuntimeError(f"Failed to retrieve documents: {e}")