import pytest
import random
from synthex.models import JobStatusResponseModel, JobStatus

from artifex.models.base_model import BaseModel

        
def job_status_func_success() -> JobStatusResponseModel:
    """
    Simulates a job status function that returns a Status object with either full (1.0) or half (0.5) progress.
    Returns:
        Status: An instance of Status with the progress attribute set to 1.0 or 0.5, chosen randomly.
    """
    
    return JobStatusResponseModel(progress=1.0, status=JobStatus.COMPLETED) if random.random() < 0.5 else \
        JobStatusResponseModel(progress=0.5, status=JobStatus.IN_PROGRESS)

def job_status_func_timeout() -> JobStatusResponseModel:
    """
    Simulates a job status function that returns a Status object with progress set to 0.5.
    Returns:
        Status: An instance of Status with progress initialized to 0.5.
    """
    
    return JobStatusResponseModel(progress=0.5, status=JobStatus.IN_PROGRESS)


@pytest.mark.unit
def test_await_data_generation_success(
    base_model: BaseModel
):
    """
    Test that the `_await_data_generation` method successfully waits for the data generation to complete.
    Args:
        base_model (BaseModel): An instance of the BaseModel class.
    """
    
    result = base_model._await_data_generation( # type: ignore
        get_status_fn=job_status_func_success,
        check_interval=1.0,
        timeout=10.0
    )
    
    assert result.progress == 1.0
    
    
@pytest.mark.unit
def test_await_data_generation_timeout_failure(
    base_model: BaseModel
):
    """
    Test that the _await_data_generation method raises a TimeoutError when the data generation process does 
    not produce any update within the specified timeout period.
    Args:
        base_model (BaseModel): An instance of the BaseModel class.
    """
    
    with pytest.raises(TimeoutError):
        base_model._await_data_generation( # type: ignore
            get_status_fn=job_status_func_timeout,
            check_interval=1.0,
            timeout=3.0
        )