"""
ODIN Protocol CLI Tool
Command-line interface for managing ODIN Protocol.
"""

import asyncio
import json
import sys
from pathlib import Path
import click
from odin_sdk import OdinClient, OdinMessageBuilder
from odin_sdk.plugins import PluginManager


@click.group()
@click.version_option(version="1.0.0")
def main():
    """ODIN Protocol CLI - Manage AI communication and plugins."""
    pass


@main.command()
@click.option("--api-key", required=True, help="ODIN Protocol API key")
@click.option("--message", required=True, help="Message content to send")
@click.option("--sender", default="cli-user", help="Sender ID")
@click.option("--receiver", default="odin-ai", help="Receiver ID")
def send(api_key, message, sender, receiver):
    """Send a message through ODIN Protocol."""
    async def _send():
        client = OdinClient(api_key=api_key)
        
        # Create message
        builder = OdinMessageBuilder()
        odin_message = builder.set_ids(
            trace_id=f"cli-{int(time.time())}",
            session_id="cli-session",
            sender_id=sender,
            receiver_id=receiver
        ).set_role("user").set_content(message).build()
        
        try:
            response = await client.send_message(odin_message)
            click.echo(f"✅ Message sent successfully!")
            click.echo(f"Action taken: {response.action_taken}")
            click.echo(f"Confidence: {response.confidence_score:.2f}")
            if response.explanation:
                click.echo(f"Explanation: {response.explanation}")
        except Exception as e:
            click.echo(f"❌ Error: {e}")
        finally:
            await client.close()
    
    asyncio.run(_send())


@main.command()
@click.option("--api-key", required=True, help="ODIN Protocol API key")
def analytics(api_key):
    """Get ODIN Protocol analytics."""
    async def _analytics():
        client = OdinClient(api_key=api_key)
        
        try:
            data = await client.get_analytics()
            click.echo("📊 ODIN Protocol Analytics")
            click.echo("=" * 40)
            for key, value in data.items():
                click.echo(f"{key}: {value}")
        except Exception as e:
            click.echo(f"❌ Error: {e}")
        finally:
            await client.close()
    
    asyncio.run(_analytics())


@main.group()
def plugin():
    """Manage ODIN Protocol plugins."""
    pass


@plugin.command("list")
@click.option("--plugin-dir", default="plugins", help="Plugin directory")
def list_plugins(plugin_dir):
    """List installed plugins."""
    async def _list():
        manager = PluginManager(plugin_dir)
        await manager.load_plugins_from_directory()
        
        plugins = manager.list_plugins()
        if not plugins:
            click.echo("No plugins found.")
            return
        
        click.echo("🔌 Installed Plugins")
        click.echo("=" * 40)
        for plugin in plugins:
            status = "✅ Enabled" if plugin["enabled"] else "❌ Disabled"
            click.echo(f"{plugin['name']} v{plugin['version']} - {status}")
            click.echo(f"  {plugin['description']}")
            click.echo()
    
    asyncio.run(_list())


@plugin.command("create")
@click.argument("name")
@click.option("--plugin-dir", default="plugins", help="Plugin directory")
def create_plugin(name, plugin_dir):
    """Create a new plugin template."""
    plugin_dir = Path(plugin_dir)
    plugin_dir.mkdir(exist_ok=True)
    
    plugin_file = plugin_dir / f"{name}.py"
    config_file = plugin_dir / f"{name}.json"
    
    # Plugin template
    plugin_code = f'''"""
{name.title()} Plugin for ODIN Protocol
Generated by ODIN CLI
"""

from odin_sdk.plugins import BasePlugin
from typing import Dict, Any


class {name.title()}Plugin(BasePlugin):
    """Custom plugin for ODIN Protocol."""
    
    @property
    def name(self) -> str:
        return "{name}"
    
    @property
    def version(self) -> str:
        return "1.0.0"
    
    @property
    def description(self) -> str:
        return "Custom {name} plugin for ODIN Protocol"
    
    async def initialize(self) -> bool:
        """Initialize the plugin."""
        self.logger.info(f"Initializing {{self.name}} plugin")
        
        # Add your initialization logic here
        # Access config with self.config.get("key", default)
        
        return True
    
    async def process_message(self, message: Any, context: Dict[str, Any]) -> Any:
        """Process ODIN messages."""
        self.logger.info(f"Processing message in {{self.name}} plugin")
        
        # Add your message processing logic here
        # Modify message or return None to leave unchanged
        
        return message
    
    async def shutdown(self):
        """Clean shutdown."""
        self.logger.info(f"Shutting down {{self.name}} plugin")
'''
    
    # Plugin config template
    plugin_config = {
        "enabled": True,
        "settings": {
            "example_setting": "example_value"
        }
    }
    
    # Write files
    with open(plugin_file, 'w') as f:
        f.write(plugin_code)
    
    with open(config_file, 'w') as f:
        json.dump(plugin_config, f, indent=2)
    
    click.echo(f"✅ Created plugin: {plugin_file}")
    click.echo(f"✅ Created config: {config_file}")
    click.echo(f"")
    click.echo(f"Edit the files and then run:")
    click.echo(f"  odin plugin list --plugin-dir {plugin_dir}")


@main.command()
def quickstart():
    """Show quickstart guide."""
    guide = '''
🚀 ODIN Protocol Quickstart Guide

1. Install the SDK:
   pip install odin-protocol

2. Get your API key:
   - Sign up at https://odin-protocol.com
   - Go to Dashboard > API Keys
   - Create a new API key

3. Send your first message:
   odin send --api-key YOUR_KEY --message "Hello ODIN!"

4. Check analytics:
   odin analytics --api-key YOUR_KEY

5. Create a custom plugin:
   odin plugin create my-plugin

6. Use in Python:
   ```python
   from odin_sdk import OdinClient
   
   client = OdinClient(api_key="your-key")
   message = client.create_message()\\
       .set_content("Hello from Python!")\\
       .build()
   
   response = client.send_message(message)
   print(f"Response: {response.action_taken}")
   ```

📚 Documentation: https://docs.odin-protocol.com
💬 Community: https://discord.gg/odin-protocol
🎫 Support: https://odin-protocol.com/support

Happy building! 🎯
'''
    click.echo(guide)


if __name__ == "__main__":
    import time
    main()
