"""Generate an ensemble of conformations from a parameters file."""

# IMPORTS
## Standard Library Imports
import os
import shutil

## Local Imports
from ensemblify.generation.ensemble_utils import read_input_parameters, run_sampling, setup_ensemble_gen_params

# MAIN FUNCTION
def generate_ensemble(parameters_path: str):
    """Generate an ensemble of conformations given a parameters file.

    Args:
        parameters_path (str):
            Path to a .yaml parameters file (generated by .html form or manually setup by user).
    
    Returns:
        str:
            Path to directory where the generated .pdb files making up the ensemble are stored.
    """
    # Read parameters file
    input_parameters = read_input_parameters(parameter_path=parameters_path)

    # Get relevant info
    JOB_NAME = input_parameters['job_name']
    ENSEMBLE_SIZE = input_parameters['size']
    OUTPUT_DIR = input_parameters['output_path']
    N_CORES = input_parameters['core_amount']

    # Setup job_name output folder
    OUTPUT_DIR_JOB = os.path.join(OUTPUT_DIR,JOB_NAME)
    if not os.path.isdir(OUTPUT_DIR_JOB):
        os.mkdir(OUTPUT_DIR_JOB)

    # Setup ensemble output folder
    ENSEMBLE_DIR = os.path.join(OUTPUT_DIR_JOB,'ensemble')
    assert not os.path.isdir(ENSEMBLE_DIR), 'You are trying to overwrite an existing ensemble!'
    os.mkdir(ENSEMBLE_DIR) # make

    # Create directory in our script output folder to store the given inputs, original and processed
    INPUTS_DIR = os.path.join(ENSEMBLE_DIR,'inputs')
    if os.path.isdir(INPUTS_DIR):
        shutil.rmtree(INPUTS_DIR) # remove
    os.mkdir(INPUTS_DIR) # make

    # Setup logs folder
    LOGS_DIR = os.path.join(ENSEMBLE_DIR,'logs')
    if os.path.isdir(LOGS_DIR):
        shutil.rmtree(LOGS_DIR) # remove
    os.mkdir(LOGS_DIR) # make

    # Setup processed pdbs folder
    VALID_PDBS_DIR = os.path.join(ENSEMBLE_DIR,'valid_pdbs')
    if os.path.isdir(VALID_PDBS_DIR):
        shutil.rmtree(VALID_PDBS_DIR) # remove
    os.mkdir(VALID_PDBS_DIR) # make

    # After these steps our directory structure is:
        # OUTPUT_PATH/job_name/ensemble/
        #                           inputs/
        #                           logs/
        #                           valid_pdbs/
    # And we are ready to generate our ensemble.

    # Save parameters used for running script
    with open(os.path.join(INPUTS_DIR,'1_parameters_input.yaml'),'w',encoding='utf-8') as f:
        for key in input_parameters:
            f.write(f'{key}:\n')
            if isinstance(input_parameters[key],dict):
                for nested_key in input_parameters[key]:
                    f.write(f'  {nested_key}: ')
                    if input_parameters[key][nested_key] is not None:
                        f.write(f'{input_parameters[key][nested_key]}\n')
                    else:
                        f.write('\n')
            else:
                f.write(f'  {input_parameters[key]}\n')

    # Process parameters and input .pdb
    PROCESSED_PARAMETERS_PATH,\
    INPUT_CLASHES_INFO = setup_ensemble_gen_params(input_params=input_parameters,
                                                   inputs_dir=INPUTS_DIR)

    # Setup log file
    LOG_FILE = os.path.join(LOGS_DIR,'sampling.log')
    if os.path.isfile(LOG_FILE):
        os.remove(LOG_FILE) # remove

    # Generate ensemble of valid .pdbs
    print(f'Generating ensemble of {ENSEMBLE_SIZE} valid pdbs using {N_CORES} processor cores... ')

    # Run sampling + FASPR + Pulchra until we have ENSEMBLE_SIZE valid pdbs
    run_sampling(input_parameters=PROCESSED_PARAMETERS_PATH,
                 input_clashes_file=INPUT_CLASHES_INFO,
                 valid_pdbs_dir=VALID_PDBS_DIR,
                 sampling_log=LOG_FILE)

    print('Ensemble Generation Finished!')

    return VALID_PDBS_DIR
