import numpy as np
import scipy as sp
import simsoptpp as sopp
from .magneticfield import MagneticField
from simsopt.geo.surfacerzfourier import SurfaceRZFourier
from simsopt.geo.curvexyzfourier import CurveXYZFourier

mu0 = 4.0*np.pi*1e-7

__all__ = ['WireframeField', 'enclosed_current']


class WireframeField(sopp.WireframeField, MagneticField):
    """
    Computes the magnetic field generated by currents in a wireframe grid.

    Parameters
    ----------
        wframe: instance of a ToroidalWireframe class

    """

    def __init__(self, wframe):

        sopp.WireframeField.__init__(self, wframe.nodes, wframe.segments,
                                     wframe.seg_signs, wframe.currents)
        MagneticField.__init__(self)
        self.wireframe = wframe

    def dB_by_dsegmentcurrents(self, compute_derivatives):
        """
        Calculates the derivative of the magnetic field or its spatial 
        derivatives at each field reference point with respect to the current
        in each wireframe segment.

        Parameters
        ----------
            compute_derivatives: integer (must be 0 or 1)
                If zero, will provide derivatives of the magnetic field with
                respect to segment currents. If one, will provide derivatives
                of the first spatial derivatives of the magnetic field with
                respect to segment currents.

        Returns
        -------
            dB_by_dsegmentcurrents: list of arrays
                Arrays of the requested derivatives, including one array for
                each segment in the wireframe.
        """

        points = self.get_points_cart_ref()
        n_points = len(points)
        if any([not self.fieldcache_get_status(f'B_{i}')
                for i in range(self.wireframe.n_segments)]):
            assert compute_derivatives >= 0
            self.compute(compute_derivatives)

        self._dB_by_dcoilcurrents = \
            [self.fieldcache_get_or_create(f'B_{i}', [n_points, 3])
             for i in range(self.wireframe.n_segments)]
        return self._dB_by_dcoilcurrents

    def dBnormal_by_dsegmentcurrents_matrix(self, surface, area_weighted=False):
        """
        Generates a matrix with derivatives of normal magnetic field on a
        surface of interest with respect to the current in each wireframe
        segment, useful for optimization of the segment currents.

        Parameters
        ----------
            surface: Surface class instance
                Surface on which to calculate the magnetic field. It is assumed
                that the quadrature points are evenly spaced in the toroidal
                and poloidal angles, and that they cover a whole number of
                half-periods.
            area_weighted: logical (optional)
                If true, will multiply each matrix element by the square root
                of the surface area ascribed to the corresponding quadrature 
                point. In this way, the expression (A*x)^2 gives the surface
                integral of the squared flux, where A is the matrix with 
                weighted elements and x is a vector with the currents in each 
                wireframe segment. Default is False.
        """

        points = self.get_points_cart_ref()
        n_points = len(points)

        if not isinstance(surface, SurfaceRZFourier):
            raise ValueError('Surface must be a SurfaceRZFourier object')

        n = surface.normal()
        absn = np.linalg.norm(n, axis=2)
        unitn = n * (1. / absn)[:, :, None]

        if area_weighted:
            fac = np.sqrt(absn/float(absn.size))
        else:
            fac = np.ones(absn.shape)

        matrix = np.ascontiguousarray(
            np.zeros((n_points, self.wireframe.n_segments)))
        dB_dsc = self.dB_by_dsegmentcurrents(0)
        for i in range(self.wireframe.n_segments):
            dB_dsc_i = dB_dsc[i].reshape(n.shape)
            matrix[:, i] = (fac*np.sum(dB_dsc_i * unitn, axis=2)).reshape((-1))

        return matrix


def enclosed_current(curve, field, n_quadpoints, preserve_points=True):
    """
    Calculates the integral of a vector field along a curve in 3D space.
    Useful for tests to verify consistency of wireframe solutions with their
    constraints.

    Parameters
    ----------
        curve: CurveXYZFourier class instance
            Curve along which the integral is to be taken. The more quadrature
            points the curve has, the greater precision the result will have.
        field: MagneticField class instance
            Magnetic field in which the curve is integrated.
        n_quadpoints: integer
            Number of quadrature points for the integral.
        preserve_points: boolean
            If true, the existing field points of `field` will be restored 
            before the function returns. If false, the field points will be
            changed to the quadrature points for integration. Default is true. 
            Setting to false may save time.

    Returns
    -------
        integral: double
            Integral of the magnetic field along the curve.
    """

    if not isinstance(curve, CurveXYZFourier):
        raise ValueError('curve must be an instance of the '
                         + 'CurveXYZFourier class')

    if not isinstance(field, MagneticField):
        raise ValueError('field must be an instance of the MagneticField class')

    # Make a copy of the input Curve
    _curve = CurveXYZFourier(np.linspace(0, 1, n_quadpoints), curve.order,
                             dofs=curve.dofs)

    # Obtain the field vectors along the curve
    if preserve_points:
        original_field_points = field.get_points_cart_ref()

    # Set the field points equal to the curve's quadrature points
    field.set_points(_curve.gamma())
    field_on_curve = field.B()

    if preserve_points:
        field.set_points(original_field_points)

    abs_tangent = np.linalg.norm(_curve.gammadash(), axis=1).reshape((-1, 1))
    unit_tangent = _curve.gammadash()/abs_tangent

    # Find the projection of the field onto the curve tangent vector
    B_dot_unit_tangent = np.sum(field_on_curve*unit_tangent, axis=1)

    # Estimate the cumulative arc length
    inc_arc = _curve.incremental_arclength()
    midpoint_inc_arc = 0.5*(inc_arc[:-1] + inc_arc[1:])
    delta_arclength = np.diff(_curve.quadpoints)*midpoint_inc_arc
    arclength = np.concatenate(([0], np.cumsum(delta_arclength)))

    return sp.integrate.simpson(B_dot_unit_tangent, x=arclength)/mu0
