"""
Output Style Decorator Module - Defines Base Class and Interface for Output Styles
"""

from pathlib import Path
from typing import Dict, List
from abc import ABC, abstractmethod

from ...config.config_schema import RepomixConfig
from ...core.file.file_types import ProcessedFile


HEADER = "This file is a merged representation of the entire codebase, combining all repository files into a single document.\nGenerated by Repomix on: {generation_date}"

SUMMARY_PURPOSE = """This file contains a packed representation of the entire repository's contents.
It is designed to be easily consumable by AI systems for analysis, code review,
or other automated processes."""

SUMMARY_FILE_FORMAT = """The content is organized as follows:
1. This summary section
2. Repository information
3. Repository structure"""

SUMMARY_USAGE_GUIDELINES = """- This file should be treated as read-only. Any changes should be made to the
  original repository files, not this packed version.
- When processing this file, use the file path to distinguish
  between different files in the repository.
- Be aware that this file may contain sensitive information. Handle it with
  the same level of security as you would the original repository.{header_text}{repository_instruction}"""

SUMMARY_NOTES = """- Some files may have been excluded based on .gitignore rules and Repomix's
  configuration.
- Binary files are not included in this packed representation. Please refer to
  the Repository Structure section for a complete list of file paths, including
  binary files.{remove_comments_tip}{show_line_numbers_tip}"""


SUMMARY_ADDITIONAL_INFO = "For more information about Repomix, visit: https://github.com/andersonby/python-repomix"


class OutputStyle(ABC):
    """Base Output Style Class"""

    def __init__(self, config: RepomixConfig):
        self.config = config

    @property
    def header(self) -> str:
        """Get header"""
        return HEADER.format(generation_date=self._get_current_time())

    @property
    def summary_purpose(self) -> str:
        """Get summary purpose"""
        return SUMMARY_PURPOSE

    @property
    def summary_file_format(self) -> str:
        """Get summary file format"""
        return SUMMARY_FILE_FORMAT

    @property
    def summary_usage_guidelines(self) -> str:
        """Get summary usage guidelines"""
        header_text = (
            "\n- Pay special attention to the Repository Description. These contain important context and guidelines specific to this project.\n"
            if self.config.output.header_text
            else ""
        )
        repository_instruction = ""
        if self.config.output.instruction_file_path:
            if Path(self.config.output.instruction_file_path).exists():
                with open(self.config.output.instruction_file_path, encoding="utf-8") as file:
                    repository_instruction = f"\n{file.read().strip()}"
        return SUMMARY_USAGE_GUIDELINES.format(header_text=header_text, repository_instruction=repository_instruction)

    @property
    def summary_notes(self) -> str:
        """Get summary notes"""
        remove_comments_tip = "\n- Code comments have been removed." if self.config.output.remove_comments else ""
        show_line_numbers_tip = "\n- Line numbers have been added to the beginning of each line." if self.config.output.show_line_numbers else ""
        parsable_style_tip = f"\n- Content has been formatted for parsing in {self.config.output.style_enum.value} style." if self.config.output.parsable_style else ""
        return SUMMARY_NOTES.format(
            remove_comments_tip=remove_comments_tip,
            show_line_numbers_tip=show_line_numbers_tip,
        ) + parsable_style_tip

    @property
    def summary_additional_info(self) -> str:
        """Get summary additional info"""
        return SUMMARY_ADDITIONAL_INFO

    def _get_current_time(self) -> str:
        """Get formatted current time string

        Returns:
            Formatted time string
        """
        from datetime import datetime

        return datetime.now().strftime("%Y-%m-%d %H:%M:%S")

    @abstractmethod
    def generate_header(self) -> str:
        """Generate output header

        Returns:
            Header content
        """
        pass

    @abstractmethod
    def generate_footer(self) -> str:
        """Generate output footer

        Returns:
            Footer content
        """
        pass

    @abstractmethod
    def generate_files_section(
        self,
        files: List[ProcessedFile],
        file_char_counts: Dict[str, int],
        file_token_counts: Dict[str, int],
    ) -> str:
        """Generate files section

        Args:
            files: List of processed files

        Returns:
            Files section content
        """
        pass

    @abstractmethod
    def generate_file_section(self, file_path: str, content: str, char_count: int, token_count: int) -> str:
        """Generate file section

        Args:
            file_path: File path
            content: File content
            char_count: Character count
            token_count: Token count

        Returns:
            File section content
        """
        pass

    @abstractmethod
    def generate_statistics(self, total_files: int, total_chars: int, total_tokens: int) -> str:
        """Generate statistics

        Args:
            total_files: Total number of files
            total_chars: Total character count
            total_tokens: Total token count

        Returns:
            Statistics content
        """
        pass

    @abstractmethod
    def generate_file_tree_section(self, file_tree: Dict) -> str:
        """Generates the file tree section."""
        pass


def format_file_size(size: float | int) -> str:
    """Format file size

    Args:
        size: File size (in bytes)

    Returns:
        Formatted size string
    """
    for unit in ["B", "KB", "MB", "GB"]:
        if size < 1024:
            return f"{size:.1f}{unit}"
        size /= 1024
    return f"{size:.1f}TB"
