from __future__ import annotations

from typing import NamedTuple

import pytest

from qrules.settings import InteractionType
from qrules.transition import StateTransitionManager


class Input(NamedTuple):
    initial_state: list
    final_state: list
    intermediate_states: list
    final_state_grouping: list


@pytest.mark.parametrize(
    (
        "test_input",
        "ingoing_state",
        "relative_parity_prefactor",
    ),
    [
        (
            Input(
                [("J/psi(1S)", [1])],
                [("gamma", [-1, 1]), ("pi0", [0]), ("pi0", [0])],
                ["f(0)(980)"],
                ["pi0", "pi0"],
            ),
            "J/psi(1S)",
            1.0,
        ),
        (
            Input(
                [("J/psi(1S)", [1])],
                [("pi0", [0]), ("pi+", [0]), ("pi-", [0])],
                ["rho(770)"],
                ["pi+", "pi-"],
            ),
            "J/psi(1S)",
            -1.0,
        ),
    ],
)
def test_parity_prefactor(
    test_input: Input,
    ingoing_state: str,
    relative_parity_prefactor: float,
) -> None:
    stm = StateTransitionManager(
        test_input.initial_state,
        test_input.final_state,
        allowed_intermediate_particles=test_input.intermediate_states,
    )
    stm.add_final_state_grouping(test_input.final_state_grouping)
    stm.set_allowed_interaction_types([InteractionType.EM])
    problem_sets = stm.create_problem_sets()

    reaction = stm.find_solutions(problem_sets)

    assert len(reaction.group_by_topology()) == 1
    for transition in reaction.transitions:
        in_edges = [
            state_id
            for state_id, state in transition.states.items()
            if state.particle.name == ingoing_state
        ]
        assert len(in_edges) == 1

        node_id = transition.topology.edges[in_edges[0]].ending_node_id
        assert isinstance(node_id, int)
        assert (
            relative_parity_prefactor
            == transition.interactions[node_id].parity_prefactor
        )
