"""
realestate_tvm.core
===================

Core time value of money helpers that are not provided by ``numpy_financial``.

Functions
---------
- compound(pv, i, n): Future value with simple compounding.
- eay(fv, pv, n): Effective annual yield (EAY) from PV and FV.
- annuity_value(pmt, i, n): Future value of an ordinary annuity.
"""

from __future__ import annotations

__all__ = [
    "compound",
    "eay",
    "annuity_value",
]


def compound(pv: float, i: float, n: int) -> float:
    """
    Compute the future value of a lump sum under simple compounding.

    Parameters
    ----------
    pv : float
        Present value today. Use negative for an outflow (investment)
        and positive for an inflow, consistent with Excel and numpy_financial.
    i : float
        Periodic interest rate in decimal form (e.g., 0.05 for 5%).
    n : int
        Number of compounding periods.

    Returns
    -------
    float
        Future value after ``n`` periods.

    Examples
    --------
    >>> compound(pv=-300_000, i=0.05, n=1)
    -315000.0
    """
    return pv * (1 + i) ** n


def eay(fv: float, pv: float, n: float) -> float:
    """
    Effective Annual Yield (EAY) / compound annual growth rate.

    Given a present value ``pv`` and future value ``fv`` separated by
    ``n`` periods, this returns the constant annual rate that links them.

    Parameters
    ----------
    fv : float
        Future value after ``n`` periods.
    pv : float
        Present value at the beginning of the period.
    n : float
        Number of periods (often years) between ``pv`` and ``fv``.

    Returns
    -------
    float
        Effective annual yield (EAY) as a decimal.

    Notes
    -----
    EAY = (FV / PV) ** (1 / n) - 1

    Examples
    --------
    >>> eay(fv=500_000, pv=400_000, n=5)
    0.04563955259127317
    """
    if pv == 0:
        raise ValueError("pv must be non-zero to compute EAY.")
    return (fv / pv) ** (1.0 / n) - 1.0


def annuity_value(pmt: float, i: float, n: int) -> float:
    """
    Future value of an ordinary annuity (payments at end of each period).

    Parameters
    ----------
    pmt : float
        Payment per period (negative for deposits, positive for receipts).
    i : float
        Periodic interest rate (decimal).
    n : int
        Total number of payments.

    Returns
    -------
    float
        Future value of the annuity at the end of period ``n``.

    Notes
    -----
    If interest rate ``i`` is zero, this reduces to ``pmt * n``.

    Examples
    --------
    >>> annuity_value(pmt=-5_000, i=0.04, n=3)
    -15608.0...

    This matches ``numpy_financial.fv`` when ``pv=0``:

    >>> import numpy_financial as npf
    >>> npf.fv(rate=0.04, nper=3, pmt=-5_000, pv=0)
    -15608.000000000011
    """
    if i == 0:
        return pmt * n
    return (pmt * ((1 + i) ** n - 1.0)) / i
