"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsSolutionsChecks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../nag-pack");
const apigw_1 = require("../rules/apigw");
const appsync_1 = require("../rules/appsync");
const athena_1 = require("../rules/athena");
const autoscaling_1 = require("../rules/autoscaling");
const cloud9_1 = require("../rules/cloud9");
const cloudfront_1 = require("../rules/cloudfront");
const codebuild_1 = require("../rules/codebuild");
const cognito_1 = require("../rules/cognito");
const documentdb_1 = require("../rules/documentdb");
const dynamodb_1 = require("../rules/dynamodb");
const ec2_1 = require("../rules/ec2");
const ecr_1 = require("../rules/ecr");
const ecs_1 = require("../rules/ecs");
const efs_1 = require("../rules/efs");
const eks_1 = require("../rules/eks");
const elasticache_1 = require("../rules/elasticache");
const elasticbeanstalk_1 = require("../rules/elasticbeanstalk");
const elb_1 = require("../rules/elb");
const emr_1 = require("../rules/emr");
const eventbridge_1 = require("../rules/eventbridge");
const glue_1 = require("../rules/glue");
const iam_1 = require("../rules/iam");
const kinesis_1 = require("../rules/kinesis");
const kms_1 = require("../rules/kms");
const mediastore_1 = require("../rules/mediastore");
const msk_1 = require("../rules/msk");
const neptune_1 = require("../rules/neptune");
const opensearch_1 = require("../rules/opensearch");
const quicksight_1 = require("../rules/quicksight");
const rds_1 = require("../rules/rds");
const redshift_1 = require("../rules/redshift");
const s3_1 = require("../rules/s3");
const sagemaker_1 = require("../rules/sagemaker");
const secretsmanager_1 = require("../rules/secretsmanager");
const sns_1 = require("../rules/sns");
const sqs_1 = require("../rules/sqs");
const stepfunctions_1 = require("../rules/stepfunctions");
const timestream_1 = require("../rules/timestream");
const vpc_1 = require("../rules/vpc");
/**
 * Check Best practices based on AWS Solutions Security Matrix
 *
 */
class AwsSolutionsChecks extends nag_pack_1.NagPack {
    constructor(props) {
        super(props);
        this.packName = 'AwsSolutions';
    }
    visit(node) {
        if (node instanceof monocdk_1.CfnResource) {
            this.checkCompute(node);
            this.checkStorage(node);
            this.checkDatabases(node);
            this.checkNetworkDelivery(node);
            this.checkManagementGovernance(node);
            this.checkMachineLearning(node);
            this.checkAnalytics(node);
            this.checkSecurityCompliance(node);
            this.checkServerless(node);
            this.checkApplicationIntegration(node);
            this.checkMediaServices(node);
            this.checkDeveloperTools(node);
        }
    }
    /**
     * Check Compute Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCompute(node) {
        this.applyRule({
            ruleSuffixOverride: 'EB1',
            info: 'The Elastic Beanstalk environment is not configured to use a specific VPC.',
            explanation: 'Use a non-default in order to seperate your environment from default resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticbeanstalk_1.ElasticBeanstalkVPCSpecified,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EB3',
            info: 'The Elastic Beanstalk environment does not have managed updates enabled.',
            explanation: 'Enable managed platform updates for beanstalk environments in order to receive bug fixes, software updates and new features. Managed platform updates perform immutable environment updates.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticbeanstalk_1.ElasticBeanstalkManagedUpdatesEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EB4',
            info: 'The Elastic Beanstalk environment does not upload EC2 Instance logs to S3.',
            explanation: 'Beanstalk environment logs should be retained and uploaded to Amazon S3 in order to keep the logging data for future audits, historical purposes or to track and analyze the EB application environment behavior for a long period of time.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: elasticbeanstalk_1.ElasticBeanstalkEC2InstanceLogsToS3,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC23',
            info: 'The Security Group allows for 0.0.0.0/0 or ::/0 inbound access.',
            explanation: 'Large port ranges, when open, expose instances to unwanted attacks. More than that, they make traceability of vulnerabilities very difficult. For instance, your web servers may only require 80 and 443 ports to be open, but not all. One of the most common mistakes observed is when  all ports for 0.0.0.0/0 range are open in a rush to access the instance. EC2 instances must expose only to those ports enabled on the corresponding security group level.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2RestrictedInbound,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC26',
            info: 'The EBS volume has encryption disabled.',
            explanation: "With EBS encryption, you aren't required to build, maintain, and secure your own key management infrastructure. EBS encryption uses KMS keys when creating encrypted volumes and snapshots. This helps protect data at rest.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2EBSVolumeEncrypted,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC27',
            info: 'The Security Group does not have a description.',
            explanation: 'Descriptions help simplify operations and remove any opportunities for operator errors.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2SecurityGroupDescription,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC28',
            info: 'The EC2 instance/AutoScaling launch configuration does not have detailed monitoring enabled.',
            explanation: 'Monitoring data helps make better decisions on architecting and managing compute resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceDetailedMonitoringEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC29',
            info: 'The EC2 instance is not part of an ASG and has Termination Protection disabled.',
            explanation: 'Termination Protection safety feature enabled in order to protect the instances from being accidentally terminated.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceTerminationProtection,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ECR1',
            info: 'The ECR Repository allows open access.',
            explanation: 'Removing * principals in an ECR Repository helps protect against unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ecr_1.ECROpenAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ECS2',
            info: 'The ECS Task Definition includes a container definition that directly specifies environment variables.',
            explanation: 'Use secrets to inject environment variables during container startup from AWS Systems Manager Parameter Store or Secrets Manager instead of directly specifying plaintext environment variables. Updates to direct environment variables require operators to change task definitions and perform new deployments.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ecs_1.ECSTaskDefinitionNoEnvironmentVariables,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ECS4',
            info: 'The ECS Cluster has CloudWatch Container Insights disabled.',
            explanation: 'CloudWatch Container Insights allow operators to gain a better perspective on how the cluster’s applications and microservices are performing.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ecs_1.ECSClusterCloudWatchContainerInsights,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ECS7',
            info: 'The ECS Task Definition does not have awslogs logging enabled at the minimum.',
            explanation: 'Container logging allows operators to view and aggregate the logs from the container.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ecs_1.ECSTaskDefinitionContainerLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EKS1',
            info: "The EKS cluster's Kubernetes API server endpoint has public access enabled.",
            explanation: "A cluster's Kubernetes API server endpoint should not be publicly accessible from the Internet in order to avoid exposing private data and minimizing security risks. The API server endpoints should only be accessible from within a AWS Virtual Private Cloud (VPC).",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: eks_1.EKSClusterNoEndpointPublicAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EKS2',
            info: "The EKS Cluster does not publish 'api', 'audit', 'authenticator, 'controllerManager', and 'scheduler' control plane logs.",
            explanation: 'EKS control plane logging provides audit and diagnostic logs directly from the Amazon EKS control plane to CloudWatch Logs in your account. These logs make it easy for you to secure and run your clusters.' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'LogExport::<log>' for exported logs. Example: appliesTo: ['LogExport::api'].",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: eks_1.EKSClusterControlPlaneLogs,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB1',
            info: 'The CLB is used for incoming HTTP/HTTPS traffic. Use ALBs instead.',
            explanation: 'HTTP/HTTPS applications (monolithic or containerized) should use the ALB instead of the CLB for enhanced incoming traffic distribution, better performance and lower costs.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.CLBNoInboundHttpHttps,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB2',
            info: 'The ELB does not have access logs enabled.',
            explanation: 'Access logs allow operators to to analyze traffic patterns and identify and troubleshoot security issues.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB3',
            info: 'The CLB does not have connection draining enabled.',
            explanation: 'With Connection Draining feature enabled, if an EC2 backend instance fails health checks The CLB will not send any new requests to the unhealthy instance. However, it will still allow existing (in-flight) requests to complete for the duration of the configured timeout.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.CLBConnectionDraining,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB4',
            info: 'The CLB does not use at least two AZs with the Cross-Zone Load Balancing feature enabled.',
            explanation: 'CLBs can distribute the traffic evenly across all backend instances. To use Cross-Zone Load Balancing at optimal level, the system should maintain an equal EC2 capacity distribution in each of the AZs registered with the load balancer.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBCrossZoneLoadBalancingEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB5',
            info: 'The CLB listener is not configured for secure (HTTPs or SSL) protocols for client communication.',
            explanation: 'The HTTPs or SSL protocols enable secure communication by encrypting the communication between the client and the load balancer.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBTlsHttpsListenersOnly,
            node: node,
        });
    }
    /**
     * Check Storage Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkStorage(node) {
        this.applyRule({
            ruleSuffixOverride: 'S1',
            info: 'The S3 Bucket has server access logs disabled.',
            explanation: 'The bucket should have server access logging enabled to provide detailed records for the requests that are made to the bucket.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'S2',
            info: 'The S3 Bucket does not have public access restricted and blocked.',
            explanation: 'The bucket should have public access restricted and blocked to prevent unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketLevelPublicAccessProhibited,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'S3',
            info: 'The S3 Bucket does not default encryption enabled.',
            explanation: 'The bucket should minimally have SSE enabled to help protect data-at-rest.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketServerSideEncryptionEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'S5',
            info: 'The S3 static website bucket either has an open world bucket policy or does not use a CloudFront Origin Access Identity (OAI) in the bucket policy for limited getObject and/or putObject permissions.',
            explanation: 'An OAI allows you to provide access to content in your S3 static website bucket through CloudFront URLs without enabling public access through an open bucket policy, disabling S3 Block Public Access settings, and/or through object ACLs.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3WebBucketOAIAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'S10',
            info: 'The S3 Bucket does not require requests to use SSL.',
            explanation: 'You can use HTTPS (TLS) to help prevent potential attackers from eavesdropping on or manipulating network traffic using person-in-the-middle or similar attacks. You should allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition on Amazon S3 bucket policies.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketSSLRequestsOnly,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EFS1',
            info: 'The EFS is not configured for encryption at rest.',
            explanation: 'By using an encrypted file system, data and metadata are automatically encrypted before being written to the file system. Similarly, as data and metadata are read, they are automatically decrypted before being presented to the application. These processes are handled transparently by EFS without requiring modification of applications.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: efs_1.EFSEncrypted,
            node: node,
        });
    }
    /**
     * Check Database Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDatabases(node) {
        this.applyRule({
            ruleSuffixOverride: 'RDS2',
            info: 'The RDS instance or Aurora DB cluster does not have storage encryption enabled.',
            explanation: 'Storage encryption helps protect data-at-rest by encrypting the underlying storage, automated backups, read replicas, and snapshots for the database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSStorageEncrypted,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS3',
            info: 'The non-Aurora RDS DB instance does not have multi-AZ support enabled.',
            explanation: 'Use multi-AZ deployment configurations for high availability and automatic failover support fully managed by AWS.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSMultiAZSupport,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS6',
            info: 'The RDS Aurora MySQL/PostgresSQL cluster does not have IAM Database Authentication enabled.',
            explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the MySQL/PostgreSQL database instances, instead it uses an authentication token.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.AuroraMySQLPostgresIAMAuth,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS8',
            info: 'The RDS DB Security Group allows for 0.0.0.0/0 inbound access.',
            explanation: 'RDS DB security groups should not allow access from 0.0.0.0/0 (i.e. anywhere, every machine that has the ability to establish a connection) in order to reduce the risk of unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSRestrictedInbound,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS10',
            info: 'The RDS instance or Aurora DB cluster does not have deletion protection enabled.',
            explanation: 'The deletion protection feature helps protect the database from being accidentally deleted.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS11',
            info: 'The RDS instance or Aurora DB cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MySQL/Aurora port 3306, SQL Server port 1433, PostgreSQL port 5432, etc).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSNonDefaultPort,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS13',
            info: 'The RDS instance is not configured for automated backups.',
            explanation: 'Automated backups allow for point-in-time recovery.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceBackupEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS14',
            info: 'The RDS Aurora MySQL cluster does not have Backtrack enabled.',
            explanation: 'Backtrack helps order to rewind cluster tables to a specific time, without using backups.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.AuroraMySQLBacktrack,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS15',
            info: 'The RDS DB instance or Aurora DB cluster does not have deletion protection enabled.',
            explanation: 'Enabling Deletion Protection at the cluster level for Amazon Aurora databases or instance level for non Aurora instances helps protect from accidental deletion.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS16',
            info: 'The RDS Aurora MySQL serverless cluster does not have audit, error, general, and slowquery Log Exports enabled.',
            explanation: 'This allows operators to use CloudWatch to view logs to help diagnose problems in the database.' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'LogExport::<log>' for exported logs. Example: appliesTo: ['LogExport::audit'].",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.AuroraMySQLLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DDB3',
            info: 'The DynamoDB table does not have Point-in-time Recovery enabled.',
            explanation: 'DynamoDB continuous backups represent an additional layer of insurance against accidental loss of data on top of on-demand backups. The DynamoDB service can back up the data with per-second granularity and restore it to any single second from the time PITR was enabled up to the prior 35 days.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: dynamodb_1.DynamoDBPITREnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DDB4',
            info: 'The DAX cluster does not have server-side encryption enabled.',
            explanation: 'Data in cache, configuration data and log files should be encrypted using Server-Side Encryption in order to protect from unauthorized access to the underlying storage.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.DAXEncrypted,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC1',
            info: 'The ElastiCache cluster is not provisioned in a VPC.',
            explanation: 'Provisioning the cluster within a VPC allows for better flexibility and control over the cache clusters security, availability, traffic routing and more.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheClusterInVPC,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC3',
            info: 'The ElastiCache Redis cluster does not have both encryption in transit and at rest enabled.',
            explanation: 'Encryption in transit helps secure communications to the cluster. Encryption at rest helps protect data at rest from unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheRedisClusterEncryption,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC4',
            info: 'The ElastiCache Redis cluster is not deployed in a Multi-AZ configuration.',
            explanation: 'The cluster should use a Multi-AZ deployment configuration for high availability.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheRedisClusterMultiAZ,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC5',
            info: 'The ElastiCache cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redis port 6379 and Memcached port 11211).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheClusterNonDefaultPort,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC6',
            info: 'The ElastiCache Redis cluster does not use Redis AUTH for user authentication.',
            explanation: 'Redis authentication tokens enable Redis to require a token (password) before allowing clients to execute commands, thereby improving data security.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheRedisClusterRedisAuth,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N1',
            info: 'The Neptune DB cluster is not deployed in a Multi-AZ configuration.',
            explanation: 'The cluster should use a Multi-AZ deployment configuration for high availability.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterMultiAZ,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N2',
            info: 'The Neptune DB instance does have Auto Minor Version Upgrade enabled.',
            explanation: 'The Neptune service regularly releases engine updates. Enabling Auto Minor Version Upgrade will allow the service to automatically apply these upgrades to DB Instances.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterAutomaticMinorVersionUpgrade,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N3',
            info: 'The Neptune DB cluster does not have a reasonable minimum backup retention period configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterBackupRetentionPeriod,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N4',
            info: 'The Neptune DB cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterEncryptionAtRest,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N5',
            info: 'The Neptune DB cluster does not have IAM Database Authentication enabled.',
            explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the cluster.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterIAMAuth,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS1',
            info: 'The Redshift cluster does not require TLS/SSL encryption.',
            explanation: 'Enabling the "require_ssl" parameter secures data-in-transit by encrypting the connection between the clients and the Redshift clusters.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftRequireTlsSSL,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS2',
            info: 'The Redshift cluster is not provisioned in a VPC.',
            explanation: 'Provisioning the cluster within a VPC allows for better flexibility and control over the Redshift clusters security, availability, traffic routing and more.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterInVPC,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS3',
            info: 'The Redshift cluster uses the default "awsuser" username.',
            explanation: 'Using a custom master user name instead of the default master user name (i.e. "awsuser") provides an additional layer of defense against non-targeted attacks.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterNonDefaultUsername,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS4',
            info: 'The Redshift cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redshift port 5439).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterNonDefaultPort,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS5',
            info: 'The Redshift cluster does not have audit logging enabled.',
            explanation: 'Audit logging helps operators troubleshoot issues and ensure security.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterAuditLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS6',
            info: 'The Redshift cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterEncryptionAtRest,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS8',
            info: 'The Redshift cluster is publicly accessible.',
            explanation: 'Disabling public accessibility helps minimize security risks.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterPublicAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS9',
            info: 'The Redshift cluster does not have version upgrade enabled.',
            explanation: 'Version Upgrade must enabled to enable the cluster to automatically receive upgrades during the maintenance window.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterVersionUpgrade,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS10',
            info: 'The Redshift cluster does not have a retention period for automated snapshots configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A positive retention period should be set to configure this feature.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftBackupEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS11',
            info: 'The Redshift cluster does not have user activity logging enabled.',
            explanation: 'User activity logging logs each query before it is performed on the clusters databse. To enable this feature associate a Resdhsift Cluster Parameter Group with the "enable_user_activity_logging" parameter set to "true".',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterUserActivityLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC1',
            info: 'The Document DB cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBClusterEncryptionAtRest,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC2',
            info: 'The Document DB cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MongoDB port 27017).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBClusterNonDefaultPort,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC3',
            info: 'The Document DB cluster does not have the username and password stored in Secrets Manager.',
            explanation: "Secrets Manager enables operators to replace hardcoded credentials in your code, including passwords, with an API call to Secrets Manager to retrieve the secret programmatically. This helps ensure the secret can't be compromised by someone examining system code, because the secret no longer exists in the code. Also, operators can configure Secrets Manager to automatically rotate the secret for you according to a specified schedule. This enables you to replace long-term secrets with short-term ones, significantly reducing the risk of compromise.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBCredentialsInSecretsManager,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC4',
            info: 'The Document DB cluster does not have a reasonable minimum backup retention period configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBClusterBackupRetentionPeriod,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC5',
            info: 'The Document DB cluster does not have authenticate, createIndex, and dropCollection Log Exports enabled.',
            explanation: 'This allows operators to use CloudWatch to view logs to help diagnose problems in the database. The events recorded by the AWS DocumentDB audit logs include successful and failed authentication attempts, creating indexes or dropping a collection in a database within the DocumentDB cluster.' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'LogExport::<log>' for exported logs. Example: appliesTo: ['LogExport::authenticate'].",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBClusterLogExports,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'TS3',
            info: 'The Timestream database does not use a Customer Managed KMS Key for at rest encryption.',
            explanation: 'All Timestream tables in a database are encrypted at rest by default using AWS Managed Key. These keys are rotated every three years. Data at rest must be encrypted using CMKs if you require more control over the permissions and lifecycle of your keys, including the ability to have them automatically rotated on an annual basis.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: timestream_1.TimestreamDatabaseCustomerManagedKey,
            node: node,
        });
    }
    /**
     * Check Network and Delivery Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkNetworkDelivery(node) {
        this.applyRule({
            ruleSuffixOverride: 'VPC3',
            info: 'A Network ACL or Network ACL entry has been implemented.',
            explanation: 'Network ACLs should be used sparingly for the following reasons: they can be complex to manage, they are stateless, every IP address must be explicitly opened in each (inbound/outbound) direction, and they affect a complete subnet. Use security groups when possible as they are stateful and easier to manage.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: vpc_1.VPCNoNACLs,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'VPC7',
            info: 'The VPC does not have an associated Flow Log.',
            explanation: 'VPC Flow Logs capture network flow information for a VPC, subnet, or network interface and stores it in Amazon CloudWatch Logs. Flow log data can help customers troubleshoot network issues; for example, to diagnose why specific traffic is not reaching an instance, which might be a result of overly restrictive security group rules.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: vpc_1.VPCFlowLogsEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR1',
            info: 'The CloudFront distribution may require Geo restrictions.',
            explanation: 'Geo restriction may need to be enabled for the distribution in order to allow or deny a country in order to allow or restrict users in specific locations from accessing content.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: cloudfront_1.CloudFrontDistributionGeoRestrictions,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR2',
            info: 'The CloudFront distribution may require integration with AWS WAF.',
            explanation: 'The Web Application Firewall can help protect against application-layer attacks that can compromise the security of the system or place unnecessary load on them.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: cloudfront_1.CloudFrontDistributionWAFIntegration,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR3',
            info: 'The CloudFront distributions does not have access logging enabled.',
            explanation: 'Enabling access logs helps operators track all viewer requests for the content delivered through the Content Delivery Network.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudfront_1.CloudFrontDistributionAccessLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR5',
            info: 'The CloudFront distributions uses SSLv3 or TLSv1 for communication to the origin.',
            explanation: 'Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Using a security policy with minimum TLSv1.1 or TLSv1.2 and appropriate security ciphers for HTTPS helps protect viewer connections.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudfront_1.CloudFrontDistributionNoOutdatedSSL,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR6',
            info: 'The CloudFront distribution does not use an origin access identity an S3 origin.',
            explanation: 'Origin access identities help with security by restricting any direct access to objects through S3 URLs.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudfront_1.CloudFrontDistributionS3OriginAccessIdentity,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG1',
            info: 'The API does not have access logging enabled.',
            explanation: 'Enabling access logs helps operators view who accessed an API and how the caller accessed the API.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWAccessLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG2',
            info: 'The REST API does not have request validation enabled.',
            explanation: 'The API should have basic request validation enabled. If the API is integrated with custom source (Lambda, ECS, etc..) in the backend, deeper input validation should be considered for implementation.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWRequestValidation,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG3',
            info: 'The REST API stage is not associated with AWS WAFv2 web ACL.',
            explanation: 'AWS WAFv2 is a web application firewall that helps protect web applications and APIs from attacks by allowing configured rules to allow, block, or monitor (count) web requests based on customizable rules and conditions that are defined.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: apigw_1.APIGWAssociatedWithWAF,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG4',
            info: 'The API does not implement authorization.',
            explanation: 'In most cases an API needs to have an authentication and authorization implementation strategy. This includes using such approaches as IAM, Cognito User Pools, Custom authorizer, etc.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWAuthorization,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG6',
            info: 'The REST API Stage does not have CloudWatch logging enabled for all methods.',
            explanation: 'Enabling CloudWatch logs at the stage level helps operators to track and analyze execution behavior at the API stage level.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWExecutionLoggingEnabled,
            node: node,
        });
    }
    /**
     * Check Management and Governance Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkManagementGovernance(node) {
        this.applyRule({
            ruleSuffixOverride: 'AS1',
            info: 'The Auto Scaling Group does not have a cooldown period.',
            explanation: 'A cooldown period temporarily suspends any scaling activities in order to allow the newly launched EC2 instance(s) some time to start handling the application traffic.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingGroupCooldownPeriod,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AS2',
            info: 'The Auto Scaling Group does not have properly configured health checks.',
            explanation: 'The health check feature enables the service to detect whether its registered EC2 instances are healthy or not.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingGroupHealthCheck,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AS3',
            info: 'The Auto Scaling Group does not have notifications configured for all scaling events.',
            explanation: 'Notifications on EC2 instance launch, launch error, termination, and termination errors allow operators to gain better insights into systems attributes such as activity and health.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingGroupScalingNotifications,
            node: node,
        });
    }
    /**
     * Check Machine Learning Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkMachineLearning(node) {
        this.applyRule({
            ruleSuffixOverride: 'SM1',
            info: 'The SageMaker notebook instance is not provisioned inside a VPC.',
            explanation: 'Provisioning the notebook instances inside a VPC enables the notebook to access VPC-only resources such as EFS file systems.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookInVPC,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SM2',
            info: 'The SageMaker notebook instance does not have an encrypted storage volume.',
            explanation: 'Encrypting storage volumes helps protect SageMaker data-at-rest.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookInstanceKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SM3',
            info: 'The SageMaker notebook instance has direct internet access enabled.',
            explanation: 'Disabling public accessibility helps minimize security risks.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookNoDirectInternetAccess,
            node: node,
        });
    }
    /**
     * Check Analytics Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAnalytics(node) {
        this.applyRule({
            ruleSuffixOverride: 'ATH1',
            info: 'The Athena workgroup does not encrypt query results.',
            explanation: 'Encrypting query results stored in S3 helps secure data to meet compliance requirements for data-at-rest encryption.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: athena_1.AthenaWorkgroupEncryptedQueryResults,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EMR2',
            info: 'The EMR cluster does not have S3 logging enabled.',
            explanation: 'Uploading logs to S3 enables the system to keep the logging data for historical purposes or to track and analyze the clusters behavior.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.EMRS3AccessLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EMR4',
            info: 'The EMR cluster does not use a security configuration with local disk encryption enabled.',
            explanation: 'Local disk encryption uses a combination of open-source HDFS encryption and LUKS encryption to secure data at rest.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.EMRLocalDiskEncryption,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EMR5',
            info: 'The EMR cluster does not use a security configuration with encryption in transit enabled and configured.',
            explanation: 'EMR Clusters should have a method for encrypting data in transit using Transport Layer Security (TLS).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.EMREncryptionInTransit,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EMR6',
            info: 'The EMR cluster does not implement authentication via an EC2 Key Pair or Kerberos.',
            explanation: 'SSH clients can use an EC2 key pair to authenticate to cluster instances. Alternatively, with EMR release version 5.10.0 or later, solutions can configure Kerberos to authenticate users and SSH connections to the master node.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.EMRAuthEC2KeyPairOrKerberos,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'GL1',
            info: 'The Glue crawler or job does not use a security configuration with CloudWatch Log encryption enabled.',
            explanation: 'Enabling encryption at rest helps prevent unauthorized users from getting access to the logging data published to CloudWatch Logs.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: glue_1.GlueEncryptedCloudWatchLogs,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'GL3',
            info: 'The Glue job does not have use a security configuration with job bookmark encryption enabled.',
            explanation: 'Job bookmark encryption encrypts bookmark data before it is sent to Amazon S3 for storage.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: glue_1.GlueJobBookmarkEncrypted,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KDA3',
            info: 'The Kinesis Data Analytics Flink Application does not have checkpointing enabled.',
            explanation: 'Checkpoints are backups of application state that KDA automatically creates periodically and uses to restore from faults.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: kinesis_1.KinesisDataAnalyticsFlinkCheckpointing,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KDF1',
            info: 'The Kinesis Data Firehose delivery stream does have server-side encryption enabled.',
            explanation: 'This allows the system to meet strict regulatory requirements and enhance the security of system data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: kinesis_1.KinesisDataFirehoseSSE,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KDS1',
            info: 'The Kinesis Data Stream does not has server-side encryption enabled.',
            explanation: "Data is encrypted before it's written to the Kinesis stream storage layer, and decrypted after it’s retrieved from storage. This allows the system to meet strict regulatory requirements and enhance the security of system data.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: kinesis_1.KinesisDataStreamSSE,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KDS3',
            info: 'The Kinesis Data Stream specifies server-side encryption and does not use the "aws/kinesis" key.',
            explanation: 'Customer Managed Keys can incur additional costs that scale with the amount of consumers and producers. Ensure that Customer Managed Keys are required for compliance before using them (https://docs.aws.amazon.com/streams/latest/dev/costs-performance.html).',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: kinesis_1.KinesisDataStreamDefaultKeyWhenSSE,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MSK2',
            info: 'The MSK cluster uses plaintext communication between clients and brokers.',
            explanation: 'TLS only communication secures data-in-transit by encrypting the connection between the clients and brokers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: msk_1.MSKClientToBrokerTLS,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MSK3',
            info: 'The MSK cluster uses plaintext communication between brokers.',
            explanation: 'TLS communication secures data-in-transit by encrypting the connection between brokers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: msk_1.MSKBrokerToBrokerTLS,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MSK6',
            info: 'The MSK cluster does not send broker logs to a supported destination.',
            explanation: 'Broker logs enable operators to troubleshoot Apache Kafka applications and to analyze their communications with the MSK cluster. The cluster can deliver logs to the following resources: a CloudWatch log group, an S3 bucket, a Kinesis Data Firehose delivery stream.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: msk_1.MSKBrokerLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS1',
            info: 'The OpenSearch Service domain is not provisioned inside a VPC.',
            explanation: 'Provisioning the domain within a VPC enables better flexibility and control over the clusters access and security as this feature keeps all traffic between the VPC and OpenSearch domains within the AWS network instead of going over the public Internet.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchInVPCOnly,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS2',
            info: 'The OpenSearch Service domain does not have node-to-node encryption enabled.',
            explanation: 'Enabling the node-to-node encryption feature adds an extra layer of data protection on top of the existing ES security features such as HTTPS client to cluster encryption and data-at-rest encryption.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchNodeToNodeEncryption,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS3',
            info: 'The OpenSearch Service domain does not only grant access via allowlisted IP addresses.',
            explanation: 'Using allowlisted IP addresses helps protect the domain against unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchAllowlistedIPs,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS4',
            info: 'The OpenSearch Service domain does not use dedicated master nodes.',
            explanation: 'Using dedicated master nodes helps improve environmental stability by offloading all the management tasks from the data nodes.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchDedicatedMasterNode,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS5',
            info: 'The OpenSearch Service domain does not allow for unsigned requests or anonymous access.',
            explanation: 'Restricting public access helps prevent unauthorized access and prevents any unsigned requests to be made to the resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchNoUnsignedOrAnonymousAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS7',
            info: 'The OpenSearch Service domain does not have Zone Awareness enabled.',
            explanation: 'Enabling cross-zone replication (Zone Awareness) increases the availability of the OpenSearch Service domain by allocating the nodes and replicate the data across two AZs in the same region in order to prevent data loss and minimize downtime in the event of node or AZ failure.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchZoneAwareness,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS8',
            info: 'The OpenSearch Service domain does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchEncryptedAtRest,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS9',
            info: 'The OpenSearch Service domain does not minimally publish SEARCH_SLOW_LOGS and INDEX_SLOW_LOGS to CloudWatch Logs.',
            explanation: 'These logs enable operators to gain full insight into the performance of these operations.' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'LogExport::<log>' for exported logs. Example: appliesTo: ['LogExport::SEARCH_SLOW_LOGS'].",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchSlowLogsToCloudWatch,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'QS1',
            info: 'The Quicksight data sources connection is not configured to use SSL.',
            explanation: 'SSL secures communications to data sources, especially when using public networks. Using SSL with QuickSight requires the use of certificates signed by a publicly-recognized certificate authority.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: quicksight_1.QuicksightSSLConnections,
            node: node,
        });
    }
    /**
     * Check Security and Compliance Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSecurityCompliance(node) {
        this.applyRule({
            ruleSuffixOverride: 'IAM4',
            info: 'The IAM user, role, or group uses AWS managed policies.',
            explanation: 'An AWS managed policy is a standalone policy that is created and administered by AWS. Currently, many AWS managed policies do not restrict resource scope. Replace AWS managed policies with system specific (customer) managed policies.' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'Policy::<policy>' for AWS managed policies. Example: appliesTo: ['Policy::arn:<AWS::Partition>:iam::aws:policy/foo'].",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMNoManagedPolicies,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'IAM5',
            info: 'The IAM entity contains wildcard permissions and does not have a cdk_nag rule suppression with evidence for those permission.',
            explanation: 'Metadata explaining the evidence (e.g. via supporting links) for wildcard permissions allows for transparency to operators. ' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'Action::<action>' for policy actions and 'Resource::<resource>' for resources. Example: appliesTo: ['Action::s3:*'].",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMNoWildcardPermissions,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG1',
            info: 'The Cognito user pool does not have a password policy that minimally specify a password length of at least 8 characters, as well as requiring uppercase, numeric, and special characters.',
            explanation: 'Strong password policies increase system security by encouraging users to create reliable and secure passwords.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cognito_1.CognitoUserPoolStrongPasswordPolicy,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG2',
            info: 'The Cognito user pool does not require MFA.',
            explanation: 'Multi-factor authentication (MFA) increases security for the application by adding another authentication method, and not relying solely on user name and password.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: cognito_1.CognitoUserPoolMFA,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG3',
            info: 'The Cognito user pool does not have AdvancedSecurityMode set to ENFORCED.',
            explanation: 'Advanced security features enable the system to detect and act upon malicious sign-in attempts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cognito_1.CognitoUserPoolAdvancedSecurityModeEnforced,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG4',
            info: 'The API GW method does not use a Cognito user pool authorizer.',
            explanation: 'API Gateway validates the tokens from a successful user pool authentication, and uses them to grant your users access to resources including Lambda functions, or your own API.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cognito_1.CognitoUserPoolAPIGWAuthorizer,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG7',
            info: 'The Cognito identity pool allows for unauthenticated logins and does not have a cdk_nag rule suppression with a reason.',
            explanation: 'In many cases applications do not warrant unauthenticated guest access applications. Metadata explaining the use case allows for transparency to operators.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cognito_1.CognitoUserPoolNoUnauthenticatedLogins,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KMS5',
            info: 'The KMS Symmetric key does not have automatic key rotation enabled.',
            explanation: 'KMS key rotation allow a system to set an yearly rotation schedule for a KMS key so when a AWS KMS key is required to encrypt new data, the KMS service can automatically use the latest version of the HSA backing key to perform the encryption.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: kms_1.KMSBackingKeyRotationEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SMG4',
            info: 'The secret does not have automatic rotation scheduled.',
            explanation: 'AWS Secrets Manager can be configured to automatically rotate the secret for a secured service or database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: secretsmanager_1.SecretsManagerRotationEnabled,
            node: node,
        });
    }
    /**
     * Check Serverless Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkServerless(node) {
        this.applyRule({
            ruleSuffixOverride: 'ASC3',
            info: 'The GraphQL API does not have request leveling logging enabled.',
            explanation: 'It is important to use CloudWatch Logs to log metrics such as who has accessed the GraphQL API, how the caller accessed the API, and invalid requests.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: appsync_1.AppSyncGraphQLRequestLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SF1',
            info: 'The Step Function does not log "ALL" events to CloudWatch Logs.',
            explanation: 'Logging "ALL" events to CloudWatch logs help operators troubleshoot and audit systems.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: stepfunctions_1.StepFunctionStateMachineAllLogsToCloudWatch,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SF2',
            info: 'The Step Function does not have X-Ray tracing enabled.',
            explanation: 'X-ray provides an end-to-end view of how an application is performing. This helps operators to discover performance issues, detect permission problems, and track requests made to and from other AWS services.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: stepfunctions_1.StepFunctionStateMachineXray,
            node: node,
        });
    }
    /**
     * Check Application Integration Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkApplicationIntegration(node) {
        this.applyRule({
            ruleSuffixOverride: 'EVB1',
            info: 'The event bus policy allows for open access.',
            explanation: 'An open policy ("*" principal without a condition) grants anonymous access to an event bus. Use a condition to limit the permission to accounts that fulfill a certain requirement, such as being a member of a certain AWS organization.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: eventbridge_1.EventBusOpenAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SNS2',
            info: 'The SNS Topic does not have server-side encryption enabled.',
            explanation: 'Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sns_1.SNSEncryptedKMS,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SNS3',
            info: 'The SNS Topic does not require publishers to use SSL.',
            explanation: "Without HTTPS (TLS), a network-based attacker can eavesdrop on network traffic or manipulate it, using an attack such as man-in-the-middle. Allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition and the 'sns: Publish' action in the topic policy to force publishers to use SSL. If SSE is already enabled then this control is auto enforced.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sns_1.SNSTopicSSLPublishOnly,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SQS2',
            info: 'The SQS Queue does not have server-side encryption enabled.',
            explanation: 'Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sqs_1.SQSQueueSSE,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SQS3',
            info: 'The SQS queue does not have a dead-letter queue (DLQ) enabled or have a cdk_nag rule suppression indicating it is a DLQ.',
            explanation: 'Using a DLQ helps maintain the queue flow and avoid losing data by detecting and mitigating failures and service disruptions on time.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sqs_1.SQSQueueDLQ,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SQS4',
            info: 'The SQS queue does not require requests to use SSL.',
            explanation: 'Without HTTPS (TLS), a network-based attacker can eavesdrop on network traffic or manipulate it, using an attack such as man-in-the-middle. Allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition in the queue policy to force requests to use SSL.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sqs_1.SQSQueueSSLRequestsOnly,
            node: node,
        });
    }
    /**
     * Check Media Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkMediaServices(node) {
        this.applyRule({
            ruleSuffixOverride: 'MS1',
            info: 'The MediaStore container does not have container access logging enabled.',
            explanation: 'The container should have access logging enabled to provide detailed records for the requests that are made to the container.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: mediastore_1.MediaStoreContainerAccessLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS3',
            info: 'The MediaStore container does not require requests to use SSL.',
            explanation: 'You can use HTTPS (TLS) to help prevent potential attackers from eavesdropping on or manipulating network traffic using person-in-the-middle or similar attacks. You should allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition on MediaStore container policies.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: mediastore_1.MediaStoreContainerSSLRequestsOnly,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS4',
            info: 'The MediaStore container does not define a metric policy to send metrics to CloudWatch.',
            explanation: 'Using a combination of MediaStore metrics and CloudWatch alarms helps operators gain better insights into container operations.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: mediastore_1.MediaStoreCloudWatchMetricPolicy,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS7',
            info: 'The MediaStore container does not define a container policy.',
            explanation: 'Using a container policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: mediastore_1.MediaStoreContainerHasContainerPolicy,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS8',
            info: 'The MediaStore container does not define a CORS policy.',
            explanation: 'Using a CORS policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: mediastore_1.MediaStoreContainerCORSPolicy,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS10',
            info: 'The MediaStore container does not define a lifecycle policy.',
            explanation: 'Many use cases warrant the usage of lifecycle configurations to manage container objects during their lifetime.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: mediastore_1.MediaStoreContainerLifecyclePolicy,
            node: node,
        });
    }
    /**
     * Check Developer Tools Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDeveloperTools(node) {
        this.applyRule({
            ruleSuffixOverride: 'CB3',
            info: 'The CodeBuild project has privileged mode enabled.',
            explanation: 'Privileged grants elevated rights to the system, which introduces additional risk. Privileged mode should only be set to true only if the build project is used to build Docker images. Otherwise, a build that attempts to interact with the Docker daemon fails.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: codebuild_1.CodeBuildProjectPrivilegedModeDisabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CB4',
            info: 'The CodeBuild project does not use an AWS KMS key for encryption.',
            explanation: 'Using an AWS KMS key helps follow the standard security advice of granting least privilege to objects generated by the project.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: codebuild_1.CodeBuildProjectKMSEncryptedArtifacts,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CB5',
            info: 'The Codebuild project does not use images provided by the CodeBuild service or have a cdk_nag suppression rule explaining the need for a custom image.',
            explanation: 'Explaining differences/edits to Docker images helps operators better understand system dependencies.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: codebuild_1.CodeBuildProjectManagedImages,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'C91',
            info: 'The Cloud9 instance does not use a no-ingress EC2 instance with AWS Systems Manager.',
            explanation: 'SSM adds an additional layer of protection as it allows operators to control access through IAM permissions and does not require opening inbound ports.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloud9_1.Cloud9InstanceNoIngressSystemsManager,
            node: node,
        });
    }
}
exports.AwsSolutionsChecks = AwsSolutionsChecks;
_a = JSII_RTTI_SYMBOL_1;
AwsSolutionsChecks[_a] = { fqn: "monocdk-nag.AwsSolutionsChecks", version: "1.10.0" };
//# sourceMappingURL=data:application/json;base64,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