"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_codebuild_1 = require("monocdk/aws-codebuild");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * CodeBuild projects do not store AWS credentials as plaintext environment variables
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_codebuild_1.CfnProject) {
        //Check for the presence of OAUTH
        const environment = monocdk_1.Stack.of(node).resolve(node.environment);
        const environmentVars = monocdk_1.Stack.of(node).resolve(environment.environmentVariables);
        if (environmentVars != undefined) {
            //For each envvar, check if its a sensitive credential being stored
            for (const envVar of environmentVars) {
                const resolvedEnvVar = monocdk_1.Stack.of(node).resolve(envVar);
                const name = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedEnvVar.name);
                const type = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedEnvVar.type);
                if (name == 'AWS_ACCESS_KEY_ID' || name == 'AWS_SECRET_ACCESS_KEY') {
                    //is this credential being stored as plaintext?
                    if (type == undefined || type == 'PLAINTEXT') {
                        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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