"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_ec2_1 = require("monocdk/aws-ec2");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
const EBS_OPTIMIZED_SUPPORTED = [
    'c1.xlarge',
    'c3.xlarge',
    'c3.2xlarge',
    'c3.4xlarge',
    'g2.2xlarge',
    'i2.xlarge',
    'i2.2xlarge',
    'i2.4xlarge',
    'm1.large',
    'm1.xlarge',
    'm2.2xlarge',
    'm2.4xlarge',
    'm3.xlarge',
    'm3.2xlarge',
    'r3.xlarge',
    'r3.2xlarge',
    'r3.4xlarge',
];
const DEFAULT_TYPE = 'm1.small';
/**
 * EC2 instance types that support EBS optimization and are not EBS optimized by default have EBS optimization enabled
 * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-optimized.html#previous
 *  @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_ec2_1.CfnInstance) {
        const instanceType = node.instanceType
            ? nag_rules_1.NagRules.resolveIfPrimitive(node, node.instanceType)
            : DEFAULT_TYPE;
        const ebsOptimized = monocdk_1.Stack.of(node).resolve(node.ebsOptimized);
        if (EBS_OPTIMIZED_SUPPORTED.includes(instanceType) &&
            ebsOptimized !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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