"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_glue_1 = require("monocdk/aws-glue");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * Glue crawlers and jobs have CloudWatch Log encryption enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_glue_1.CfnCrawler || node instanceof aws_glue_1.CfnJob) {
        let securityConfigurationId;
        if (node instanceof aws_glue_1.CfnCrawler) {
            securityConfigurationId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.crawlerSecurityConfiguration);
        }
        else {
            securityConfigurationId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.securityConfiguration);
        }
        if (securityConfigurationId == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_glue_1.CfnSecurityConfiguration) {
                if (isMatchingSecurityConfig(child, securityConfigurationId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether the referenced Security Configuration encrypts CloudWatch Logs
 * @param node the CfnSecurityConfiguration to check
 * @param securityConfigurationId the Cfn Logical ID of the security configuration
 * returns whether the CfnSecurityConfiguration encrypts CloudWatch Logs
 */
function isMatchingSecurityConfig(node, securityConfigurationId) {
    const resolvedSecurityConfigurationLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
    if (resolvedSecurityConfigurationLogicalId === securityConfigurationId ||
        node.name === securityConfigurationId) {
        const encryptionConfiguration = monocdk_1.Stack.of(node).resolve(node.encryptionConfiguration);
        const cloudWatchEncryption = encryptionConfiguration.cloudWatchEncryption;
        if (cloudWatchEncryption !== undefined) {
            const cloudWatchEncryptionMode = nag_rules_1.NagRules.resolveIfPrimitive(node, cloudWatchEncryption.cloudWatchEncryptionMode);
            if (cloudWatchEncryptionMode === 'SSE-KMS') {
                return true;
            }
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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