"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_iam_1 = require("monocdk/aws-iam");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
const flatten_cfn_reference_1 = require("../../utils/flatten-cfn-reference");
/**
 * IAM users, roles, and groups do not use AWS managed policies
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_iam_1.CfnGroup ||
        node instanceof aws_iam_1.CfnUser ||
        node instanceof aws_iam_1.CfnRole) {
        const managedPolicyArns = (monocdk_1.Stack.of(node).resolve(node.managedPolicyArns));
        const findings = new Set();
        if (managedPolicyArns !== undefined) {
            managedPolicyArns
                .map((policy) => flatten_cfn_reference_1.flattenCfnReference(monocdk_1.Stack.of(node).resolve(policy)))
                .filter((policy) => policy.includes(':iam::aws:'))
                .forEach((policy) => findings.add(`Policy::${policy}`));
        }
        return findings.size ? [...findings] : nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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