"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_iam_1 = require("monocdk/aws-iam");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
const flatten_cfn_reference_1 = require("../../utils/flatten-cfn-reference");
/**
 * IAM entities with wildcard permissions have a cdk_nag rule suppression with evidence for those permission
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_iam_1.CfnGroup ||
        node instanceof aws_iam_1.CfnUser ||
        node instanceof aws_iam_1.CfnRole) {
        const inlinePolicies = monocdk_1.Stack.of(node).resolve(node.policies);
        const findings = new Set();
        if (inlinePolicies != undefined) {
            for (const policy of inlinePolicies) {
                const resolvedPolicy = monocdk_1.Stack.of(node).resolve(policy);
                const resolvedPolicyDocument = monocdk_1.Stack.of(node).resolve(resolvedPolicy.policyDocument);
                analyzePolicy(resolvedPolicyDocument).forEach((finding) => findings.add(finding));
            }
        }
        return findings.size ? [...findings] : nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node instanceof aws_iam_1.CfnPolicy || node instanceof aws_iam_1.CfnManagedPolicy) {
        const policyDocument = monocdk_1.Stack.of(node).resolve(node.policyDocument);
        const findings = analyzePolicy(policyDocument);
        return findings.length ? findings : nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
const analyzePolicy = (policy) => {
    var _a;
    const result = new Set();
    for (const statement of (_a = policy.Statement) !== null && _a !== void 0 ? _a : []) {
        // we don't report on denys
        if (statement.Effect === 'Allow') {
            const actions = normalizeArray(statement.Action);
            actions
                .filter((action) => action.includes('*'))
                .map((action) => `Action::${action}`)
                .forEach((action) => result.add(action));
            const resources = normalizeArray(statement.Resource);
            resources
                .map(flatten_cfn_reference_1.flattenCfnReference)
                .filter((resource) => resource.includes('*'))
                .map((resource) => `Resource::${resource}`)
                .forEach((resource) => result.add(resource));
        }
    }
    return [...result];
};
const normalizeArray = (arrOrStr) => Array.isArray(arrOrStr) ? arrOrStr : [arrOrStr];
//# sourceMappingURL=data:application/json;base64,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