"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_msk_1 = require("monocdk/aws-msk");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * MSK clusters send broker logs to a supported destination
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_msk_1.CfnCluster) {
        const loggingInfo = monocdk_1.Stack.of(node).resolve(node.loggingInfo);
        if (loggingInfo == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const resolvedBrokerLogs = monocdk_1.Stack.of(node).resolve(loggingInfo.brokerLogs);
        let enabled = false;
        const s3 = monocdk_1.Stack.of(node).resolve(resolvedBrokerLogs.s3);
        if (s3 != undefined) {
            const s3Enabled = nag_rules_1.NagRules.resolveIfPrimitive(node, s3.enabled);
            if (s3Enabled) {
                enabled = true;
            }
        }
        const cloudWatchLogs = monocdk_1.Stack.of(node).resolve(resolvedBrokerLogs.cloudWatchLogs);
        if (cloudWatchLogs != undefined) {
            const cloudWatchLogsEnabled = nag_rules_1.NagRules.resolveIfPrimitive(node, cloudWatchLogs.enabled);
            if (cloudWatchLogsEnabled) {
                enabled = true;
            }
        }
        const firehose = monocdk_1.Stack.of(node).resolve(resolvedBrokerLogs.firehose);
        if (firehose != undefined) {
            const firehoseEnabled = nag_rules_1.NagRules.resolveIfPrimitive(node, firehose.enabled);
            if (firehoseEnabled) {
                enabled = true;
            }
        }
        if (!enabled) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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